/*
 * ADS7846 platform initialization for the Nokia 770
 *
 * Copyright (c) 2006 Nokia Corporation
 *
 * Filtering for the LS041Y3 LCD by
 *   Jarkko Oikarinen <jarkko.oikarinen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/device.h>
#include <linux/input.h>
#include <linux/spi/spi.h>
#include <linux/spi/ads7846.h>

#include <asm/arch/board.h>
#include <asm/arch/gpio.h>

#define ADS7846_PENDOWN_GPIO	15

/* This filter plugin is created to work around bug in LS041Y3 display.
 * The display interferes with the Y coordinate measurement of the
 * touchscreen, and additional filtering has to be done to the Y
 * coordinate.
 */

/* Number of samples used in median filter */
#define LS041Y3_MEDIAN_COUNT	16

struct ads7846_ls041y3 {
	int			read_cnt;
	int			read_rep;
	int			last_read;

	u16			debounce_max;
	u16			debounce_tol;
	u16			debounce_rep;

	u16			median_buf[LS041Y3_MEDIAN_COUNT];
};

/*
 * utility function to find the average from a buffer ignoring the 2/3
 * furthest from the median
 */
static inline u16 ads7846_ls041y3_avg(u16 *buf, int len)
{
	int i, j;
	unsigned sum;

	for (i = 0; i < len * 2 / 3; i++) {
		for (j = i + 1; j < len; j++) {
			if (buf[i] > buf[j]) {
				u16 swap = buf[i];
				buf[i] = buf[j];
				buf[j] = swap;
			}
		}
	}
	sum = 0;
	for (i = 0; i < len / 3; i++)
		sum += buf[len / 3 + i];
	return sum / i;
}

static int ads7846_ls041y3_filter(void *data, int data_idx, int *val)
{
	struct ads7846_ls041y3 *fdata = data;

	/*
	 * Special handling for coordinates due to noise created in the Y
	 * touchpanel drive.
	 */
	if (fdata->read_cnt < LS041Y3_MEDIAN_COUNT) {
		fdata->median_buf[fdata->read_cnt] = *val;
		fdata->read_cnt++;
	} else {
		*val = ads7846_ls041y3_avg(fdata->median_buf,
						      LS041Y3_MEDIAN_COUNT);
		fdata->read_cnt = 0;
		fdata->read_rep = 0;
		return ADS7846_FILTER_OK;
	}
	return ADS7846_FILTER_REPEAT;
}

static int ads7846_ls041y3_filter_init(struct ads7846_platform_data *pdata,
				       void **filter_data)
{

	struct ads7846_ls041y3 *fdata;

	fdata = kzalloc(sizeof(*filter_data), SLAB_KERNEL);
	if (filter_data == NULL)
		return -ENOMEM;
	*filter_data = fdata;
	fdata->debounce_max = pdata->debounce_max;
	fdata->debounce_tol = pdata->debounce_tol;
	fdata->debounce_rep = pdata->debounce_rep;

	return 0;
}

static void ads7846_ls041y3_filter_cleanup(void *filter_data)
{
	kfree(filter_data);
}

static int ads7846_get_pendown_state(void)
{
	return !omap_get_gpio_datain(ADS7846_PENDOWN_GPIO);
}

static struct ads7846_platform_data ads7846_platform_data __initdata = {
	.x_max				= 0x0fff,
	.y_max				= 0x0fff,
	.x_plate_ohms			= 180,
	.pressure_max			= 255,
	.debounce_max			= 10,
	.debounce_tol			= 3,
	.debounce_rep			= 1,
	.keep_vref_on			= 1,
	.get_pendown_state		= ads7846_get_pendown_state,
};

static struct ads7846_platform_data ads7846_ls041y3_platform_data __initdata = {
	.x_max				= 0x0fff,
	.y_max				= 0x0fff,
	.x_plate_ohms			= 250,
	.pressure_max			= 512,
	.debounce_max			= 12,
	.debounce_tol			= 4,
	.debounce_rep			= 1,
	.keep_vref_on			= 1,
	.filter_init			= ads7846_ls041y3_filter_init,
	.filter_cleanup			= ads7846_ls041y3_filter_cleanup,
	.filter				= ads7846_ls041y3_filter,
	.get_pendown_state		= ads7846_get_pendown_state,
};

static struct spi_board_info ads7846_spi_board_info[] __initdata = {
	[0] = {
		.modalias       = "ads7846",
		.bus_num        = 2,
		.chip_select    = 0,
		.max_speed_hz   = 2500000,
		.irq		= OMAP_GPIO_IRQ(ADS7846_PENDOWN_GPIO),
		/* platform data set in the init function */
	},
};

void __init nokia770_ts_init(void)
{
	const struct omap_lcd_config *conf;

	/* get the panel model for selecting the proper tsc parameters
	 * and filtering logic */
	conf = omap_get_config(OMAP_TAG_LCD, struct omap_lcd_config);
	if (conf == NULL) {
		printk(KERN_ERR "LCD configuration missing, "
				"using default touchscreen configuration\n");
	}

	if (conf != NULL && strncmp(conf->panel_name, "ls041y3", 8) == 0)
		ads7846_spi_board_info[0].platform_data	=
			&ads7846_ls041y3_platform_data;
	else
		ads7846_spi_board_info[0].platform_data =
			&ads7846_platform_data;

	omap_request_gpio(ADS7846_PENDOWN_GPIO);

	spi_register_board_info(ads7846_spi_board_info, 1);
}

