/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>
#include <gst/gst.h>

#include <dsp/interface_common.h>
#include "dspvideo.h"
#include "omapdsp.h"

static gchar *avsyncname = "/dev/dsptask/avsync";

/**
 *
 *
 */

static inline void
gst_dspvideo_init (GstDSPVideo * dsp)
{
  dsp->device = NULL;
  dsp->fd = -1;
  dsp->syncfd = -1;
  dsp->xpos = DSP_XPOS_DEFAULT;
  dsp->ypos = DSP_YPOS_DEFAULT;
  dsp->scrn_width = DSP_SCRNWIDTH_DEFAULT;
  dsp->scrn_height = DSP_SCRNHEIGHT_DEFAULT;
  dsp->rotation = 0;
  dsp->flipping = 0;
  dsp->audio_stream_id = 0;
  dsp->color_format = 0;      // FIXME
  dsp->aspect_ratio = 0;
  dsp->dsp_mutex = g_mutex_new();
  dsp->sync_mutex = g_mutex_new();
  dsp->fullscreen = FALSE;
  dsp->videoonly = FALSE;
  dsp->waiting_for_eos = FALSE;
  dsp->stream_id_set = FALSE;
  dsp->pixel_aspect_ratio = 1.0;
}

/**
 *
 *
 */

static inline void
gst_dspvideo_exit (GstDSPVideo * dsp)
{
  if(dsp->device) {
    g_free (dsp->device);
  }
  g_mutex_free(dsp->dsp_mutex);
  g_mutex_free(dsp->sync_mutex);
}


/**
 *
 *
 */

GstDSPVideo *
gst_dspvideo_new ()
{
  GstDSPVideo *dsp = g_new (GstDSPVideo, 1);
  gst_dspvideo_init(dsp);
  return dsp;
}


/**
 *
 *
 */

void
gst_dspvideo_destroy (GstDSPVideo * dsp)
{
  gst_dspvideo_exit(dsp);
  g_free (dsp);
}




/**
 *
 *
 */

// FIXME: Duplicate code with audio

gboolean
gst_dspvideo_check_read(GstDSPVideo *dsp)
{
  struct pollfd pfd;

  pfd.fd = dsp->fd;
  pfd.events = POLLIN;

  if(poll(&pfd,1,0)>0) return TRUE;
  else return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_read_pending(GstDSPVideo *dsp,
                          gboolean need_write,
                          gboolean need_cmd,
                          gboolean block)
{
  unsigned short int data[3];
  unsigned short int param_data[100];
  unsigned short int status;
  gboolean cmd_received = FALSE;
  gint done;

  if(need_write && dsp->write_pending) {
    return TRUE;
  }

  while(TRUE) {

    // Does the caller want a blocking read?
    if(block == FALSE) {
      // If no data available, break out...
      if(gst_dspvideo_check_read(dsp) == FALSE) {
        return FALSE;
      }
    }

    done = read(dsp->fd, data, sizeof(short int));

    if(done == -1) {
      g_warning("dsp read returned -1");
      status = DSP_ERROR_GENERAL;
    }
    else {
//       DBG_PRINT("dspvideo: Got cmd %d\n", data[0]);
      switch(data[0]) {
        case DSP_CMD_DATA_WRITE:
          dsp->writereqs_got++;
          done = read(dsp->fd, &data[1], 2 * sizeof(short int));
          dsp->read_index = data[2] << 1;
          dsp->write_pending = TRUE;
          status = data[1];
          break;

        case DSP_CMD_PAUSE:
          DBG_PRINT("Got pause!\n");
        case DSP_CMD_PLAY:
        case DSP_CMD_STOP:
        case DSP_CMD_DISCONT:
        case DSP_CMD_SET_PARAMS:
        case DSP_CMD_VIDEO_SET_MIRROR:
        case DSP_CMD_VIDEO_RESET_MIRROR:
        case DSP_CMD_VIDEO_ROTATION_0:
        case DSP_CMD_VIDEO_ROTATION_90:
        case DSP_CMD_VIDEO_ROTATION_180:
        case DSP_CMD_VIDEO_ROTATION_270:
        case DSP_CMD_VIDEO_SET_LOCKEDRATIO:
        case DSP_CMD_VIDEO_RESET_LOCKEDRATIO:
          done = read(dsp->fd, &data[1], sizeof(short int));
          cmd_received = TRUE;
          status = data[1];
          break;
        case DSP_CMD_EOF:
          DBG_PRINT("dspvideo: Got EOF\n");
          dsp->mode = DSP_MODE_EOS;
          done = read(dsp->fd, &data[1], sizeof(short int));
          cmd_received = TRUE;
          status = data[1];
          break;
        case DSP_CMD_STATE:
          cmd_received = TRUE;
          read( dsp->fd,
               &param_data[1],
               sizeof(VIDEO_STATUS_DATA)-sizeof(short int));
/*          DBG_PRINT( "\nnode_state = %d",
                   ((VIDEO_STATUS_DATA *)param_data)->node_state);
          DBG_PRINT("\nframe_count = %li",
                  ((VIDEO_STATUS_DATA *)param_data)->frame_count);
          DBG_PRINT("\noutImagePos_x = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->outImagePos_x);
          DBG_PRINT("\noutImagePos_y = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->outImagePos_y);
          DBG_PRINT("\noutImageHeight = %d",
                   ((VIDEO_STATUS_DATA *)param_data)->outImageHeight);
          DBG_PRINT("\noutImageWidth = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->outImageWidth);
          DBG_PRINT("\ninImageHeight = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->inImageHeight);
          DBG_PRINT("\ninImageWidth = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->inImageWidth);
          DBG_PRINT("\nlocked_ratio = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->locked_ratio);
          DBG_PRINT("\nmirror = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->mirror);
          DBG_PRINT("\nrotation = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->rotation);
          DBG_PRINT("\nzoom_factor = %d",
                  ((VIDEO_STATUS_DATA *)param_data)->zoom_factor);
          DBG_PRINT("\nSkipped frames = %li",
                  ((VIDEO_STATUS_DATA *)param_data)->skipped_frame_count);          */
          status = DSP_OK;
          break;
        case DSP_CMD_NONE:
            status = DSP_OK;
          break;
        default:
          // g_warning("illegal cmd_status received : 0x%02X", data[0]);
          //status = DSP_ERROR_GENERAL;
          status = DSP_OK;
          break;
      }
    }

    if(done == -1) {
      g_warning("dsp read returned -1");
      status = DSP_ERROR_GENERAL;
    }

    if(status != DSP_OK) {
      g_warning("dsp returned error, status = %d", status);
      dsp->mode = DSP_MODE_ERROR;
      dsp->error_status = status;
      dsp->error_cmd = data[0];
      return FALSE;
    }

    if(gst_dspvideo_check_read(dsp)) {
      continue;
    }

    if(need_write && dsp->write_pending)
      break;

    if(need_cmd && cmd_received)
      break;
  }

  return TRUE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_open(GstDSPVideo *dsp, gchar *device)
{
  DBG_PRINT("OPENING DSP\n");

  if ( dsp->mode == DSP_MODE_ERROR ) {
    DBG_PRINT("mpeg4: Error occured. Not opening dev\n");
    return FALSE;
  }

  DBG_PRINT("    Opening %s\n", device);
  dsp->fd = open (device, O_RDWR);
  if (dsp->fd == -1) {
    DBG_PRINT("    Failed to open the device\n");
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  dsp->device = g_strdup(device);
  DBG_PRINT("    %s opened\n", dsp->device);

  DBG_PRINT("Opening %s ... ", avsyncname);
  dsp->syncfd = open(avsyncname, O_RDWR);
  if (dsp->syncfd == -1) {
    g_warning("Opening %s failed", avsyncname);
    close(dsp->fd);
    return FALSE;
  }
  DBG_PRINT("Opened avsync ok\n");

//  DBG_PRINT("Opening video log file for ts\n");
//  dsp->log = open("/tmp/video_time", O_RDWR | O_CREAT);

  return TRUE;
}


gboolean
gst_dspvideo_close_dsp(GstDSPVideo *dsp)
{

  if(dsp==NULL) return FALSE;
  if ( dsp->mode == DSP_MODE_UNINITIALIZED ) return FALSE;

  int len = 0;
  DSP_CMD_STATUS cmd;

  cmd.dsp_cmd = DSP_CMD_CLOSE;

  g_mutex_lock(dsp->dsp_mutex);
  DBG_PRINT("dspvideo: Sending close to DSP\n");
  len = write( dsp->fd, &cmd, sizeof(short int));

  DBG_PRINT("dspvideo: Reading response for CLOSE\n");

  if ( len != -1 ) {
    read( dsp->fd, &cmd, 2*sizeof(short int));
    DBG_PRINT("CMD: %d ||| Status %d\n", cmd.dsp_cmd, cmd.status);
    while ( cmd.dsp_cmd != DSP_CMD_CLOSE ) {
      if ( cmd.dsp_cmd == DSP_CMD_DATA_WRITE ) {
        read(dsp->fd, &cmd, sizeof(short int));
      }
      read( dsp->fd, &cmd, 2*sizeof(short int));
    }
  } else {
    g_mutex_unlock(dsp->dsp_mutex);
    return FALSE;
  }
  g_mutex_unlock(dsp->dsp_mutex);
  DBG_PRINT("dspvideo: DSP_CMD_CLOSE confirmed\n");
  return TRUE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_close(GstDSPVideo *dsp)
{

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;

  if ( close(dsp->fd) != 0 ) {
    DBG_PRINT("dspvideo: closing device failed!!!!\n");
    perror("Reason:\n");
  }
  dsp->fd = -1;

  if(dsp->syncfd != -1) {
    close(dsp->syncfd);
    dsp->syncfd = -1;
  }

  // Free the string allocated with g_strdup
  if(dsp->device != NULL) {
    g_free (dsp->device);
    dsp->device = NULL;
  }
  DBG_PRINT("dspvideo: devices close successfully\n");
//  close( dsp->log);
  return TRUE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_reg_stream(GstDSPVideo * dsp)
{
  DSP_CMD_STATUS cmd;
  STREAM_CMD_DATA data;

  data.dsp_cmd = DSP_CMD_REG_VDSTRM;
  data.stream_ID = dsp->audio_stream_id;

  if(write(dsp->syncfd, &data, sizeof(STREAM_CMD_DATA)) > 0) {
    if(read(dsp->syncfd, &cmd, sizeof(DSP_CMD_STATUS)) > 0) {
      if(cmd.status == DSP_OK) {
        DBG_PRINT("dspvideo: stream registered ok\n");
        return TRUE;
      }
    }
  }
  DBG_PRINT("dspvideo: ERROR:stream register. Status = %d\n", cmd.status);
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 * gst_dspvideo_get_time:
 * @dsp: DSP video object
 *
 * Retrieve the timestamp of the video
 *
 * Returns: GStreamer timestamp, or 0 if error
 */

GstClockTime
gst_dspvideo_get_time(GstDSPVideo *dsp)
{
  TIME_DATA timedata;

  timedata.dsp_cmd = DSP_CMD_GET_TIME;
  timedata.stream_ID = dsp->audio_stream_id;

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, &timedata, sizeof(TIME_DATA)) > 0) {
    if(read(dsp->syncfd, &timedata, sizeof(TIME_DATA)) > 0) {
      if(timedata.status == DSP_OK) {
        g_mutex_unlock(dsp->sync_mutex);
        return GST_MSECOND * timedata.time_ms;
      }
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  DBG_PRINT("dspvideo: error getting time from avsync\n");
  dsp->mode = DSP_MODE_ERROR;
  return 0;
}

/**
 * gst_dspvideo_avsync_initialize:
 * @dsp: DSP video object
 *
 * Initializes the A/V sync clock node with given video object.
 * After the avsync is initialized, it can be used to retrieve
 * timestamps with gst_dspvideo_get_time() method.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */

gboolean
gst_dspvideo_avsync_initialize(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS cmd;
  cmd.dsp_cmd = DSP_CMD_INIT;

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, &cmd, sizeof(short int)) > 0) {
    if(read(dsp->syncfd, &cmd, sizeof(DSP_CMD_STATUS)) > 0) {
      if(cmd.status == DSP_OK) {
        if(gst_dspvideo_reg_stream(dsp)) {
          DBG_PRINT("dspvideo: AVSync initialized!\n");
          g_mutex_unlock(dsp->sync_mutex);
          return TRUE;
        }
      }
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  DBG_PRINT("dspvideo: Cannot initialize avsync node!\n");
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 * gst_dspvideo_avsync_seek:
 * @dsp: DSP video object
 * @time: Time where to seek (GStreamer representation)
 *
 * Adjusts DSP's internal A/V sync clock time to value given
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */

gboolean
gst_dspvideo_avsync_seek(GstDSPVideo *dsp, gint64 time)
{
  TIME_DATA data;
  GstClockTime c;

  if(dsp==NULL) return FALSE;
  if(dsp->syncfd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  c = gst_dspvideo_get_time(dsp);
  time = time - c;

  data.dsp_cmd = DSP_CMD_SET_TIME;
  data.stream_ID = dsp->audio_stream_id;
  data.time_ms = (long) (time / GST_MSECOND);
  data.status = -1;

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, &data, sizeof(TIME_DATA)) > 0) {
    if(read(dsp->syncfd, &data, sizeof(TIME_DATA)) > 0) {
      if(data.status == DSP_OK) {
        g_mutex_unlock(dsp->sync_mutex);
        return TRUE;
      }
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  dsp->mode = DSP_MODE_ERROR;
  dsp->error_status = data.status;
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_endstream(GstDSPVideo * dsp)
{
  short int avbuf[2];
  DSP_CMD_STATUS cmd;

  DBG_PRINT("dspvideo: Sending endstream to avsync node\n");

  avbuf[0] = DSP_CMD_ENDSTREAM;
  avbuf[1] = dsp->audio_stream_id;

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, avbuf, 2*sizeof(short int)) > 0) {
    if(read(dsp->syncfd, &cmd, sizeof(DSP_CMD_STATUS)) > 0) {
      if(cmd.status == DSP_OK) {
        g_mutex_unlock(dsp->sync_mutex);
        DBG_PRINT("dspvideo: got response to endstream\n");
        return TRUE;
      }
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  DBG_PRINT("dspvideo: endstream error\n");
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_initialize(GstDSPVideo *dsp, void *data, guint dataLen)
{
  DSP_CMD_STATUS init_status;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;

  DBG_PRINT("INITIALIZE_DSPVIDEO CALLED\n");

  write(dsp->fd, data, dataLen);
  read(dsp->fd, &init_status, sizeof(DSP_CMD_STATUS));
  if(init_status.status != DSP_OK) {
    DBG_PRINT("    FAILED TO INITIALIZE DSP, code %d\n", init_status.status );
    return FALSE;
  }

  if(gst_dspvideo_avsync_initialize(dsp) == FALSE) {
    DBG_PRINT("AVSYNC INIT FAIL\n");
    return FALSE;
  }

  DBG_PRINT("    %s initialized\n", dsp->device);

  return TRUE;
}

/**
 *
 *
 */
/*
typedef struct VPOP_DATA {
  short int dsp_cmd;
  short int scrn_image_width;
  short int scrn_image_height;
  short int scrn_image_xpos;
  short int scrn_image_ypos;
  short int rotation;
  short int flipping;
} VPOP_DATA;
*/

/**
 *
 *
 */
/*
gboolean
gst_dspvideo_setup_vpop(GstDSPVideo *dsp)
{
//  VIDEO_INIT_STATUS init_status;
  VPOP_DATA vpop;
  unsigned short int init_status[3];

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;

  vpop.dsp_cmd = DSP_CMD_SET_POSTPROC;
  vpop.scrn_image_width = dsp->scrn_width;
  vpop.scrn_image_height = dsp->scrn_height;
  vpop.scrn_image_xpos = dsp->xpos;
  vpop.scrn_image_ypos = dsp->ypos;
  vpop.rotation = dsp->rotation;
  vpop.flipping = dsp->flipping;

  DBG_PRINT("INITIALIZE_VPOP CALLED\n");

  ioctl(dsp->fd, DSPCTL_WRITEREQ);
  write(dsp->fd, &vpop, sizeof(vpop));
  read(dsp->fd, init_status, 3*sizeof(unsigned short int));
  if(init_status[1] != DSP_OK) {
    DBG_PRINT("    FAILED TO INITIALIZE VPOP\n");
    return FALSE;
  }

  DBG_PRINT("    VPOP initialized\n");
  return TRUE;
}
*/

/**
 *
 * This method is used for calculating video output resolution when
 * Screen coordinates are given
 *
 *
 */

void
gst_dspvideo_aspect( GstDSPVideo * dsp, VIDEO_PARAMS_DATA * params_data )
{
/*
  gdouble ty = 0;
  ty = (gdouble) dsp->scrn_width / dsp->aspect_ratio;
  if ( ty < dsp->scrn_height ) {
    params_data->outImageWidth = dsp->scrn_width;
    params_data->outImageHeight = lround (dsp->scrn_width * dsp->aspect_ratio );
  } else {
    params_data->outImageHeight = dsp->scrn_height;
    params_data->outImageWidth = lround ( dsp->scrn_height * dsp->aspect_ratio );
  }
*/

  gdouble x_ratio = (gdouble) dsp->scrn_width /
    ( (gdouble) dsp->width * dsp->pixel_aspect_ratio);
  gdouble y_ratio = (gdouble) dsp->scrn_height / (gdouble) dsp->height;

  // scale by smaller ratio
  if ( x_ratio < y_ratio ) {
    params_data->outImageWidth = dsp->scrn_width;
    params_data->outImageHeight = lround (dsp->height * x_ratio );
  } else {
    params_data->outImageHeight = dsp->scrn_height;
    params_data->outImageWidth = lround ( ( dsp->width * dsp->pixel_aspect_ratio )
      * y_ratio );
  }

//  params_data.outImageHeight = dsp->height;
//  params_data.outImageWidth = dsp->width;

  // output_width must be multiple of 8
  // output_height multiple of 4

  params_data->outImageWidth &= ~7;
  params_data->outImageHeight &= ~3;

  params_data->outImagePos_x = dsp->xpos;
  params_data->outImagePos_y = dsp->ypos;

  params_data->outImagePos_x +=
    ( dsp->scrn_width - params_data->outImageWidth) / 2;
  params_data->outImagePos_y +=
    ( dsp->scrn_height - params_data->outImageHeight) / 2;

//   if ( params_data->outImageWidth & 0x0001 ) {
//     params_data->outImageWidth--;
//   }
//   if ( params_data->outImageHeight & 0x0001 ) {
//     params_data->outImageHeight--;
//   }

   DBG_PRINT("dspvideo: Aspect set x:%d, y:%d, w:%d, h:%d, sw:%d, sh:%d\n",
     params_data->outImagePos_x, params_data->outImagePos_y,
     dsp->width, dsp->height, dsp->scrn_width, dsp->scrn_height );
   DBG_PRINT("dspvideo: Scaled w:%d, h:%d\n", params_data->outImageWidth,
     params_data->outImageHeight );

}

/**
 *
 *
 */
gboolean
gst_dspvideo_setparams(GstDSPVideo *dsp)
{
  if ( dsp->mode == DSP_MODE_UNINITIALIZED ) {
    return FALSE;
  }
  if ( dsp->fd == -1 ) return FALSE;

  gboolean ret = TRUE;
  gint retval = 0;
  VIDEO_PARAMS_DATA params_data;

  DBG_PRINT("setparams called:\n");
  params_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  params_data.fbuf_address = 0x180000;
  params_data.fbuf_height = DSP_SCRNHEIGHT_MAX;
  params_data.fbuf_width = DSP_SCRNWIDTH_MAX;
  params_data.fbuf_bpp = 16;
  params_data.fbuf_orientation = 0;
  params_data.outImageHeight = dsp->height;
  params_data.outImageWidth = dsp->width;
  params_data.outImagePos_x = dsp->xpos;
  params_data.outImagePos_y = dsp->ypos;

  params_data.inImageWidth = dsp->width;
  params_data.inImageHeight = dsp->height;

  gst_dspvideo_aspect( dsp, &params_data );

//  if ( dsp->audio_stream_id != 0 ) {
    DBG_PRINT("dspvideo: video stream id set: %d\n", dsp->audio_stream_id );
    params_data.audio_strm_ID = dsp->audio_stream_id;
//  }

  params_data.framebyframe = dsp->frame_by_frame ? 1 : 0;
  params_data.synconarm = dsp->arm_timestamps ? 1 : 0;

//  DBG_PRINT("dspvideo: frame by frame mode is %d\n", params_data.framebyframe);
//  DBG_PRINT("dspvideo: synconarm is %d\n", params_data.synconarm);

  g_mutex_lock(dsp->dsp_mutex);
  retval = write(dsp->fd, &params_data, sizeof(params_data) );
  if (retval == -1) {
    dsp->mode = DSP_MODE_ERROR;
    g_mutex_unlock(dsp->dsp_mutex);
    return FALSE;
  }
  ret = gst_dspvideo_read_pending(dsp, FALSE, TRUE, TRUE);
  g_mutex_unlock(dsp->dsp_mutex);
  DBG_PRINT("setparams successful\n");
  return ret;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_play(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS cmd;
  int ret;

  if(dsp == NULL) { return FALSE; }
  if(dsp->fd==-1) { return FALSE; }
  if(dsp->mode==DSP_MODE_UNINITIALIZED) { return FALSE; }
  if(dsp->mode==DSP_MODE_PLAYING) return TRUE;

  DBG_PRINT("dspvideo: play called\n");

  cmd.dsp_cmd = DSP_CMD_PLAY;

  g_mutex_lock(dsp->dsp_mutex);
  ret = write(dsp->fd, &cmd, sizeof(short int));
  g_mutex_unlock(dsp->dsp_mutex);
  DBG_PRINT("dspvideo: play command sent\n");

//  char c[100];
//  gettimeofday( &(dsp->time), NULL );
//  sprintf(c,"play at :%li.%li\n", (dsp->time).tv_sec,(dsp->time).tv_usec);
//  write( dsp->log, c, strlen(c) );

//  if(ret == -1 || cmd.status == DSP_MODE_ERROR) {
  if (  ret == -1 ) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  // Set the AVSYNC node to play mode too
  if(!gst_dspvideo_play_sync(dsp, dsp->audio_stream_id)) {
    return FALSE;
  }
  DBG_PRINT("dspvideo: play successful\n");
  dsp->mode = DSP_MODE_PLAYING;
  return TRUE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_play_sync(GstDSPVideo * dsp, gint strm_id)
{
  DSP_CMD_STATUS cmd;
  unsigned short avbuf[2];

  avbuf[0] = DSP_CMD_PLAY;
  avbuf[1] = strm_id;

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, avbuf, 2*sizeof(short int)) > 0) {
    DBG_PRINT("gst_dspvideo_play_sync: PLAY written, reading reply\n");
    if(read(dsp->syncfd, &cmd, sizeof(DSP_CMD_STATUS)) > 0) {
      DBG_PRINT("gst_dspvideo_play_sync: got reply for PLAY\n");
      if(cmd.status == DSP_OK) {
        g_mutex_unlock(dsp->sync_mutex);
        DBG_PRINT("dspvideo: sync node play successful\n");
        return TRUE;
      }
      printf("AVSYNC ERROR 3, status = %d\n", cmd.status);
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_discont_sync(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS cmd;
  short int avbuf[2];

  avbuf[0] = DSP_CMD_STRM_RESET;
  avbuf[1] = dsp->audio_stream_id;

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, avbuf, 2*sizeof(short int)) > 0) {
    if(read(dsp->syncfd, &cmd, sizeof(DSP_CMD_STATUS)) > 0) {
      if(cmd.status == DSP_OK) {
        g_mutex_unlock(dsp->sync_mutex);
        return TRUE;
      }
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  DBG_PRINT("dspvideo: STRM_RESET error status %d\n", cmd.status);
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_discont(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS cmd;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;

  //DBG_PRINT("Discont called\n");

  // Sync node must be on play state when disconting video node
  gst_dspvideo_play_sync(dsp, dsp->audio_stream_id);

  DBG_PRINT("dspvideo: writing discont and strm_reset cmd\n");
  cmd.dsp_cmd = DSP_CMD_DISCONT;

  g_mutex_lock(dsp->dsp_mutex);
  if (write(dsp->fd, &cmd, sizeof(short int)) > 0) {
    if(gst_dspvideo_read_pending(dsp, FALSE, TRUE, TRUE)) {
      if(gst_dspvideo_discont_sync(dsp)) {
        g_mutex_unlock(dsp->dsp_mutex);
        DBG_PRINT("dspvideo: discont and strm_reset successful\n");
        return TRUE;
      }
    }
  }
  g_mutex_unlock(dsp->dsp_mutex);
  dsp->mode = DSP_MODE_ERROR;
  DBG_PRINT("dspvideo: discont unsuccesfull\n");
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_pause_sync(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS status;
  unsigned short avbuf[2];

  avbuf[0] = DSP_CMD_PAUSE;
  avbuf[1] = dsp->audio_stream_id;
  DBG_PRINT("gst_dspvideo_pause_sync: PAUSE written, reading reply\n");

  g_mutex_lock(dsp->sync_mutex);
  if(write(dsp->syncfd, avbuf, 2*sizeof(short int)) > 0) {
    if(read(dsp->syncfd, &status, sizeof(DSP_CMD_STATUS)) > 0) {
      if(status.status == DSP_OK) {
        g_mutex_unlock(dsp->sync_mutex);
        DBG_PRINT("gst_dspvideo_pause_sync: Reply OK\n");
        return TRUE;
      }
    }
  }
  g_mutex_unlock(dsp->sync_mutex);
  DBG_PRINT("gst_dspvideo_pause_sync: ERROR Reply: %d\n", status.status);
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_pause(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS status;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;
  if(dsp->mode==DSP_MODE_EOS) return FALSE;

  DBG_PRINT("dspvideo: pause called\n");

  if (dsp->mode != DSP_MODE_PLAYING) {
    DBG_PRINT("not in playing mode -> cannot pause");
    return FALSE;
  }

  status.dsp_cmd = DSP_CMD_PAUSE;

  g_mutex_lock(dsp->dsp_mutex);
  if(write(dsp->fd, &status, sizeof(short int)) > 0 ) {
    if(gst_dspvideo_read_pending(dsp, FALSE, TRUE, TRUE)) {
      if(gst_dspvideo_pause_sync(dsp)) {
        g_mutex_unlock(dsp->dsp_mutex);
        DBG_PRINT("dspvideo: pause successful\n");
        dsp->mode = DSP_MODE_PAUSED;
        return TRUE;
      }
    }
  }
  g_mutex_unlock(dsp->dsp_mutex);
  dsp->mode = DSP_MODE_ERROR;
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_stop(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS cmd;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  DBG_PRINT("stop called.\n");

  // We can stop only from PLAY / PAUSE mode
  if(dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_PAUSED) {

    // Sync node must be on play state when stopping video node
    gst_dspvideo_play_sync(dsp, dsp->audio_stream_id);

    cmd.dsp_cmd = DSP_CMD_STOP;

    g_mutex_lock(dsp->dsp_mutex);
    if(write(dsp->fd, &cmd, sizeof(short int)) > 0) {
      DBG_PRINT("dspvideo: Reading response to stop from DSP\n");
      if(gst_dspvideo_read_pending(dsp, FALSE, TRUE, TRUE)) {
        DBG_PRINT("dspvideo: Got response to stop from dsp\n");
        if(gst_dspvideo_endstream(dsp)) {
          g_mutex_unlock(dsp->dsp_mutex);
          DBG_PRINT("dspvideo: endstream successful\n");
          dsp->mode = DSP_MODE_STOPPED;
          return TRUE;
        }
      }
    }
    g_mutex_unlock(dsp->dsp_mutex);
    dsp->mode = DSP_MODE_ERROR;
  }
  else {
    DBG_PRINT("not in playing mode -> cannot stop");
  }
  return FALSE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_wait_eof( GstDSPVideo * dsp )
{

  DSP_CMD_STATUS status;
  short int tmp;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;

  g_mutex_lock( dsp->dsp_mutex );
  read(dsp->fd, &status, 2 * sizeof(short int) );
  if ( status.dsp_cmd == DSP_CMD_DATA_WRITE ) {
    read(dsp->fd, &tmp, sizeof(short int));
  }
  g_mutex_unlock( dsp->dsp_mutex );
  if ( status.dsp_cmd == DSP_CMD_EOF ) {
    DBG_PRINT("dspvideo: Got EOF.\n");
    return TRUE;
  }

  DBG_PRINT("dspvideo: No EOF but got %d instead\n", status.dsp_cmd );
  return FALSE;
}

/**
 *
 *
 */

gboolean
gst_dspvideo_set_mirror(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS status;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;

  DBG_PRINT("set mirror called, value = %d", dsp->flipping);

  status.dsp_cmd = dsp->flipping ? DSP_CMD_VIDEO_SET_MIRROR : DSP_CMD_VIDEO_RESET_MIRROR;

  g_mutex_lock(dsp->dsp_mutex);
  write(dsp->fd, &status, sizeof(short int));
  gst_dspvideo_read_pending(dsp, FALSE, TRUE, TRUE);
  g_mutex_unlock(dsp->dsp_mutex);

  DBG_PRINT("set_mirror successful");
  return TRUE;
}

/**
 *
 *
 */

gboolean
gst_dspvideo_set_rotation(GstDSPVideo *dsp)
{
  DSP_CMD_STATUS status;

  if(dsp==NULL) return FALSE;
  if(dsp->fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;

  DBG_PRINT("set rotation called, value = %d", dsp->rotation);

  switch(dsp->rotation) {
    case 90:
      status.dsp_cmd = DSP_CMD_VIDEO_ROTATION_90;
      break;
    case 180:
      status.dsp_cmd = DSP_CMD_VIDEO_ROTATION_180;
      break;
    case 270:
      status.dsp_cmd = DSP_CMD_VIDEO_ROTATION_270;
      break;
    default:
      status.dsp_cmd = DSP_CMD_VIDEO_ROTATION_0;
  }
  g_mutex_lock(dsp->dsp_mutex);
  write(dsp->fd, &status, sizeof(short int));
  gst_dspvideo_read_pending(dsp, FALSE, TRUE, TRUE);
  g_mutex_unlock(dsp->dsp_mutex);

  DBG_PRINT("set_rotation successful");
  return TRUE;
}


/**
 *
 *
 */

gboolean
gst_dspvideo_set_property (GstDSPVideo *dsp, guint prop_id, const GValue *value)
{
  gboolean ret = TRUE;
  gint tmp = 0;

  if(dsp==NULL) return FALSE;

  switch (prop_id) {
    case DSPVIDEO_PROP_XPOS:
      dsp->xpos = g_value_get_uint(value);
//      gst_dspvideo_setparams(dsp);
      break;
    case DSPVIDEO_PROP_YPOS:
      dsp->ypos = g_value_get_uint(value);
//      gst_dspvideo_setparams(dsp);
      break;
    case DSPVIDEO_PROP_SCRNWIDTH:
      dsp->scrn_width = g_value_get_uint(value);
//      gst_dspvideo_setparams(dsp);
      break;
    case DSPVIDEO_PROP_SCRNHEIGHT:
      dsp->scrn_height = g_value_get_uint(value);
      gst_dspvideo_setparams(dsp);
      break;
    case DSPVIDEO_PROP_ROTATION:
      dsp->rotation = g_value_get_uint(value);
      DBG_PRINT("rotation set\n");
      gst_dspvideo_set_rotation(dsp);
      break;
    case DSPVIDEO_PROP_FLIPPING:
      dsp->flipping = g_value_get_uint(value);
      gst_dspvideo_set_mirror(dsp);
      break;
    case DSPVIDEO_PROP_AUDIO_STREAMID:
      tmp = g_value_get_uint(value);
      GstClockTime curr_time;
      DBG_PRINT("dspvideo: Got streamid: %d\n", tmp);
      if (tmp == dsp->audio_stream_id) {
        DBG_PRINT("dspvideo: Given streamid already set.");
        break;
      }
      else if (tmp == 0) {
        curr_time = gst_dspvideo_get_time(dsp);
//        DBG_PRINT("dspvideo: streamid %d has pres.time %"GST_TIME_FORMAT".",
//                   dsp->audio_stream_id,GST_TIME_ARGS(curr_time));
        dsp->videoonly = TRUE;
        dsp->waiting_for_eos = TRUE;
        gst_dspvideo_endstream(dsp);
      }

      dsp->audio_stream_id = tmp;
      gst_dspvideo_setparams(dsp);

      if (dsp->audio_stream_id) {
        g_mutex_lock(dsp->sync_mutex);
        gst_dspvideo_reg_stream(dsp);
        g_mutex_unlock(dsp->sync_mutex);
      }
      else {
        gst_dspvideo_avsync_seek(dsp, curr_time);
        gst_dspvideo_play_sync(dsp, dsp->audio_stream_id);
//        curr_time = gst_dspvideo_get_time(dsp);
//        DBG_PRINT("dspvideo: streamid %d has pres.time %"GST_TIME_FORMAT".",
//                  dsp->audio_stream_id,GST_TIME_ARGS(curr_time));
      }

      dsp->stream_id_set = TRUE;

      break;
    case DSPVIDEO_PROP_FULLSCREEN:
      DBG_PRINT("dspvideo: FULLSCREEN parameter set...\n");
      if ( dsp->fullscreen != g_value_get_boolean(value) ) {
        dsp->fullscreen = g_value_get_boolean(value);
        if( dsp->fullscreen ) {
          DBG_PRINT("dspvideo: Going fullscreen\n");
          dsp->xpos = 0;
          dsp->ypos = 0;
          dsp->scrn_width = DSP_SCRNWIDTH_MAX;
          dsp->scrn_height = DSP_SCRNHEIGHT_MAX;
        } else {
          DBG_PRINT("dspvideo: Going normal screen\n");
          dsp->xpos = DSP_XPOS_DEFAULT;
          dsp->ypos = DSP_YPOS_DEFAULT;
          dsp->scrn_width = DSP_SCRNWIDTH_DEFAULT;
          dsp->scrn_height = DSP_SCRNHEIGHT_DEFAULT;
        }
        gst_dspvideo_setparams(dsp);
      }
      break;
    case DSPVIDEO_PROP_VIDEOONLY:
      DBG_PRINT("videoonly set!\n");
      dsp->videoonly = g_value_get_boolean(value);
      break;
    case DSPVIDEO_PROP_FRAME_BY_FRAME:
      dsp->frame_by_frame = g_value_get_boolean(value);
      DBG_PRINT("dspvideo: frame by frame mode set %s\n",
                dsp->frame_by_frame ? "TRUE" : "FALSE");
      break;
    case DSPVIDEO_PROP_ARM_TIMESTAMPS:
      dsp->arm_timestamps = g_value_get_boolean(value);
      DBG_PRINT("dspvideo: arm timestamps set %s\n",
      dsp->arm_timestamps ? "TRUE" : "FALSE");
      DBG_PRINT("dspvideo: enabling also frame by frame mode.\n");
      dsp->frame_by_frame = TRUE;
      break;
    default:
      ret = FALSE;
      break;
  }
  return ret;
}

/**
 *
 *
 */

gboolean
gst_dspvideo_get_property (GstDSPVideo *dsp, guint prop_id, GValue *value)
{
  gboolean ret = TRUE;

  if(dsp==NULL) return FALSE;

  switch (prop_id) {
    case DSPVIDEO_PROP_XPOS:
      g_value_set_uint (value, dsp->xpos);
      break;
    case DSPVIDEO_PROP_YPOS:
      g_value_set_uint (value, dsp->ypos);
      break;
    case DSPVIDEO_PROP_SCRNWIDTH:
      g_value_set_uint (value, dsp->scrn_width);
      break;
    case DSPVIDEO_PROP_SCRNHEIGHT:
      g_value_set_uint (value, dsp->scrn_height);
      break;
    case DSPVIDEO_PROP_ROTATION:
      g_value_set_uint (value, dsp->rotation);
      gst_dspvideo_set_rotation(dsp);
      break;
    case DSPVIDEO_PROP_FLIPPING:
      g_value_set_uint (value, dsp->flipping);
      break;
    case DSPVIDEO_PROP_AUDIO_STREAMID:
      g_value_set_uint (value, dsp->audio_stream_id);
      break;
    case DSPVIDEO_PROP_FULLSCREEN:
      g_value_set_boolean(value, dsp->fullscreen);
      break;
    case DSPVIDEO_PROP_VIDEOONLY:
      g_value_set_boolean(value, dsp->videoonly);
      break;
    case DSPVIDEO_PROP_FRAME_BY_FRAME:
      g_value_set_boolean(value, dsp->frame_by_frame);
      break;
    case DSPVIDEO_PROP_ARM_TIMESTAMPS:
      g_value_set_boolean(value, dsp->arm_timestamps);
      break;
    default:
      ret = FALSE;
      break;
  }
  return ret;
}


/**
 *
 *
 */

void
gst_dspvideo_install_properties (GObjectClass *klass)
{
  g_object_class_install_property (klass, DSPVIDEO_PROP_XPOS,
      g_param_spec_uint ("xpos", "X-position",
          "X coordinate of the left-top corner of the image", 0, DSP_XPOS_MAX,
          DSP_XPOS_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_YPOS,
      g_param_spec_uint ("ypos", "Y-position",
          "Y coordinate of the left-top corner of the image", 0, DSP_YPOS_MAX,
          DSP_YPOS_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_SCRNWIDTH,
      g_param_spec_uint ("width", "Screen width",
          "Width of the view area on screen", 0, DSP_SCRNWIDTH_MAX,
          DSP_SCRNWIDTH_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_SCRNHEIGHT,
      g_param_spec_uint ("height", "Screen height",
          "Height of the view area on screen", 0, DSP_SCRNHEIGHT_MAX,
          DSP_SCRNHEIGHT_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_ROTATION,
      g_param_spec_uint ("rotation", "Screen rotation",
          "Screen rotation in degrees (multiple of 90)", 0, DSP_ROTATION_MAX,
          DSP_ROTATION_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_FLIPPING,
      g_param_spec_uint ("flipping", "Screen flipping",
          "Vertical (1)/Horizontal (2)/Both (3) screen mirroring", 0, DSP_FLIPPING_MAX,
          DSP_FLIPPING_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_AUDIO_STREAMID,
      g_param_spec_uint ("audio_stream_id", "Stream id of video",
          "Stream identifier for video stream that is related to this video",
          0, G_MAXUSHORT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_FULLSCREEN,
      g_param_spec_boolean ("fullscreen", "FS status",
          "Fullscreen disable/enable",
          FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_VIDEOONLY,
      g_param_spec_boolean ("videoonly", "Video only flag",
          "If true, the stream is video without audio.",
          FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_FRAME_BY_FRAME,
      g_param_spec_boolean ("frame_by_frame", "Frame by frame mode",
          "If true, sends data to dsp one frame at a time.",
          TRUE, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPVIDEO_PROP_ARM_TIMESTAMPS,
      g_param_spec_boolean ("arm_timestamps", "Use timestamps from ARM for AV sync",
          "If true, timestamps from gst buffers are sent to DSP for AV sync.",
          TRUE, G_PARAM_READWRITE));

}

/**
 *
 *
 */

static gboolean
plugin_init (GstPlugin * plugin)
{
  return TRUE;
}

/**
 * gst_dspvideo_peer_query:
 * @pad: query this pad's peer
 * @query: query to perform
 *
 * Returns: TRUE, iff the query was successful.
 *
 */

gboolean
gst_dspvideo_peer_query (GstPad *pad, GstQuery *query)
{
  GstPad *peer = NULL;
  gboolean ret = FALSE;

  DBG_PRINT("dspvideo: peer query\n");
  peer = gst_pad_get_peer(pad);
  if (peer) {
    ret = gst_pad_query (peer, query);
    gst_object_unref (peer);
  }
  return ret;
}

/**
 *
 *
 */

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "gstdspvideo",
    "Utility plugin for all DSP video elements",
    plugin_init,
    VERSION,
    "LGPL",
    "dspvideo",
    "Nokia Corporation")
