/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

// System includes
#define _XOPEN_SOURCE 500

#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/select.h>

#define __USE_BSD 1
#include <sys/time.h>
#include <time.h>

// DSP specific
#include "omapdsp.h"
#include <dsp/interface_common.h>

// GStreamer includes
#include <gst/gst.h>
#include <gst/gstmarshal.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdspfb.h"

// Just to be sure
#ifndef MIN
#define MIN(a,b) ((a)<(b)) ? (a) : (b)
#endif

#define DSPFB_QUEUE_MAX_FRAMES 10

enum {
  FBSINK_QOS_NORMAL = 0,
  FBSINK_QOS_HURRY,
  FBSINK_QOS_SKIP
};


GstElementDetails gst_dspfb_details = GST_ELEMENT_DETAILS ("DSP FB Sink",
    "Sink",
    "Raw FB video sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

static gchar *device = "/dev/dsptask/videofb";

static GstStaticPadTemplate dspfb_factory =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-yuv, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "fps = (double) [ 0, 40 ], "
        "format = (fourcc) I420" )
    );



static GstVideoSinkClass *parent_class = NULL;

// Function prototypes
static void gst_dspfb_class_init (GstDSPFBSinkClass * klass);
static void gst_dspfb_base_init (gpointer g_class);
static void gst_dspfb_init (GstDSPFBSink * dspfb);

static void gst_dspfb_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_dspfb_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

//static const GstFormat *gst_dspfb_get_formats (GstElement * element);
static const GstQueryType * gst_dspfb_get_query_types (GstPad * pad);

static GstStateChangeReturn gst_dspfb_change_state (GstElement * element,
						    GstStateChange transition);
static GstStateChangeReturn gst_dspfb_initialize(GstDSPFBSink *dsp);
static gboolean gst_dspfb_query (GstElement * element, GstQuery * query);
static gboolean gst_dspfb_sink_query(GstPad *pad, GstQuery *query);
static gboolean gst_dspfb_event (GstElement * element, GstEvent * event);

// GstBaseSink callbacks
static GstCaps *gst_dspfb_get_caps (GstBaseSink *sink);
static gboolean gst_dspfb_set_caps (GstBaseSink * bsink, GstCaps * caps);
static GstFlowReturn gst_dspfb_render (GstBaseSink * bsink, GstBuffer * buf);
static gboolean gst_dspfb_event2 (GstBaseSink *sink, GstEvent * event);
// static GstFlowReturn gst_dspfb_preroll (GstBaseSink * bsink, GstBuffer * buf);

// Clocking stuff
//static GstClock *gst_dspfb_provide_clock (GstElement * elem);
//static GstClockTime gst_dspfb_get_time (GstClock * clock, GstDSPFBSink * src);


/**
 *
 *
 */

GType
gst_dspfbsink_get_type (void)
{
  static GType dspfb_type = 0;

  if (!dspfb_type) {
    static const GTypeInfo dspfb_info = {
      sizeof (GstDSPFBSinkClass),
      gst_dspfb_base_init,
      NULL,
      (GClassInitFunc) gst_dspfb_class_init,
      NULL,
      NULL,
      sizeof (GstDSPFBSink),
      0,
      (GInstanceInitFunc) gst_dspfb_init,
    };

    dspfb_type =
      g_type_register_static (GST_TYPE_VIDEO_SINK, "GstDSPFBSink", &dspfb_info,
			      0);

  }
  return dspfb_type;
}

/**
 *
 *
 */

static void
gst_dspfb_dispose (GObject * object)
{
  GstDSPFBSink *dsp = (GstDSPFBSink *) object;
  gst_dspvideo_destroy(dsp->video);
  DBG_PRINT("DSP-FB DISPOSE\n");

  while (!g_queue_is_empty(dsp->buf_queue)){
    gst_buffer_unref(g_queue_pop_head(dsp->buf_queue));
  }
  g_queue_free(dsp->buf_queue);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/**
 *
 *
 */

static void
gst_dspfb_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspfb_factory));
  gst_element_class_set_details (gstelement_class, &gst_dspfb_details);

}


/**
 *
 *
 */

static void
gst_dspfb_class_init (GstDSPFBSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbasesink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesink_class = (GstBaseSinkClass *) klass;

  parent_class = g_type_class_ref (GST_TYPE_VIDEO_SINK);


  gobject_class->set_property = gst_dspfb_set_property;
  gobject_class->get_property = gst_dspfb_get_property;

  // Use dspvideo utility to install default properties
  gst_dspvideo_install_properties(G_OBJECT_CLASS (klass));

  gstelement_class->change_state = gst_dspfb_change_state;
  gstelement_class->send_event = gst_dspfb_event;
  gstelement_class->query = gst_dspfb_query;
//   gstelement_class->provide_clock = gst_dspfb_provide_clock;

  gstbasesink_class->render = gst_dspfb_render;
  gstbasesink_class->get_caps = gst_dspfb_get_caps;
  gstbasesink_class->set_caps = gst_dspfb_set_caps;
  gstbasesink_class->event = gst_dspfb_event2;
//   gstbasesink_class->preroll = gst_dspfb_preroll;

  gobject_class->dispose = gst_dspfb_dispose;

}


/**
 *
 *
 */

static void
gst_dspfb_init (GstDSPFBSink * dspfb)
{
  dspfb->video = gst_dspvideo_new();
  dspfb->buf_queue = g_queue_new();
  dspfb->mode = DSP_MODE_UNINITIALIZED;

  dspfb->sinkpad = GST_BASE_SINK_PAD(dspfb);

  gst_pad_set_query_type_function (dspfb->sinkpad,
                                   gst_dspfb_get_query_types);

  gst_pad_set_query_function (dspfb->sinkpad,
                              gst_dspfb_sink_query);

  // FIXME: Do we need to delete this?
//  dspfb->clock = gst_audio_clock_new ("clock",
//                                      (GstAudioClockGetTimeFunc) gst_dspfb_get_time,
//                                       dspfb);
//  DBG_PRINT("FB CLOCK CREATED\n");

  // Discont timestamp handling
  dspfb->wait_pts = FALSE;
  dspfb->segment_start = 0;
  dspfb->diff_pts = 0;
  dspfb->dsp_read_count = 0;
  dspfb->prev_gst_pts = 0;

  dspfb->qos_state = FBSINK_QOS_NORMAL;

}

static GstCaps *
gst_dspfb_get_caps (GstBaseSink *sink) {
  return gst_caps_copy (
    gst_pad_get_pad_template_caps (GST_VIDEO_SINK_PAD(sink)));
}



/*
static const GstFormat *
gst_dspfb_get_formats (GstElement * element)
{
  static const GstFormat formats[] = {
    GST_FORMAT_TIME,
    0
  };

  return formats;
}
*/



/**
 *
 *
 */

static const GstQueryType *
    gst_dspfb_get_query_types (GstPad * pad)
{
  static const GstQueryType types[] = {
    GST_QUERY_DURATION,
    GST_QUERY_POSITION,
    0,
  };

  return types;
}


/**
 *
 *
 */

static gboolean gst_dspfb_do_query(GstDSPFBSink *dspfb,
                                   GstQuery *query,
                                   gboolean use_avsync)
{
  gboolean res = FALSE;
  gint64 avsync_time = 0;

  switch (GST_QUERY_TYPE (query)) {
    case GST_QUERY_DURATION:
      res = gst_pad_query_default (dspfb->sinkpad, query);
      break;
    case GST_QUERY_POSITION:
      if ((dspfb->video->audio_stream_id != 0 ||
           dspfb->video->videoonly) && dspfb->parsed)
      {
        if(use_avsync) {
          dspfb->avsync_pts = gst_dspvideo_get_time(dspfb->video);
        }
        avsync_time = dspfb->avsync_pts - dspfb->diff_pts;
        DBG_PRINT("fb: position = %lld\n", avsync_time);
        gst_query_set_position(query, GST_FORMAT_TIME, avsync_time);
        res = TRUE;
      }
      else {
        res = gst_pad_query_default (dspfb->sinkpad, query);
      }
      break;
    default:
      res = gst_pad_query_default (dspfb->sinkpad, query);
      break;
  }
  return res;
}


/**
 * Element query
 *
 */

static gboolean gst_dspfb_query (GstElement * element, GstQuery * query)
{
  DBG_PRINT("fb: element query\n");
  GstDSPFBSink *dspfb = GST_DSPFBSINK(element);
  return gst_dspfb_do_query(dspfb, query, FALSE);
}

/**
 * Sinkpad query
 *
 */

static gboolean
gst_dspfb_sink_query(GstPad * pad, GstQuery *query)
{
  DBG_PRINT("fb: sinkpad query\n");
  GstDSPFBSink *dspfb = GST_DSPFBSINK (GST_PAD_PARENT(pad));
  return gst_dspfb_do_query(dspfb, query, TRUE);
}


/**
 *
 *
 */

static GstPadLinkReturn
gst_dspfb_parse_caps (GstDSPFBSink * dsp, const GstCaps * caps)
{
  //gint bpp;
  GstStructure *structure;
  const gchar *mime;

  structure = gst_caps_get_structure (caps, 0);

  mime = gst_structure_get_name (structure);
  //DBG_PRINT("prev element is offering: %s\n", mime );

  if(gst_structure_has_field(structure,"width")) {
    gst_structure_get_int(structure, "width", &dsp->video->width);
  }
  else {
    GST_WARNING("fb: connect: no width, refused\n");
    return GST_PAD_LINK_REFUSED;
  }

  if(gst_structure_has_field(structure,"height")) {
    gst_structure_get_int(structure, "height", &dsp->video->height);
  }
  else {
    GST_WARNING("fb: connect: no height, refused\n");
    return GST_PAD_LINK_REFUSED;
  }

  if (dsp->video->height * dsp->video->width > 352 * 288) {
    GST_WARNING("Video is too large\n");
    GST_ELEMENT_ERROR (dsp, STREAM, DECODE, (NULL), (NULL));
    return GST_PAD_LINK_REFUSED;
  }

  if (dsp->video->height > 352) {
    GST_WARNING("Video height is too big\n");
    GST_ELEMENT_ERROR (dsp, STREAM, DECODE, (NULL), (NULL));
    return GST_PAD_LINK_REFUSED;
  }

  if (dsp->video->width > 352) {
    GST_WARNING("Video width is too big\n");
    GST_ELEMENT_ERROR (dsp, STREAM, DECODE, (NULL), (NULL));
    return GST_PAD_LINK_REFUSED;
  }


  if (dsp->video->width % 4 || dsp->video->height % 4) {
    GST_WARNING("Video width/height are not multiple by 4\n");
    GST_ELEMENT_ERROR (dsp, STREAM, DECODE, (NULL), (NULL));
    return GST_PAD_LINK_REFUSED;
  }

  if(gst_structure_has_field(structure,"framerate")) {
    gst_structure_get_double(structure, "framerate", &dsp->video->fps);
  }
  else {
    GST_WARNING("fb: connect: no framerate, refused\n");
    return GST_PAD_LINK_REFUSED;
  }

  // Aspect ratio, x/y
  if ( ( dsp->video->width != 0 ) &&
       ( dsp->video->height != 0 ) ) {
    dsp->video->aspect_ratio = (gdouble) dsp->video->width / dsp->video->height;
    GST_DEBUG("Aspect ratio: %lf \n", dsp->video->aspect_ratio );
  }

  DBG_PRINT("Offered w x h : %d x %d ok\n", dsp->video->width,
  dsp->video->height);
  gst_dspvideo_setparams(dsp->video);
  dsp->parsed = TRUE;

  return GST_PAD_LINK_OK;
}

/**
 *
 *
 */

// gboolean
// gst_dspfb_discont(GstDSPFBSink *dspfb)
// {
//   DSP_CMD_STATUS cmd;
//   int ret;
//
//   GstDSPVideo *dsp = dspfb->video;
//
//
//   if(dsp==NULL) return FALSE;
//   if(dsp->fd==-1) return FALSE;
//   if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;
//
//   //DBG_PRINT("Discont called\n");
//
//   // Sync node must be on play state when disconting video node
//   // gst_dspvideo_play_sync( dsp, dsp->audio_stream_id );
//
//   //g_mutex_lock(dsp->dsp_mutex);
//
//   DBG_PRINT("fb: writing strm_reset cmd\n");
//
//   short int avbuf[10];
//   avbuf[0] = DSP_CMD_STRM_RESET;
//   avbuf[1] = dsp->audio_stream_id;
//   ret = write(dsp->syncfd, avbuf, 2*sizeof(short int));
//   if(ret == -1) {
//     dsp->mode = DSP_MODE_ERROR;
//     //g_mutex_unlock(dsp->dsp_mutex);
//     return FALSE;
//   }
//
//   ret = read(dsp->syncfd, &cmd, sizeof(DSP_CMD_STATUS));
//   if(ret == -1 || cmd.status != DSP_OK) {
//     dsp->mode = DSP_MODE_ERROR;
//     DBG_PRINT("fb: error with STRM_RESET, returned error status %d\n",
//       cmd.status);
//     //g_mutex_unlock(dsp->dsp_mutex);
//     return FALSE;
//   }
//
//   //g_mutex_unlock(dsp->dsp_mutex);
//
//   DBG_PRINT("fb: strm_reset successful\n");
//   //gst_dspvideo_play_sync( dsp, 0 );
//   //dsp->mode = DSP_MODE_PAUSED;
//
//   return TRUE;
// }


/**
 * gst_dspfb_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, if the query was successful
 */

static gboolean
gst_dspfb_event (GstElement * element, GstEvent * event)
{
  GstEventType type;
  GstDSPFBSink *dsp = GST_DSPFBSINK (element);

  type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GST_DEBUG ("dsp: event %p %d", event, type);

  DBG_PRINT("fb: element event %d\n", type);

  switch (type) {
    case GST_EVENT_SEEK:
      if (!dsp->data_written) {
        DBG_PRINT("fb: delaying seeking\n");
        dsp->pending_seek = event;
        break;
      }
      if ( dsp->eos_mode || dsp->video->waiting_for_eos ||
           dsp->seeking)
      {
        DBG_PRINT("FB IN EOS -> NO SEEK\n");
        gst_event_unref(event);
        return FALSE;
      }

      DBG_PRINT("fb: SEEK EVENT\n");
      if ( gst_pad_send_event (GST_PAD_PEER (dsp->sinkpad), event) ) {
        DBG_PRINT("fb: seek event succesfully sent\n");
        dsp->seeking = TRUE;
        dsp->qos_ok = FALSE; // This will update the QoS info in decoder
      }
      else {
        DBG_PRINT("fb: SEEK FAILED\n");
      }
      break;

    default:
      DBG_PRINT("fb: event default1\n");
      gst_pad_event_default(dsp->sinkpad, event);
      break;
  }

  return TRUE;
}



/**
 * gst_dspfb_event2
 * @dspfb: framebuffer sink
 * @event: GStreamer event
 * @pad: Sink pad
 *
 * Handles the sink events
 */

static gboolean gst_dspfb_event2 (GstBaseSink *sink, GstEvent * event)
{
  DBG_PRINT("fb: event %d\n", GST_EVENT_TYPE(event));

  GstDSPFBSink *dspfb = GST_DSPFBSINK (sink);

  switch (GST_EVENT_TYPE (event)) {

    case GST_EVENT_FLUSH_START:
      DBG_PRINT("dspfb: Got FLUSH START event.\n");
      while( !g_queue_is_empty(dspfb->buf_queue) ) {
        gst_buffer_unref(g_queue_pop_head(dspfb->buf_queue));
      }
      break;
    case GST_EVENT_NEWSEGMENT:
    {
      gboolean update;
      gdouble rate;
      GstFormat format;
      gint64 start, stop, position;

      gst_event_parse_new_segment(event, &update, &rate, &format,
                                  &start, &stop, &position);

      DBG_PRINT("NEWSEGMENT start=%lld, stop=%lld, position=%lld\n",
                start, stop, position);

      if(dspfb->video->waiting_for_eos == FALSE) {
        if (dspfb->first_discont) {
          dspfb->first_discont = FALSE;
          DBG_PRINT("fb: skip first discont\n");
        }
        else {
          DBG_PRINT("fb: send discont\n");
          dspfb->wait_pts = TRUE;
        }
        dspfb->segment_start = start;
        DBG_PRINT("fb: set segment start as %"GST_TIME_FORMAT"\n",
                  GST_TIME_ARGS(start));
      }
      break;
    }
    case GST_EVENT_EOS:
      DBG_PRINT("fb: eos");
      dspfb->eos_mode = TRUE;
      GstBuffer *buf = NULL;
      while (!g_queue_is_empty(dspfb->buf_queue)){
        // Render the remaining frames in queue
        buf = g_queue_pop_head(dspfb->buf_queue);
        gst_dspfb_render(sink, buf);
        gst_buffer_unref(buf);
      }
      DBG_PRINT("fb: all frames rendered from queue\n");
      break;

    default:
      DBG_PRINT("fb: event default2\n");
      break;
  }

  DBG_PRINT("fb: event handling done\n");
  return TRUE;
}


/**
 *
 *
 */

// gboolean
// gst_dspfb_set_avsync_time(GstDSPVideo *dsp, guint64 cur_time, guint64 time)
// {
//   int len;
//   TIME_DATA data;
//
//   if(dsp==NULL) return FALSE;
//   if(dsp->syncfd==-1) return FALSE;
//   if(dsp->mode==DSP_MODE_ERROR) return FALSE;
//
//   //g_mutex_lock(dsp->dsp_mutex);
//   data.dsp_cmd = DSP_CMD_SET_TIME;
//   data.stream_ID = 0;
//
//   DBG_PRINT("fb: setting avsync time %lld to pts %lld\n", cur_time, time);
//
//   data.time_ms = time - cur_time;
//
//   DBG_PRINT("fb: pts time dif %ld\n", data.time_ms);
//
//   len = write(dsp->syncfd, &data, sizeof(TIME_DATA));
//   if(len == -1) {
//     DBG_PRINT("fb: error setting sync pts\n");
//     dsp->mode = DSP_MODE_ERROR;
//     dsp->error_status = -1;
//     //g_mutex_unlock(dsp->dsp_mutex);
//     return FALSE;
//   }
//
//   len = read(dsp->syncfd, &data, sizeof(TIME_DATA));
//   if(len == -1 || data.status != DSP_OK) {
//     DBG_PRINT("fb: error setting sync pts\n");
//     dsp->mode = DSP_MODE_ERROR;
//     dsp->error_status = data.status;
//     //g_mutex_unlock(dsp->dsp_mutex);
//     return FALSE;
//   }
//   //g_mutex_unlock(dsp->dsp_mutex);
//
//   return TRUE;
// }


/**
 *
 *
 */

static gboolean gst_dspfb_set_caps (GstBaseSink * bsink, GstCaps * caps) {

  GstDSPFBSink *dspfb = GST_DSPFBSINK(bsink);

  if(gst_dspfb_parse_caps(dspfb, caps) == GST_PAD_LINK_REFUSED) {
    GST_WARNING("Cannot use given caps\n");
    return FALSE;
  }
  DBG_PRINT("fb: linking successful\n");
  return TRUE;

}


/**
 *
 *
 */

// static GstFlowReturn
// gst_dspfb_preroll (GstBaseSink * bsink, GstBuffer * buf)
// {
//   DBG_PRINT("dspfb: preroll\n");
//   GstDSPFBSink *dspfb = GST_DSPFBSINK (bsink);
//
//   return GST_FLOW_OK;
// }

/**
 *
 *
 */
/*
static gboolean gst_dspfb_wait_data(int fd, int millis)
{
  struct timeval tv;
  fd_set fds;

  DBG_PRINT("WAITING DATA FOR %d millis\n", millis);

  tv.tv_sec = millis / 1000;
  tv.tv_usec = (millis % 1000) * 1000;
  FD_ZERO(&fds);
  FD_SET(fd, &fds);

  if(select(fd+1, &fds, NULL, NULL, &tv) > 0) {
    DBG_PRINT("DESCRIPTOR HAS DATA\n");
    return TRUE;
  }
  else {
    return FALSE;
  }
}
*/

/**
 *
 *
 */

static GstFlowReturn gst_dspfb_render (GstBaseSink * bsink, GstBuffer * inbuf) {

  GstDSPFBSink *dspfb;
  guchar *data;
  guint64 buffer_pts;
  guint32 bufsize;
  guint32 bsize;
  guint32 i;
  gboolean ret = FALSE;

  DBG_PRINT("fb: RENDER ENTER, incoming pts %lld\n",
            GST_BUFFER_TIMESTAMP(inbuf) / GST_MSECOND);


  if (!GST_CLOCK_TIME_IS_VALID(GST_BUFFER_TIMESTAMP(inbuf))) {
    DBG_PRINT("fb: invalid timestamp, dropping\n");
    return GST_FLOW_OK;
  }

  dspfb = GST_DSPFBSINK (bsink);

  gst_buffer_ref(inbuf);
  g_queue_push_tail(dspfb->buf_queue, inbuf);

  while( !dspfb->video->stream_id_set && !dspfb->video->videoonly ) {
    DBG_PRINT("fb: waiting for audio\n");
    g_usleep(200000);

    if ( g_queue_get_length(dspfb->buf_queue) >= DSPFB_QUEUE_MAX_FRAMES &&
         !dspfb->paused ){
      continue;
    }
    return GST_FLOW_OK;
  }

  while (!g_queue_is_empty(dspfb->buf_queue)){
    inbuf = g_queue_peek_head(dspfb->buf_queue);
    buffer_pts = GST_BUFFER_TIMESTAMP(inbuf);

    if(dspfb->paused == TRUE) {
      DBG_PRINT("fb: paused during render\n");
      return GST_FLOW_OK;
    }

    dspfb->avsync_pts = gst_dspvideo_get_time(dspfb->video);
    DBG_PRINT("fb: avsync_pts %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(dspfb->avsync_pts));
    if (dspfb->wait_pts) {
      // This means DISCONT
      dspfb->diff_pts = dspfb->avsync_pts - dspfb->segment_start;
      dspfb->wait_pts = FALSE;
      dspfb->prev_gst_pts = buffer_pts;
      DBG_PRINT("fb: setting diff pts %lld\n", dspfb->diff_pts);
    }
  
    // Sanity check. Sometimes the encoding/decoding messes things up
    // and timestamps are shrinking.
    if (buffer_pts < dspfb->prev_gst_pts) {
      DBG_PRINT("fb: given pts %lld is earlier than %lld, dropping frame\n",
                buffer_pts, dspfb->prev_gst_pts);
      gst_buffer_unref(g_queue_pop_head(dspfb->buf_queue));
      continue;
    }
  
    data = GST_BUFFER_DATA (inbuf);
    bufsize = GST_BUFFER_SIZE (inbuf);
  
    DBG_PRINT("fb: diff to prev %"GST_TIME_FORMAT" \n", GST_TIME_ARGS(buffer_pts - dspfb->prev_gst_pts));

    guint64 pts = buffer_pts + dspfb->diff_pts;
    dspfb->prev_gst_pts = buffer_pts;
    GstClockTimeDiff diff = dspfb->avsync_pts - pts;

  //   DBG_PRINT("fb: buffer pts %lli, pts %lli, avsync %lli\n",
  // 	    buffer_pts, pts, dspfb->avsync_pts );
    
    if(pts < dspfb->avsync_pts) {
      // We're late, let's drop frame.
      if ( g_queue_get_length(dspfb->buf_queue) > 1 ){
        if ( dspfb->qos_state != FBSINK_QOS_HURRY ){
          DBG_PRINT("Pushing HURRY event1\n");
          gst_pad_push_event (dspfb->sinkpad, gst_event_new_qos(0.5, diff, buffer_pts));
          dspfb->qos_state = FBSINK_QOS_HURRY;
        }
      } else {
        if ( dspfb->qos_state != FBSINK_QOS_SKIP ){
          DBG_PRINT("Pushing SKIP event\n");
          gst_pad_push_event (dspfb->sinkpad, gst_event_new_qos(1.0, diff, buffer_pts));
          dspfb->qos_state = FBSINK_QOS_SKIP;
        }
      }
      DBG_PRINT("fb: --------------------------------------------   "
                "DROPPING frame1. avsync ts %"GST_TIME_FORMAT", buffer ts %"GST_TIME_FORMAT"\n",
                 GST_TIME_ARGS(dspfb->avsync_pts), GST_TIME_ARGS(pts));
      gst_buffer_unref(g_queue_pop_head(dspfb->buf_queue));
      dspfb->drop_count++;
      continue;
    } else {
      if ( dspfb->qos_state == FBSINK_QOS_SKIP ){
        dspfb->qos_state = FBSINK_QOS_HURRY;
        DBG_PRINT("Pushing HURRY event2\n");
        gst_pad_push_event (dspfb->sinkpad, gst_event_new_qos(0.5, diff, buffer_pts));
      } 
    }
    
    g_mutex_lock(dspfb->video->dsp_mutex);
    // FIXME: added g_usleep instead of blocking read pending.
    if ( g_queue_get_length(dspfb->buf_queue) >= DSPFB_QUEUE_MAX_FRAMES ){
      while ( ret == FALSE ) {
        if ( dspfb->paused ) break;
        ret = gst_dspvideo_read_pending(dspfb->video, TRUE, FALSE, FALSE);
        g_usleep(10000);
      }
      if ( dspfb->qos_state == FBSINK_QOS_HURRY ){
        dspfb->qos_state = FBSINK_QOS_NORMAL;
        DBG_PRINT("Pushing NORMAL event\n");
        gst_pad_push_event (dspfb->sinkpad, gst_event_new_qos(0, diff, buffer_pts));
      }
    } else {
      ret = gst_dspvideo_read_pending(dspfb->video, TRUE, FALSE, FALSE);
    }
    g_mutex_unlock(dspfb->video->dsp_mutex);
  
    if ( !ret ) {
      // No write request waiting, let's exit.
      break;
    }
  
    // Check the node for error
    if(dspfb->video->mode == DSP_MODE_ERROR) {
      DBG_PRINT("fb: dsp error\n");
      GST_ELEMENT_ERROR (dspfb, RESOURCE, WRITE, (NULL),
                        ("chain : error status: %d",
                          dspfb->video->error_status));
      while( !g_queue_is_empty(dspfb->buf_queue) ) {
        gst_buffer_unref(g_queue_pop_head(dspfb->buf_queue));
      }
      return GST_FLOW_ERROR;
    }

    bsize = dspfb->write_size - 2*sizeof(short int);
  
    guint32 Y_size = dspfb->video->width * dspfb->video->height;
    i = Y_size + (Y_size / 2);
  
    if ( i != bufsize ) {
      DBG_PRINT("fb: wrong bufsize: it is %d, should be %d.\n", bufsize, i);
    }
  
      // Copy data
    if( dspfb->video->write_pending &&
        dspfb->video->mode != DSP_MODE_ERROR ) {
  
      memcpy( &(dspfb->mmap_buf_addr)[0], data, Y_size );
      memcpy( &(dspfb->mmap_buf_addr)[0x20000], &data[Y_size], Y_size / 2);
  
      // Write timestamp to DSP
      short int wcmd[3];
      guint64 pts_millis = pts / GST_MSECOND;
  
      wcmd[0] = DSP_CMD_DATA_WRITE;
      wcmd[1] = pts_millis >> 16 & 0x0000ffff;
      wcmd[2] = pts_millis & 0x0000ffff;
  
      g_mutex_lock(dspfb->video->dsp_mutex);
      write(dspfb->video->fd, &wcmd, sizeof(short int)*3);
      dspfb->video->write_pending = FALSE;
      dspfb->data_written = TRUE;
      g_mutex_unlock(dspfb->video->dsp_mutex);

      if (dspfb->seeking)
        dspfb->seeking = FALSE;
  
      gst_buffer_unref(g_queue_pop_head(dspfb->buf_queue));
      break;

    }
    else {
      DBG_PRINT("fb: data copy failed\n");
    }
  }

  DBG_PRINT("fb: RENDER EXIT, %d buffers queued\n",g_queue_get_length(dspfb->buf_queue));

  return GST_FLOW_OK;
}




/**
 *
 *
 */

static GstStateChangeReturn
gst_dspfb_initialize(GstDSPFBSink *dsp)
{
  DBG_PRINT("fb: initializing dspfb\n");

  DSP_CMD_STATUS cmd;
  VIDEO_INIT_STATUS status;
  VIDEO_PARAMS_DATA init_data;

  short int tmp;

  while(gst_dspvideo_check_read(dsp->video)) {
    read(dsp->video->fd, &tmp, sizeof(short int));
    //g_print("fb: from DSP: %d\n", tmp );
  }

  cmd.dsp_cmd = DSP_CMD_INIT;

  // AV sync enable
  cmd.status = 1;
  write(dsp->video->fd, &cmd, 2*sizeof(short int));

  read(dsp->video->fd, &status, sizeof(VIDEO_INIT_STATUS));
  if(status.init_status != DSP_OK) {
    dsp->mode = DSP_MODE_ERROR;
    g_warning("failed to initialize dsp DSP FB node, error = 0x%02X\n",
        status.init_status);
    return FALSE;
  }

  //g_print("mmap size from init: %d\n", status.mmap_buffer_size );
  dsp->video->dsp_buffer_size = status.mmap_buffer_size;
  dsp->write_size = status.mmap_buffer_size << 1;

  //printf("DSP BUFSIZE = %d\n", dsp->video->dsp_buffer_size );
  //printf("WRITE_SIZE = %d\n", dsp->write_size);

  //g_print("mmapping the device to memory\n");
  if ( ( dsp->mmap_buf_addr = (unsigned char *)mmap((void *)0,
         0x30000*sizeof(short int), // dsp->write_size * sizeof(unsigned char),
         PROT_READ | PROT_WRITE, MAP_SHARED, dsp->video->fd, 0) ) < 0) {
    g_warning("\nError mapping mmap buffer.\n\n");
    return FALSE;
  } else {
    //g_print("mmap successfull\n");
  }

  if ( dsp->video->width < 1 ) dsp->video->width = 16;
  if ( dsp->video->height < 1 ) dsp->video->height = 16;

  init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  init_data.fbuf_address = 0x180000;
  init_data.fbuf_height = DSP_SCRNHEIGHT_DEFAULT;
  init_data.fbuf_width = DSP_SCRNWIDTH_DEFAULT;
  init_data.fbuf_bpp = 16;
  init_data.fbuf_orientation = 0;

  init_data.outImageHeight = dsp->video->height;
  init_data.outImageWidth = dsp->video->width;

  init_data.outImagePos_x = dsp->video->xpos;
  init_data.outImagePos_y = dsp->video->ypos;

  init_data.inImageWidth = dsp->video->width;
  init_data.inImageHeight = dsp->video->height;

  init_data.audio_strm_ID = 0;

  if ( init_data.outImageWidth & 0x0001 ) {
    init_data.outImageWidth--;
  }
  if ( init_data.outImageHeight & 0x0001 ) {
    init_data.outImageHeight--;
  }

//  init_data.outImageWidth = dsp->video->width;
//  init_data.outImageHeight = dsp->video->height;

 // we don't scale to full screen size initially
  if ( ( dsp->video->scrn_width != DSP_SCRNWIDTH_DEFAULT ) &&
       ( dsp->video->scrn_height != DSP_SCRNHEIGHT_DEFAULT ) ) {
    // Aspect ratio calc
    gdouble x_ratio = (gdouble) dsp->video->scrn_width / dsp->video->width;
    gdouble y_ratio = (gdouble) dsp->video->scrn_height / dsp->video->height;

    // scale by smaller ratio
    if ( x_ratio < y_ratio ) {
      init_data.outImageWidth = dsp->video->scrn_width;
      init_data.outImageHeight = lround ( dsp->video->height * x_ratio );
    } else {
      init_data.outImageHeight = dsp->video->scrn_height;
      init_data.outImageWidth = lround ( dsp->video->width * y_ratio );
    }
  } else {
    init_data.outImageHeight = dsp->video->height;
    init_data.outImageWidth = dsp->video->width;
  }

  init_data.outImagePos_x +=
    (dsp->video->scrn_width - init_data.outImageWidth) / 2;
  init_data.outImagePos_y +=
    (dsp->video->scrn_height - init_data.outImageHeight) / 2;

  if ( init_data.outImageWidth & 0x0001 ) {
    init_data.outImageWidth--;
  }
  if ( init_data.outImageHeight & 0x0001 ) {
    init_data.outImageHeight--;
  }


  DBG_PRINT("Init params: x:%d, y:%d, w:%d, h:%d, sw:%d, sh:%d\n",
  dsp->video->xpos, dsp->video->ypos, dsp->video->width,
   dsp->video->height, dsp->video->scrn_width,
   dsp->video->scrn_height );


  if(!gst_dspvideo_initialize(dsp->video, &init_data,
    sizeof(init_data))) {
    //g_print("fb: dspvideo init failed\n");
    return FALSE;
  }

  cmd.dsp_cmd = DSP_CMD_VIDEO_RESET_LOCKEDRATIO;
  write(dsp->video->fd, &cmd, sizeof(short int));
  read(dsp->video->fd, &cmd, sizeof(DSP_CMD_STATUS));

  if ( cmd.status != DSP_OK ) {
    //g_print("locked ratio reset fail!\n");
    return FALSE;
  }
  else {
    //g_print("Locked ratio successfully reset.\n");
  }
  dsp->mode = DSP_MODE_INITIALIZED;
  dsp->video->mode = dsp->mode;
  dsp->parsed = FALSE;
  dsp->eos_mode = FALSE;

  dsp->win = 0;
  dsp->disp = NULL;
  dsp->display_name = NULL;
  dsp->drop_count = 0;

  DBG_PRINT("fb: initializing dspfb successfull\n");

  return TRUE;
}


/**
 * gst_dspfb_close
 * @dsp: DSP fb sink element
 *
 * Closes the DSP
 */

static GstStateChangeReturn
gst_dspfb_close(GstDSPFBSink *dsp)
{
  DBG_PRINT("fb: Closing DSP\n");

  DBG_PRINT("fb: ======== FRAMES DROPPED %d IN FBSINK =========\n", dsp->drop_count);

  munmap( dsp->mmap_buf_addr,
    0x30000 * sizeof(short int) );

  gst_dspvideo_close(dsp->video);

  return TRUE;
}


/**
 *
 *
 */

static void
gst_dspfb_set_property (GObject * object, guint prop_id, const GValue * value,
    GParamSpec * pspec)
{

  DBG_PRINT("fb: setting property...\n");

  GstDSPFBSink *dsp = GST_DSPFBSINK (object);

  //g_mutex_lock(dsp->video->dsp_mutex);
  switch (prop_id) {

  case DSPVIDEO_PROP_AUDIO_STREAMID:
    {
      guint16 tmp = g_value_get_uint(value);
      DBG_PRINT("Got streamid: %d\n", tmp );
      if (dsp->video->audio_stream_id != 0 && tmp == 0) {
        DBG_PRINT("fb: setting non-zero audiostream to zero\n");
        dsp->avsync_clock_set = TRUE;
      }
    }
  }
  if(!gst_dspvideo_set_property(dsp->video, prop_id, value)) {
    //printf("TRYING TO SET ILLEGAL PROPERTY\n");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }

  DBG_PRINT("fb: setting property done\n");
  //g_mutex_unlock(dsp->video->dsp_mutex);

}


/**
 *
 *
 */

static void
gst_dspfb_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDSPFBSink *dsp = GST_DSPFBSINK (object);
  //g_mutex_lock(dsp->video->dsp_mutex);
  if(!gst_dspvideo_get_property(dsp->video, prop_id, value)) {
    //printf("TRYING TO GET ILLEGAL PROPERTY\n");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
  //g_mutex_unlock(dsp->video->dsp_mutex);
}





static GstStateChangeReturn
gst_dspfb_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPFBSink *dspfb = GST_DSPFBSINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      DBG_PRINT("fb: null to ready\n");

      if (gst_dspvideo_open(dspfb->video, device) == FALSE)
        goto error;

      dspfb->first_discont = TRUE;
      dspfb->prev_gst_pts = 0;
      dspfb->data_written = FALSE;
      dspfb->seeking = FALSE;
      dspfb->prevtime.tv_sec = dspfb->prevtime.tv_usec = 0;
      dspfb->qos_val = 0.0;
      dspfb->qos_ok = TRUE;
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      DBG_PRINT("fb: ready to paused\n");
      if( !gst_dspfb_initialize(dspfb) ) {
        DBG_PRINT("fb: cannot initialize DSP\n");
        goto error;
      }
      else {
        DBG_PRINT("fb: initializing DSP ok\n");
        break;
      }

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      DBG_PRINT("fb: paused to play\n");
      if(dspfb->prev_gst_pts == 0) {
        // PLAY first time
        gst_dspvideo_play(dspfb->video);
      }
      else {
        gst_dspvideo_play_sync(dspfb->video, dspfb->video->audio_stream_id);
      }
      dspfb->paused = FALSE;
      break;

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("fb: play to pause1\n");
      dspfb->paused = TRUE;
      break;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("fb: play to pause2, eos_mode = %d\n", dspfb->eos_mode);
      gst_dspvideo_pause_sync(dspfb->video);
      DBG_PRINT("fb: play to pause2 OK\n");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      DBG_PRINT("fb: pause to ready\n");
      gst_dspvideo_stop( dspfb->video );
      DBG_PRINT("fb: pause to ready stop done\n");
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      DBG_PRINT("fb: ready to null\n");
      // Close the sync node
      gst_dspfb_close(dspfb);
      // Close the video node
      gst_dspvideo_close_dsp(dspfb->video);
      break;

    default:
      break;
  }

  return ret;

error:
  GST_ELEMENT_ERROR (element, CORE, STATE_CHANGE, (NULL), (NULL));
  return GST_STATE_CHANGE_FAILURE;
}


/**
 *
 *
 */
/*
static GstClock *
gst_dspfb_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dspfb_provide_clock\n");
  GstDSPFBSink *dsp = GST_DSPFBSINK (elem);
  return GST_CLOCK (gst_object_ref (GST_OBJECT (dsp->clock)));
}
*/

/**
 *
 *
 */
/*
static GstClockTime
gst_dspfb_get_time (GstClock * clock, GstDSPFBSink * sink)
{
//  struct timeval tv1, diff;

  if(sink->seeking) {
    DBG_PRINT("===================================== SEEKING -> CLOCK NOT VALID\n");
    return GST_CLOCK_TIME_NONE;
  }

//  gettimeofday(&tv1, NULL);
//  timersub(&tv1, &(sink->prevtime), &diff);
//  sink->prevtime.tv_sec  = tv1.tv_sec;
//  sink->prevtime.tv_usec = tv1.tv_usec;

//  if(diff.tv_sec) {
//    sink->clocktime = gst_dspvideo_get_time(sink->video) - sink->diff_pts;
//  }
//  else {
//    sink->clocktime += diff.tv_usec * GST_USECOND;
  //  }
  sink->clocktime = gst_dspvideo_get_time(sink->video) - sink->diff_pts;
  DBG_PRINT("========================= gst_dspfb_get_time: %lld\n", sink->clocktime);
  return sink->clocktime;
}
*/

/**
 *
 *
 */

static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspfbsink", GST_RANK_PRIMARY,
      GST_TYPE_DSPFBSINK);
}


GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "dspfbsink",
    "DSP FrameBuffer video sink",
    plugin_init, VERSION, "LGPL", "dspfbsink", "Nokia Corporation");
