/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <unistd.h>
#include <gst/gst.h>

#include "gstdspmp3sink.h"

GstElementDetails gst_dspmp3sink_details = GST_ELEMENT_DETAILS (
    "DSP MP3 Sink",
    "Sink/Audio",
    "MP3 audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>>");

static gchar *dsp_node_names[3] =
{
  NULL,
  "/dev/dsptask/mp2dec",
  "/dev/dsptask/mp3dec"
};

gboolean char_in_tmp = FALSE;
char tmp_char;

// MP3 bitrate lookup tables
static guint bitrates[2][3][16] =
{
  {
    {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, 0},
    {0, 32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384, 0},
    {0, 32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 0}
  },
  {
    {0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, 0},
    {0,  8, 16, 24, 32, 40, 48,  56,  64,  80,  96, 112, 128, 144, 160, 0},
    {0,  8, 16, 24, 32, 40, 48,  56,  64,  80,  96, 112, 128, 144, 160, 0}
  }
};

static guint mp3_samplerates[4][3] =
{
  { 11025, 12000,  8000, },   // MPEG version 2.5
  {     0,     0,     0, },   // Reserved
  { 22050, 24000, 16000, },   // MPEG version 2
  { 44100, 48000, 32000, }    // MPEG version 1
};

static int bs[3] = {384, 1152, 1152};

static guint gst_dspmp3sink_signals[DSPMP3_SIGNAL_LAST] = { 0 };


// Xing MP3 frame header related constants
#define XING_PEEK_BYTES 2048

#define FRAMES_FLAG     0x0001
#define BYTES_FLAG      0x0002
#define TOC_FLAG        0x0004
#define VBR_SCALE_FLAG  0x0008


static GstStaticPadTemplate dspmp3sink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/mpeg, "
                      "mpegversion = (int) 1, "
                      "layer = (int) [2, 3]" ) );

// Function prototypes

static void gst_dspmp3sink_class_init (GstDSPMP3SinkClass * klass);
static void gst_dspmp3sink_base_init (gpointer g_class);
static void gst_dspmp3sink_init (GstDSPMP3Sink *dspmp3sink, GstDSPMP3SinkClass *g_class);
static void gst_dspmp3sink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspmp3sink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);
static gboolean gst_dspmp3sink_query (GstElement * element, GstQuery *query);
static const GstQueryType * gst_dspmp3sink_get_query_types(GstElement * element);

static gboolean gst_dspmp3sink_event (GstElement * element, GstEvent *event);

// For basesink
static GstFlowReturn gst_dspmp3sink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspmp3sink_event2 (GstBaseSink *sink, GstEvent *event);
static gboolean gst_dspmp3sink_setcaps(GstBaseSink *sink, GstCaps *caps);

static gboolean gst_dspmp3sink_initialize(GstDSPMP3Sink *dsp);
static GstStateChangeReturn gst_dspmp3sink_change_state (GstElement * element, GstStateChange transition);


/**
 *
 *
 */

void _do_init()
{
  DBG_PRINT("MP3 DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPMP3Sink, gst_dspmp3sink, GstBaseSink,
                      GST_TYPE_BASE_SINK, _do_init);

/**
 * gst_dspmp3sink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes MP3 sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */

static void
gst_dspmp3sink_dispose (GObject * object)
{
  GstDSPMP3Sink *dsp = (GstDSPMP3Sink *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  g_object_unref(G_OBJECT(dsp->adapter));
  G_OBJECT_CLASS (parent_class)->dispose(object);
}


/**
 * gst_dspmp3sink_base_init:
 * @g_class: MP3 sink class
 *
 * Does the basic initialization of MP3 sink element class. This is
 * called by GStreamer framework.
 */

static void
gst_dspmp3sink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspmp3sink_sink_template));
  gst_element_class_set_details (gstelement_class, &gst_dspmp3sink_details);
}

/**
 * gst_dspmp3sink_class_init:
 * @klass: MP3 Sink class object
 *
 * Initializes MP3 sink element class. This is called by GStreamer
 * framework.
 */

static void
gst_dspmp3sink_class_init (GstDSPMP3SinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = gst_dspmp3sink_set_property;
  gobject_class->get_property = gst_dspmp3sink_get_property;
  gobject_class->dispose = gst_dspmp3sink_dispose;

  gstelement_class->change_state = gst_dspmp3sink_change_state;
  gstelement_class->get_query_types = gst_dspmp3sink_get_query_types;
  gstelement_class->send_event = gst_dspmp3sink_event;
  gstelement_class->query = gst_dspmp3sink_query;

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  gst_dspmp3sink_signals[DSPMP3_SIGNAL_STREAMID] =
      g_signal_new ("stream_id", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST,
                    G_STRUCT_OFFSET (GstDSPMP3SinkClass, stream_id), NULL, NULL,
                    g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);

  gstbase_sink_class->render = GST_DEBUG_FUNCPTR (gst_dspmp3sink_render);
  gstbase_sink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dspmp3sink_setcaps);
  gstbase_sink_class->event = GST_DEBUG_FUNCPTR (gst_dspmp3sink_event2);
}


/**
 * gst_dspmp3sink_init:
 * @dsp: DSP MP3 sink object
 *
 * Initializes the given MP3 sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */

static void
gst_dspmp3sink_init (GstDSPMP3Sink * dspmp3sink, GstDSPMP3SinkClass *g_class)
{
  GstBaseSink *bsink = GST_BASE_SINK(dspmp3sink);
  bsink->sync = FALSE;
  dspmp3sink->sinkpad = bsink->sinkpad;

  dspmp3sink->audio = gst_dspaudio_new();
  dspmp3sink->adapter = gst_adapter_new();
}


/**
 * Utility method for handling the troublesome odd character
 * in the end of MP3 frame (MMAP buffer must always be filled
 * with even number of data, so we must store the last odd
 * byte).
 */

static inline void
gst_dspmp3sink_get_missing_char(GstDSPMP3Sink *dsp)
{
  if(char_in_tmp) {
    dsp->audio->codec.mmap_buffer[dsp->mmap_ptr] = tmp_char;
    char_in_tmp = FALSE;
    dsp->total_bytes++;
    dsp->mmap_ptr++;
  }
}


/**
 * Calculate Bits-per-Second (?) value
 *
 */

gdouble
gst_dspmp3sink_bpf(MP3Frame *frame, guint bitrate)
{
  // Calculate "bytes-per-frame" value
  gdouble bpf = bitrate;
  switch (frame->layer) {
    case 0:
      bpf *= 12000.0 * 4.0;
      bpf /= frame->sample_rate_hz << frame->lsf;
      break;
    case 1:
/*      bpf *= 144000;
      bpf /= frame->sample_rate_hz;
      break;*/
    case 2:
      bpf *= 144000;
      bpf /= frame->sample_rate_hz << frame->lsf;
      break;
    default:
      bpf = 1.0;
  }
  DBG_PRINT("SR = %d, BR = %d, BPF = %f, lsf = %d\n", frame->sample_rate_hz,bitrate, bpf, frame->lsf);
  return bpf;
}


/**
 * Calculate frame size in bytes including header.
 *
 */

int
gst_dspmp3sink_framesize(MP3Frame *frame)
{
  // Calculate "bytes-per-frame" value
  int size = frame->bitrate;
  switch (frame->layer) {
    case 0:
      size *= 12000;
      size /= frame->sample_rate_hz;
      size = (size + frame->pad) << 2;
      break;
    case 1:
      size *= 144000;
      size /= frame->sample_rate_hz;
      size += frame->pad;
      break;
    case 2:
      size *= 144000;
      size /= frame->sample_rate_hz << frame->lsf;
      size += frame->pad;
      break;
    default:
      size = 0;
  }
  return size;
}


/**
 *
 *
 */

static void gst_dspmp3sink_send_discont(GstDSPMP3Sink *dsp)
{
  if(dsp->total_bytes > 0)
  {
    if(gst_dspaudio_discont(dsp->audio)) {
      DBG_PRINT("DISCONT written... reading reply\n");
      if(gst_dspaudio_read_cmd(dsp->audio, 0) != DSP_WAIT_OK) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, SEEK, (NULL), ("discont"));
      }
      DBG_PRINT("GOT REPLY\n");
      char_in_tmp = FALSE;
    }
  }
}




/**
 * gst_dspmp3sink_head_check:
 * @head: MP3 header in long int format
 *
 * Checks if the given 32-bit value contains proper MP3 header
 *
 * Returns: TRUE iff the header is valid
 */

gboolean gst_dspmp3sink_head_check(guint32 head)
{
//  if ((head & 0xfff00000) != 0xfff00000)
  if ((head & 0xffe00000) != 0xffe00000)
  return FALSE;
  if (!((head >> 17) & 3))
    return FALSE;
  if (((head >> 12) & 0xf) == 0xf)
    return FALSE;
  if (!((head >> 12) & 0xf))
    return FALSE;
  if (((head >> 10) & 0x3) == 0x3)
    return FALSE;
  if (((head >> 19) & 1) == 1 &&    // MPEG V1 ?
        ((head >> 17) & 3) == 3 &&  // Layer 1 ?
        ((head >> 16) & 1) == 1)
    return FALSE;
  if ((head & 0xffff0000) == 0xfffe0000)
    return FALSE;
  if (((head >> 19) & 0x3) == 0x1)
    return FALSE;

  return TRUE;
}


static inline guint32 gst_dspmp3sink_convert_to_header(const guint8 * buf) {
  return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}

/**
 * gst_dspmp3sink_find_sync:
 * @dsp: DSP MP3 sink object
 *
 * Tries to find the MP3 "magic key" inside the incoming
 * data stream. This key is can be used as a synchronization
 * marker, since it is a frame boundary identifier.
 */

static guint32
gst_dspmp3sink_find_sync(GstDSPMP3Sink *dsp, gboolean *gap_detected)
{
  guint32 head = 0;
  const guint8 *data = NULL;
  guint fcount = 0;
  guint avail = gst_adapter_available(dsp->adapter);

  while( avail >= 4 ) {

    data = gst_adapter_peek(dsp->adapter, 4);
    head = gst_dspmp3sink_convert_to_header(data);
    if(gst_dspmp3sink_head_check(head)) {
      if(fcount) {
        DBG_PRINT("\n-------------------------- FIND SYNC SKIPPED %d bytes\n", fcount);
      }
      return head;
    }
    gst_adapter_flush(dsp->adapter, 1);
    *gap_detected = TRUE;
    fcount++;
    avail--;
  }
  return 0;
}


/**
 * Checks if the header contains Xing extension. If so, extract the
 * number of frames.
 *
 * @dsp: DSP MP3 sink object
 * @data: the buffer containing first frame of the MP3 stream
 * Returns: Number of frames information, or 0 if no Xing header found
 */

#define GET_INT32BE(b) \
  (i = (b[0] << 24) | (b[1] << 16) | b[2] << 8 | b[3], b += 4, i)

gboolean
gst_dspmp3sink_mp3_xing_check(MP3Frame *frame, XINGTag *tag, gchar *data)
{
  guint i, head_flags;
  gchar mpegid, channelmode;

  // Later we assume that XING header is available if this
  // variable is non-zero
  tag->xframes = 0;

  mpegid      = (data[1] >> 3) & 1;   // Get MPEG version
  channelmode = (data[3] >> 6) & 3;   // Get channel mode
  data += 4;                          // Skip "normal header"

  // Calculate the index for XING
  if(mpegid) {
    if (channelmode != 3)
      data += 32;
    else
      data += 17;
  }
  else {
    if (channelmode != 3)
      data += 17;
    else
      data += 9;
  }

  // Is there a Xing header?
  if(strncmp(data, "Xing", 4)) {
    return 0;
  }

  DBG_PRINT("XING TAG FOUND\n");

  data += 4;
  head_flags = GET_INT32BE(data);

  if(head_flags & FRAMES_FLAG) {
    tag->xframes = GET_INT32BE(data);
  }

  if(head_flags & BYTES_FLAG) {
    tag->xbytes = GET_INT32BE(data);
  }

  if(head_flags & TOC_FLAG) {
    for (i = 0; i < 100; i++) {
      tag->toc[i] = data[i];
      if (i > 0 && tag->toc[i] < tag->toc[i - 1]) {
        return FALSE;
      }
    }
    if (tag->toc[99] == 0) {
      return FALSE;
    }
    data += 100;
  }
  else
    for (i = 0; i < 100; i++)
      tag->toc[i] = (i * 256) / 100;

  return TRUE;
}


/**
 *
 *
 */

void
gst_dspmp3sink_mp3_parse_head(MP3Frame *frame, guint32 head)
{
  short int br_index, sr_index;

  frame->mode     = (head & 0x00180000) >> 19;
  frame->layer    = 3 - ((head & 0x00060000) >> 17);  // between 0-2
  br_index        = (head & 0x0000F000) >> 12;
  sr_index        = (head & 0x00000C00) >> 10;
  frame->pad      = (head & 0x00000200) >> 9;
  frame->channels = ((head & 0x000000C0) >> 6) == 3 ? 1 : 2;

  if(frame->mode == 0) {
    frame->lsf = 1;
    frame->bitrate = bitrates[1][frame->layer][br_index];
  }
  else {
    frame->lsf = frame->mode == 3 ? 0 : 1;
    frame->bitrate = bitrates[frame->lsf][frame->layer][br_index];
  }
  frame->sample_rate_hz = mp3_samplerates[frame->mode][sr_index];
}


/**
 *
 *
 */

void inline
gst_dspmp3sink_mp3_frame_calc_tpf(MP3Frame *frame)
{
  frame->tpf = bs[frame->layer];
  frame->tpf /= (frame->sample_rate_hz << (frame->lsf));
  DBG_PRINT("FRAME_TPF = %f\n", frame->tpf);
}


/**
 * gst_dspmp3sink_get_stream_details:
 * @dsp: DSP MP3 sink object
 *
 * Parses the incoming MP3 data stream and extracts
 * stream information from it. This information is then
 * stored into DSP sink object itself.
 *
 * Returns: TRUE if the info was found, otherwise FALSE
 */

#define NUM_SAME_SAMPLERATES_NEEDED 5

static gboolean
gst_dspmp3sink_get_stream_details(GstDSPMP3Sink *dsp)
{

  MP3Frame first_frame;
  MP3Frame tmp_frame;
  gint same_sr_cnt = 0, sr_old, sr_curr = -1;
  guint count, curr_index = 0;
  gboolean found = FALSE;
  const guint8 *byte;
  guint32 head;
  guint flushamount;

  guint avail =  gst_adapter_available(dsp->adapter);

  //  guint layer1count = 0;

  bzero(&tmp_frame, sizeof(MP3Frame));
  bzero(&first_frame, sizeof(MP3Frame));

  // This ugly loop is needed because we want to find
  // certain amount of consecutive frames with same samplerate value

  DBG_PRINT("Starting SR loop\n");
  while(!found) {

    sr_old = sr_curr;

    if(same_sr_cnt == NUM_SAME_SAMPLERATES_NEEDED) {
//       DBG_PRINT("SYNC FOUND----------------------------------------\n");
      found = TRUE;
      break;
    }

    count = curr_index + 4;
    if(avail < count) {
      DBG_PRINT("No %d bytes available\n", count);
      return FALSE;
    }

    byte = gst_adapter_peek(dsp->adapter, count);
    head = gst_dspmp3sink_convert_to_header(&byte[curr_index]);

    if(gst_dspmp3sink_head_check(head)) {
      gst_dspmp3sink_mp3_parse_head(&tmp_frame, head);
      sr_curr = tmp_frame.sample_rate_hz;

      if(curr_index == 0) {
        // Store the first frame
        memcpy(&first_frame, &tmp_frame, sizeof(MP3Frame));
//         DBG_PRINT("Stored first frame\n");
      }

      if(dsp->total_bytes == 0 && dsp->xingtag.xframes == 0) {

        DBG_PRINT("CHECKING FOR XING\n");

        // Check for XING extension
        guint amount = count > XING_PEEK_BYTES ? count : XING_PEEK_BYTES;

        if(avail < amount) {
          DBG_PRINT("Cannot check XING, no data\n");
          return FALSE;
        }

        byte = gst_adapter_peek(dsp->adapter, amount);

        gst_dspmp3sink_mp3_xing_check(&tmp_frame, &dsp->xingtag,
                                       (gchar *) byte+curr_index);
      }

      if(sr_old == sr_curr || sr_old == -1) {
//         DBG_PRINT("FOUND SAME SAMPLERATES, index = %d\n", curr_index);
        curr_index += gst_dspmp3sink_framesize(&tmp_frame);
        same_sr_cnt++;
        continue;
      }
//       DBG_PRINT("SAMPLERATES NOT THE SAME\n");
    }

    flushamount = curr_index ? curr_index : 1;

    if(avail < flushamount) {
      DBG_PRINT("Cannot flush %d bytes\n", flushamount);
      return FALSE;
    }
    gst_adapter_flush(dsp->adapter, flushamount);
    avail -= flushamount;
    same_sr_cnt = 0;
    curr_index = 0;
  }

  // WOHOO, valid stream found. Backup frame information
  memcpy(&(dsp->frame), &first_frame, sizeof(MP3Frame));
  gst_dspmp3sink_mp3_frame_calc_tpf(&(dsp->frame));
  gst_dspaudio_map_samplerate(dsp->frame.sample_rate_hz,
                              &(dsp->sample_rate));

  DBG_PRINT("SRATE: %d, channels: %d\n", dsp->frame.sample_rate_hz,
            dsp->frame.channels);

  dsp->sync = TRUE;
  return TRUE;
}


/**
 * gst_dspmp3sink_setcaps:
 * @pad: GstPad that is the target for connect
 * @caps: GStCaps offered
 *
 * Returns: TRUE if caps are accepted
 */

static gboolean gst_dspmp3sink_setcaps(GstBaseSink *sink, GstCaps *caps)
{
  DBG_PRINT("gst_dspmp3sink_setcaps\n");

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);
  gint version, layer;

   gst_structure_get_int (structure, "mpegversion", &version);
   gst_structure_get_int (structure, "layer", &layer);

  if(strncmp(mimetype, "audio/mpeg", 10)) {
    return FALSE;
  }
  if(version != 1 && (layer<2 || layer>3)) {
    return FALSE;
  }

  return TRUE;
}


/**
 * gst_dspmp3sink_calculate_seekpos:
 * @dsp: DSP MP3 sink object
 * @time_offset: the time where to seek
 *
 * Calculate a seek position (in bytes) according to time_offset
 * stored in DSP structure
 *
 * Returns: seek position in bytes
 */

gint64 gst_dspmp3sink_calculate_seekpos(GstDSPMP3Sink *dsp, guint64 time_offset)
{
   // XING-based seek
  if(dsp->xingtag.xframes && dsp->xingtag.xbytes && dsp->frame.tpf) {
    return time_offset / GST_SECOND / dsp->frame.tpf *
        (dsp->xingtag.xbytes / dsp->xingtag.xframes);
  }
  else if (dsp->vbr) {
    GstPad *peer = gst_pad_get_peer(dsp->sinkpad);
    GstFormat fmt = GST_FORMAT_BYTES;
    gboolean ret = FALSE;
    gint64 curr, len;

    ret  = gst_pad_query_position (peer, &fmt, &curr);
    ret |= gst_pad_query_duration (peer, &fmt, &len);
    gst_object_unref(peer); // get_peer() ref'ed it

    if (curr >= 0 && len > 0 && ret) {
      gdouble rel = (gdouble) curr / len;
      guint bpf2 = dsp->total_bytes / (dsp->total_frames << dsp->frame.lsf);

      if(bpf2 != 0 && rel != 0) {
        gdouble currframes = curr / bpf2;
        gdouble written_time = dsp->frame.tpf * currframes;

        gdouble total = written_time / rel;
        guint64 tmp = time_offset / GST_SECOND;

        return (guint64) (tmp/total*len);
      }
    }
    return 0;
  }
  else if (dsp->frame.tpf) {
    gdouble bpf = gst_dspmp3sink_bpf(&dsp->frame,
                                      dsp->frame.bitrate);

    guint frame = (time_offset /  GST_SECOND) / dsp->frame.tpf;
    guint64 pos = (guint) (frame * bpf);
    return pos;
  }
  return -1;
}



/**
 * gst_dspmp3sink_perform_seek:
 * @dsp: DSP MP3 sink object
 *
 * Perform SEEK operation
 */

static gboolean
gst_dspmp3sink_perform_seek(GstDSPMP3Sink *dsp, guint64 time_offset)
{
  gint64 pos = gst_dspmp3sink_calculate_seekpos(dsp, time_offset);
  if(pos == -1) {
    return FALSE;
  }

  DBG_PRINT("SEEEEEEEEEKPOS: %lld\n", pos);

  GstEvent *new_event = gst_event_new_seek(1.0, GST_FORMAT_BYTES,
                                           GST_SEEK_FLAG_FLUSH,
                                           GST_SEEK_TYPE_SET, pos,
                                           GST_SEEK_TYPE_NONE,
                                           GST_CLOCK_TIME_NONE);

  if(gst_pad_push_event (dsp->sinkpad, new_event)) {
    g_mutex_lock(dsp->audio->dsp_mutex);
    gst_dspmp3sink_send_discont(dsp);
    if(gst_dspaudio_get_info(dsp->audio)) {
      gst_dspaudio_read_cmd(dsp->audio, 0);
      dsp->frame_seek_base = dsp->audio->info.num_frames;
    }
    g_mutex_unlock(dsp->audio->dsp_mutex);
    gst_adapter_clear(dsp->adapter);
    dsp->mmap_ptr = 0;
    dsp->curr_fc = 0;
    dsp->vbr_counter = 0;
    char_in_tmp = FALSE;
    DBG_PRINT("MMAP POINTER RESET\n");
  }
  else {
    DBG_PRINT("SEEKING FAILED\n");
//     GST_ELEMENT_ERROR (dsp, CORE, SEEK, (NULL), ("gst_event_new_seek"));
    return FALSE;
  }
  return TRUE;
}


/**
 * Initialize the DSP for playback and send PLAY command
 *
 */

static gboolean
gst_dspmp3sink_loop_initialization(GstDSPMP3Sink *dsp)
{
  if(!gst_dspmp3sink_get_stream_details(dsp)) {
    // We try again, render() function will fail
    // if this happens too many times
    return FALSE;
  }

  // Open the DSP audio device
  gchar *devname = dsp_node_names[dsp->frame.layer];

  g_mutex_lock(dsp->audio->dsp_mutex);
  if(!gst_dspaudio_open(dsp->audio, devname)) {
    g_mutex_unlock(dsp->audio->dsp_mutex);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  g_signal_emit (G_OBJECT (dsp), gst_dspmp3sink_signals[DSPMP3_SIGNAL_STREAMID],
                 0, dsp->audio->codec.stream_id);

  if(!gst_dspmp3sink_initialize(dsp)) {
    g_mutex_unlock(dsp->audio->dsp_mutex);
    GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                       ("gst_dspmp3sink_initialize"));
    return FALSE;
  }

  // Settings might have changed between PAUSE->PLAY transition
  gst_dspaudio_update_dsp_settings(dsp->audio);
  gst_dspaudio_play(dsp->audio);
  g_mutex_unlock(dsp->audio->dsp_mutex);
  return TRUE;
}


/**
 *
 *
 */

static gboolean
gst_dspmp3sink_fill_it(GstDSPMP3Sink *dsp, int btr)
{
  const guint8 *data;

  if(gst_adapter_available(dsp->adapter) < btr) {
    DBG_PRINT("fill_it cannot read %d bytes\n", btr);
    return FALSE;
  }

  data = gst_adapter_peek(dsp->adapter, btr);

  memcpy(&(dsp->audio->codec.mmap_buffer[dsp->mmap_ptr]), data, btr);
  gst_adapter_flush(dsp->adapter, btr);
  dsp->total_bytes += btr;
  dsp->mmap_ptr += btr;
  return TRUE;
}


/**
 * Fill DSP's MMAP buffer with MP3 frames
 *
 *
 * Returns: TRUE if operation was successful
 */

static gboolean
gst_dspmp3sink_copy_frame(GstDSPMP3Sink *dsp)
{
  gboolean gap_detected = FALSE;

  dsp->curr_fc = 0;

  guint32 head = gst_dspmp3sink_find_sync(dsp, &gap_detected);
  if(head != 0) {

    // Do we need to take a closer look to the stream?
    if(gap_detected || dsp->sync == FALSE) {
      // This guarantees that we really have found valid stream
      DBG_PRINT("Gap (%d) / Sync (%d) -> closer look\n",
          gap_detected, dsp->sync);
      if(!gst_dspmp3sink_get_stream_details(dsp)) {
        DBG_PRINT("Cannot examine stream\n");
        return FALSE;
      }
      char_in_tmp = FALSE;
    }
    else {
      gst_dspmp3sink_mp3_parse_head(&dsp->frame, head);
    }
    dsp->sync = TRUE;


    dsp->curr_fc++;
    dsp->frame_count++;
    dsp->total_frames++;
    dsp->bitrate_sum += dsp->frame.bitrate;
    dsp->avg_bitrate = dsp->bitrate_sum / dsp->frame_count;

    // Check if the stream is VBR
    if(dsp->prev_bitrate != dsp->frame.bitrate) {
      if(dsp->vbr_counter++ > 5) {
        dsp->vbr = TRUE;
      }
    }

    dsp->prev_bitrate = dsp->frame.bitrate;

    // Calculate avg. bitrate every 50 frames
    if(dsp->frame_count == 50) {
      dsp->bitrate_sum = 0;
      dsp->frame_count = 0;
    }

    gint btr = gst_dspmp3sink_framesize(&dsp->frame);

    gst_dspmp3sink_get_missing_char(dsp);
    if(!gst_dspmp3sink_fill_it(dsp, btr)) {
      DBG_PRINT("Cannot fill buffer - not enough data\n");
      return FALSE;
    }
//     DBG_PRINT("One frame copied. Size = %d\n", btr);

  }
  else {
    DBG_PRINT("Cannot find header\n");
    return FALSE; // Probably EOS
  }

  if((dsp->mmap_ptr % 2) == 1) {
    tmp_char = dsp->audio->codec.mmap_buffer[dsp->mmap_ptr-1];
    char_in_tmp = TRUE;
    dsp->mmap_ptr--;
  }
  return TRUE;
}


/**
 * gst_dspmp3sink_chain:
 * @pad:
 * @buffer:
 *
 * This method is automatically called by GStreamer framework, when
 * a scheduler decides to give execution to this element. It's
 * purpose is to process the data and forward it to DSP for playback.
 */

#define STREAM_DETECT_THRESHOLD 8000

static GstFlowReturn gst_dspmp3sink_render (GstBaseSink *sink, GstBuffer * buffer)
{
  gint ret;

  DBG_PRINT("MP3 RENDER: %d bytes\n", GST_BUFFER_SIZE(buffer));

  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);

  // We might be called from EOS handler. In this case buffer is NULL
  if(buffer) {
    gst_buffer_ref(buffer);
    gst_adapter_push (dsp->adapter, buffer);
  }

  // Do the possible initialization if not done yet...
  if(dsp->audio->mode == DSP_MODE_UNINITIALIZED) {

    if(gst_adapter_available(dsp->adapter) < STREAM_DETECT_THRESHOLD) {
      // First we need to gather enough data to detect the stream
      return GST_FLOW_OK;
    }

    if(gst_dspmp3sink_loop_initialization(dsp) == FALSE) {
      GST_ELEMENT_ERROR (sink, STREAM, TYPE_NOT_FOUND, (NULL),
                            ("gst_dspmp3sink_loop_initialization"));
         DBG_PRINT("RENDER RETURNING GST_FLOW_ERROR\n");
         return GST_FLOW_ERROR;
    }
    return GST_FLOW_OK;
  }

  while(gst_adapter_available(dsp->adapter) > 1400) {

    g_mutex_lock(dsp->audio->dsp_mutex);

    if(dsp->paused == TRUE) {
      DBG_PRINT("PAUSED DURING RENDER\n");
      g_mutex_unlock(dsp->audio->dsp_mutex);
      break;
    }

    if(gst_dspaudio_wait_buffer(dsp->audio) != DSP_WAIT_OK) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (sink, RESOURCE, READ, (NULL), ("gst_dspaudio_wait_buffer"));
      return GST_FLOW_ERROR;
    }

    dsp->audio->rw_pending = 1;  // Remember the DATA_WRITE

    if(!gst_dspmp3sink_copy_frame(dsp)) {
      DBG_PRINT("COPY_FRAME FAILED\n");
      g_mutex_unlock(dsp->audio->dsp_mutex);
      break;
    }
    dsp->outbuf.data_size = dsp->mmap_ptr / sizeof(short int);
    ret = write(dsp->audio->codec.fd, &dsp->outbuf, sizeof(MP3_DATA_WRITE));
    DBG_PRINT("WROTE %d bytes to DSP\n",  dsp->mmap_ptr);
    dsp->audio->rw_pending = 0;
    dsp->audio->discont_sent = FALSE;
    dsp->mmap_ptr = 0;
    g_mutex_unlock(dsp->audio->dsp_mutex);

    if(ret < 0) {
      GST_ELEMENT_ERROR (sink, RESOURCE, WRITE, (NULL), ("write returned -1"));
      return GST_FLOW_ERROR;
    }
  }
//   DBG_PRINT("MP3 RENDER EXIT\n");
  return GST_FLOW_OK;
}


/**
 * gst_dspmp3sink_initialize:
 * @dsp: DSP MP3 sink object
 *
 * Initializes the underlying audio stream object and sets up the
 * MP3 audio parameters in DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */

static gboolean
gst_dspmp3sink_initialize(GstDSPMP3Sink *dsp)
{
  AUDIO_PARAMS_DATA init_data;

   // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  init_data.sample_rate = dsp->sample_rate;
  init_data.number_channels = dsp->frame.channels;
  init_data.stream_priority = dsp->audio->priority;
  init_data.ds_stream_ID = 0;

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(AUDIO_PARAMS_DATA))) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  DBG_PRINT("DSP MP3 configured\n");
  return TRUE;
}


/**
 * gst_dspmp3sink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */

static void
gst_dspmp3sink_set_property (GObject * object,
                             guint prop_id,
                             const GValue *value,
                             GParamSpec * pspec)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dspmp3sink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */

static void
gst_dspmp3sink_get_property (GObject * object,
                             guint prop_id,
                             GValue *value,
                             GParamSpec * pspec)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 *
 *
 */

static const GstQueryType *
gst_dspmp3sink_get_query_types(GstElement * element)
{
  static const GstQueryType gst_dspmp3sink_query_types[] =
  {
    GST_QUERY_POSITION,
    GST_QUERY_DURATION,
    GST_QUERY_SEEKING,
    0
  };

  return gst_dspmp3sink_query_types;
}


/**
 * gst_dspmp3sink_query:
 * @element: GstElement that received this query
 * @type: Type of the query, e.g. TOTAL, POSITION
 * @format: The format of the response, e.g. BYTES, TIME
 * @value: Pointer where the return value should be stored
 *
 * Returns: TRUE, iff the query was successful.
 */

static gboolean
gst_dspmp3sink_query (GstElement * element, GstQuery *query)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (element);
  GstQueryType type = GST_QUERY_TYPE (query);
  gboolean res = FALSE;
  GstFormat format;
  gint64 value;

  /* We allow the query to be performed only when some data has
  already been written to DSP */
//  if(dsp->total_bytes == 0 || dsp->audio->stream_id == 0) {
  if(dsp->total_bytes == 0) {
    return FALSE;
  }

  if(type == GST_QUERY_POSITION) {
    gst_query_parse_position(query, &format, NULL);
    if (format != GST_FORMAT_TIME)
      return FALSE;

    if(dsp->audio->codec.stream_id) {
      /* Ask DSP how many frames it has played */
      g_mutex_lock(dsp->audio->dsp_mutex);
      if(gst_dspaudio_get_info(dsp->audio)) {
        gst_dspaudio_read_cmd(dsp->audio, 0);
      }
      g_mutex_unlock(dsp->audio->dsp_mutex);
    }

    /* This calculates the amount of frames played after last SEEK operation */
    guint32 frames_consumed = dsp->audio->info.num_frames - dsp->frame_seek_base;
    frames_consumed <<= dsp->frame.lsf;

    /* Time_offset is set during SEEK, so it works as a base time */
    value = dsp->time_offset + (frames_consumed  * dsp->frame.tpf) * GST_SECOND;
    gst_query_set_position (query, GST_FORMAT_TIME, value);
    res = TRUE;
  }

  else if(type == GST_QUERY_DURATION) {
    gst_query_parse_duration(query, &format, NULL);
    if (format != GST_FORMAT_TIME)
      return FALSE;

    if(gst_dspaudio_query_peer_duration(dsp->sinkpad, &value)) {
      res = TRUE;
    }
    else {
      /* XING-based length calculation */
      if(dsp->xingtag.xframes && dsp->frame.tpf != 0) {
        value = dsp->frame.tpf * dsp->xingtag.xframes * GST_SECOND;
        res = TRUE;
      }
      else { /* No XING */

        GstPad *peer = gst_pad_get_peer(dsp->sinkpad);
        GstFormat fmt = GST_FORMAT_BYTES;
        gboolean ret = FALSE;
        gint64 curr, len;

        ret  = gst_pad_query_position (peer, &fmt, &curr);
        ret |= gst_pad_query_duration (peer, &fmt, &len);
        gst_object_unref(peer); // get_peer() ref'ed it

        /* VBR needs different kind of calculation algorithm */
        if (dsp->vbr && len > 0 && curr >= 0) {
          gdouble rel = (gdouble) curr / len;
          gdouble bpf2 = 0;
          if(dsp->frame.layer == 1) {
            bpf2 = (gdouble) dsp->total_bytes / (dsp->total_frames << dsp->frame.lsf);
          }
          else {
            bpf2 = (gdouble) dsp->total_bytes / (dsp->total_frames);
          }
          DBG_PRINT("VBR BPF: %f\n", bpf2);

          if(bpf2 != 0.0 && rel != 0.0) {
            guint currframes = curr / bpf2;
            gdouble written_time = dsp->frame.tpf * currframes;
            value = GST_SECOND * written_time / rel;
            res = TRUE;
          }
        }

        /* Constant bitrate */
        else if(len != -1) {
          gdouble bpf = gst_dspmp3sink_bpf(&dsp->frame, dsp->frame.bitrate);
          DBG_PRINT("CBR BPF: %f\n", bpf);

          if(bpf != 0) {
            value = (guint64) (dsp->frame.tpf * len / bpf) * GST_SECOND;
            res = TRUE;
          }
        }
      }
    }
    if(res) {
      gst_query_set_duration (query, GST_FORMAT_TIME, value);
    }
  }
  return res;
}


/**
 *
 *
 */

static void
gst_dspmp3sink_send_remaining(GstDSPMP3Sink *dsp)
{
  DSPWaitStatus ret = DSP_WAIT_OK;

  if(gst_adapter_available(dsp->adapter)) {
    g_mutex_lock(dsp->audio->dsp_mutex);
    ret = gst_dspaudio_wait_buffer(dsp->audio);
    g_mutex_unlock(dsp->audio->dsp_mutex);

    if(ret != DSP_WAIT_OK) {
      return;
    }

    while(gst_dspmp3sink_copy_frame(dsp)) {
      DBG_PRINT("EOF FRAME COPIED\n");
    }

    if(dsp->mmap_ptr) {
      short int tmp[2];
      tmp[0] = DSP_CMD_EOF;
      tmp[1] = dsp->mmap_ptr / sizeof(short int);

      // Send remaining bytes to DSP
      DBG_PRINT("WRITING EOF CMD to DSP (%d short ints)\n", tmp[1]);
      g_mutex_lock(dsp->audio->dsp_mutex);
      write(dsp->audio->codec.fd, tmp, 2*sizeof(short int));
      DBG_PRINT("EOF Write performed\n");
      dsp->audio->rw_pending = 0;
      g_mutex_unlock(dsp->audio->dsp_mutex);
      dsp->total_bytes += dsp->mmap_ptr;
      dsp->audio->mode = DSP_MODE_EOS;
      gst_dspaudio_wait_eos(dsp->audio);
      DBG_PRINT("Wrote total of %lld bytes to DSP during playback\n", dsp->total_bytes);
    }
  }
  else {
    DBG_PRINT("NO DATA TO SEND -> NOT SENDING EOF\n");
  }
}


/**
 *
 * This method handles events coming from basesink
 */

static gboolean gst_dspmp3sink_event2 (GstBaseSink *sink, GstEvent * event)
{
  DBG_PRINT("^^^^^^^^^^^^^^^^^^^GOT EVENT FROM BASESINK: %d\n", GST_EVENT_TYPE (event));
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  if(type == GST_EVENT_EOS) {
    DBG_PRINT("EOS EVENT\n");
    dsp->audio->codec.stream_id = 0;
    g_signal_emit (G_OBJECT (sink), gst_dspmp3sink_signals[DSPMP3_SIGNAL_STREAMID], 0, 0);
    gst_dspmp3sink_send_remaining(dsp);
  }
/*  else if(type == GST_EVENT_NEWSEGMENT) {
    DBG_PRINT("NEWSEGMENT EVENT\n");
    g_mutex_lock(dsp->audio->dsp_mutex);
    gst_dspmp3sink_send_discont(dsp);
    dsp->mmap_ptr = 0;
    dsp->curr_fc = 0;
    dsp->vbr_counter = 0;
    char_in_tmp = FALSE;
    dsp->sync = FALSE;
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }*/
  else if(type == GST_EVENT_FLUSH_START) {
    DBG_PRINT("FLUSH START-------------------------------------------------------\n");
    g_mutex_lock(dsp->audio->dsp_mutex);
    gst_dspmp3sink_send_discont(dsp);
    gst_adapter_clear(dsp->adapter);
    dsp->mmap_ptr = 0;
    dsp->curr_fc = 0;
    dsp->vbr_counter = 0;
    char_in_tmp = FALSE;
    dsp->sync = FALSE;
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }
  else if(type == GST_EVENT_FLUSH_STOP) {
    DBG_PRINT("FLUSH STOP-------------------------------------------------------\n");
    gst_adapter_clear(dsp->adapter);
  }
  return TRUE;
}

/**
 * gst_dspmp3sink_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, iff the event was successful
 */

static gboolean gst_dspmp3sink_event (GstElement * element, GstEvent * event)
{
  GstDSPMP3Sink *dsp = GST_DSPMP3SINK (element);
  gboolean res = FALSE;

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;

  DBG_PRINT("EVENT------------------------------------------------------: %d \n", type);

  if(type == GST_EVENT_SEEK && dsp->audio->mode != DSP_MODE_EOS) {
    gdouble rate;
    GstFormat format;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur, stop;

    gst_event_parse_seek (event, &rate, &format, &flags,
                          &cur_type, &cur, &stop_type, &stop);

    if(format == GST_FORMAT_TIME) {
      DBG_PRINT("SEEK--------------------------------------------------------\n");
      gint64 tmp = dsp->time_offset;
      dsp->time_offset = cur;

      // Wait until the stream is detected
      while(dsp->frame.bitrate == 0) {
        DBG_PRINT("WAITING STREAM INFO\n");
        usleep(20000);
      }
      DBG_PRINT("PERFORMING SEEK\n");
      if(gst_dspmp3sink_perform_seek(dsp, cur)) {
        res = TRUE;
      }
      else {
        dsp->time_offset = tmp;
      }
    }
    gst_event_unref(event);
  }
  else {
    res = gst_pad_event_default(dsp->sinkpad, event);
  }
  return res;
}


/**
 * gst_dspmp3sink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Changes the state of given element. The desired state information
 * is contained by the element itself, and this method is just
 * a callback, so the plugin knows when the state is changing.
 * This method is called by GStreamer framework
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */

static GstStateChangeReturn
gst_dspmp3sink_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPMP3Sink *dspmp3sink = GST_DSPMP3SINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      DBG_PRINT("MP3 NULL TO READY\n");
      dspmp3sink->total_bytes = 0;
      dspmp3sink->total_frames = 0;
      dspmp3sink->vbr = FALSE;
      dspmp3sink->mmap_ptr = 0;
      dspmp3sink->xingtag.xframes = 0;
      dspmp3sink->frame_seek_base = 0;
      dspmp3sink->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
      char_in_tmp = FALSE;
      bzero(&dspmp3sink->frame, sizeof(MP3Frame));
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      DBG_PRINT("MP3 READY TO PAUSED\n");
      gst_adapter_clear(dspmp3sink->adapter);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      DBG_PRINT("MP3 PAUSED TO PLAYING\n");
      g_mutex_lock(dspmp3sink->audio->dsp_mutex);
      DBG_PRINT("MP3 PAUSED TO PLAYING GOT MUTEX\n");

      if(dspmp3sink->audio->mode == DSP_MODE_PAUSED) {
        // Set the play only if we are at paused state
        gst_dspaudio_play(dspmp3sink->audio);
        dspmp3sink->audio->rw_pending = 0;
      }
      DBG_PRINT("MP3 PAUSED TO PLAYING OK\n");
      dspmp3sink->paused = FALSE;
      g_mutex_unlock(dspmp3sink->audio->dsp_mutex);
      break;

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("MP3 PLAYING TO PAUSED V1\n");
      dspmp3sink->paused = TRUE;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch(transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("MP3 PLAYING TO PAUSED V2\n");
      if(dspmp3sink->audio->mode != DSP_MODE_EOS) {
        g_mutex_lock(dspmp3sink->audio->dsp_mutex);
        if(gst_dspaudio_pause(dspmp3sink->audio)) {
          gst_dspaudio_read_cmd(dspmp3sink->audio, 0);
          DBG_PRINT("PAUSE command sent to DSP\n");
          dspmp3sink->audio->rw_pending = 0;
        }
        g_mutex_unlock(dspmp3sink->audio->dsp_mutex);
      }
      DBG_PRINT("MP3 PLAYING TO PAUSED V2 OK\n");
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      DBG_PRINT("MP3 PAUSED TO READY\n");
/*      if(dspmp3sink->audio->mode == DSP_MODE_EOS) {
        gst_dspaudio_wait_eos(dspmp3sink->audio);
      }
      else*/ if(gst_dspaudio_stop(dspmp3sink->audio)) {
        gst_dspaudio_read_cmd(dspmp3sink->audio, 0);
      }
      gst_adapter_clear(dspmp3sink->adapter);
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      DBG_PRINT("MP3 READY To NULL\n");
      gst_dspaudio_close(dspmp3sink->audio);
      gst_dspaudio_reset(dspmp3sink->audio);
      DBG_PRINT("MP3 READY To NULL SUCCESSFUL\n");
      break;

    default:
      break;
  }

  DBG_PRINT("MP3 STATE CHANGE SUCCESSFUL, retval = %d\n", ret);
  return ret;
}


/**
 * plugin_init:
 * @plugin: GstPlugin to be initialized
 *
 * This initializes the plugin when it is loaded by GStreamer
 *
 * Returns: Boolean value, TRUE if initialization was successful
 */

static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspmp3sink", GST_RANK_PRIMARY,
      GST_TYPE_DSPMP3SINK);
}


/**
 * This is the "boilerplate" of the plugin
 *
 */

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
                   GST_VERSION_MINOR,
                   "dspmp3",
                   "DSP MP3 audio sink",
                   plugin_init,
                   VERSION,
                   GST_LICENSE_UNKNOWN,
                   "dspaudio",
                   "Nokia Corporation");
