/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspilbcsrc.h"

GstElementDetails gst_dspilbcsrc_details = GST_ELEMENT_DETAILS ("DSP ILBC Src",
    "Source/Audio",
    "ILBC audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

static gchar *devname = "/dev/dsptask/ilbc_enc";

static GstStaticPadTemplate dspilbcsrc_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/x-iLBC, "
                      "rate = (int) 8000, "
                      "channels = (int) 1, "
                      "mode = (int) { 20, 30 }"
                    ) );

// Function prototypes

static void gst_dspilbcsrc_class_init (GstDSPILBCSrcClass * klass);
static void gst_dspilbcsrc_base_init (gpointer g_class);
static void gst_dspilbcsrc_init (GstDSPILBCSrc *dspmp3src, GstDSPILBCSrcClass *g_class);
static void gst_dspilbcsrc_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspilbcsrc_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstCaps *gst_dspilbcsrc_getcaps (GstBaseSrc * src);
static gboolean gst_dspilbcsrc_setcaps (GstBaseSrc * src, GstCaps *caps);

static GstFlowReturn gst_dspilbcsrc_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dspilbcsrc_start (GstBaseSrc * bsrc);
static gboolean gst_dspilbcsrc_stop (GstBaseSrc * bsrc);

static GstStateChangeReturn
gst_dspilbcsrc_change_state (GstElement * element, GstStateChange transition);


/**
 *
 *
 */

void _ilbcsrc_do_init()
{
  DBG_PRINT("ILBCSRC DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPILBCSrc, gst_dspilbcsrc, GstPushSrc,
                      GST_TYPE_PUSH_SRC, _ilbcsrc_do_init);



#define GST_TYPE_ILBCSRC_MODE (gst_dspilbcsrc_mode_get_type())
static GType
gst_dspilbcsrc_mode_get_type (void)
{
  static GType dspilbcsrc_mode_type = 0;
  static GEnumValue dspilbcsrc_modes[] = {
    {GST_ILBCSRC_MODE_20, "20 ms frames", "20"},
    {GST_ILBCSRC_MODE_30, "30 ms frames", "30"},
    {0, NULL, NULL},
  };

  if (!dspilbcsrc_mode_type) {
    dspilbcsrc_mode_type = g_enum_register_static ("GstILBCSrcMode",
        dspilbcsrc_modes);
  }
  return dspilbcsrc_mode_type;
}

#define GST_TYPE_ILBCSRC_DTX_MODE (gst_dspilbcsrc_dtx_mode_get_type())
static GType
gst_dspilbcsrc_dtx_mode_get_type (void)
{
  static GType dspilbcsrc_dtx_mode_type = 0;
  static GEnumValue dspilbcsrc_dtx_modes[] = {
    {GST_ILBCSRC_DTX_MODE_OFF, "Don't use DTX", "off"},
    {GST_ILBCSRC_DTX_MODE_ON, "Use DTX", "on"},
    {0, NULL, NULL},
  };

  if (!dspilbcsrc_dtx_mode_type) {
    dspilbcsrc_dtx_mode_type = g_enum_register_static ("GstILBCSrcDTXMode",
        dspilbcsrc_dtx_modes);
  }
  return dspilbcsrc_dtx_mode_type;
}


/**
 * gst_dspilbcsrc_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes ILBC src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */

static void
gst_dspilbcsrc_dispose (GObject * object)
{
  GstDSPILBCSrc *dsp = (GstDSPILBCSrc *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/**
 * gst_dspilbcsrc_base_init:
 * @g_class: ILBC src class
 *
 * Does the basic initialization of ILBC src element class. This is
 * called by GStreamer framework.
 */

static void
gst_dspilbcsrc_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspilbcsrc_src_template));
  gst_element_class_set_details (gstelement_class, &gst_dspilbcsrc_details);

}


/**
 * gst_dspilbcsrc_class_init:
 * @klass: ILBC Src class object
 *
 * Initializes ILBC src element class. This is called by GStreamer
 * framework.
 */

static void
gst_dspilbcsrc_class_init (GstDSPILBCSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = gst_dspilbcsrc_set_property;
  gobject_class->get_property = gst_dspilbcsrc_get_property;
  gobject_class->dispose = gst_dspilbcsrc_dispose;

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  g_object_class_install_property (gobject_class, DSPILBC_PROP_MODE,
                                   g_param_spec_enum ("mode", "iLBC mode",
                                       "iLBC frame size in millisecs.",
                                       GST_TYPE_ILBCSRC_MODE,        /* enum type */
                                       GST_ILBCSRC_MODE_20,          /* default value */
                                       G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, DSPILBC_PROP_DTX_MODE,
                                   g_param_spec_enum ("dtx", "DTX mode",
                                       "Discontinuous transmission mode",
                                       GST_TYPE_ILBCSRC_DTX_MODE,        /* enum type */
                                       GST_ILBCSRC_DTX_MODE_ON,          /* default value */
                                       G_PARAM_READWRITE));

  gstelement_class->change_state = gst_dspilbcsrc_change_state;

  gstbasesrc_class->start = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_start);
  gstbasesrc_class->stop = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_stop);
  gstbasesrc_class->get_caps = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_getcaps);
  gstbasesrc_class->set_caps = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_setcaps);

  gstpushsrc_class->create = GST_DEBUG_FUNCPTR (gst_dspilbcsrc_create);
}


/**
 * gst_dspilbcsrc_init:
 * @dsp: DSP ILBC src object
 *
 * Initializes the given ILBC src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */

static void
gst_dspilbcsrc_init (GstDSPILBCSrc * dspilbcsrc, GstDSPILBCSrcClass *g_class)
{
  dspilbcsrc->audio = gst_dspaudio_new();
  dspilbcsrc->dtxmode = GST_ILBCSRC_DTX_MODE_ON;
  dspilbcsrc->mode = GST_ILBCSRC_MODE_20;
  dspilbcsrc->frametimemillis = 20;
}


/**
 *
 *
 */

static GstCaps *
gst_dspilbcsrc_getcaps (GstBaseSrc * src)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC(src);
  DBG_PRINT("gst_dspilbcsrc_getcaps\n");

  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  gst_caps_set_simple(caps, "mode", G_TYPE_INT, dsp->frametimemillis, NULL);
  DBG_PRINT("OUTPUT CAPS: %s\n", gst_caps_to_string(caps));
  return caps;
}


/**
 *
 *
 */

static gboolean
gst_dspilbcsrc_setcaps (GstBaseSrc * src, GstCaps *caps)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC(src);
  DBG_PRINT("gst_dspilbcsrc_getcaps\n");

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  gint mode;

  if(gst_structure_has_field(structure, "mode" )) {
    gst_structure_get_int (structure, "mode", &mode);
    DBG_PRINT("ILBC SRC GOT MODE: %d\n", mode);
    dsp->mode = (mode == 30) ? GST_ILBCSRC_MODE_30 : GST_ILBCSRC_MODE_20;
    dsp->framesize = dsp->mode == GST_ILBCSRC_MODE_20 ? 38 : 50;
    dsp->frametimemillis = dsp->mode == GST_ILBCSRC_MODE_20 ? 20 : 30;

  }
  return TRUE;
}


/**
 * gst_dspilbcsrc_create:
 * @psrc: GstPushSrc
 * @buf: GstBuffer
 *
 */

static GstFlowReturn
gst_dspilbcsrc_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC(psrc);
  guint recv_frames = 1;

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  if(dsp->dtxmode == GST_ILBCSRC_DTX_MODE_ON) {
    *buf = gst_dspaudio_read_dtx_frame(dsp->audio, dsp->framesize);
  }
  else {
    *buf = gst_dspaudio_read_normal_frame(dsp->audio, dsp->framesize, &recv_frames);
  }

  if(*buf) {
    GST_BUFFER_DURATION(*buf) = recv_frames * dsp->frametimemillis * GST_MSECOND;
    GST_BUFFER_TIMESTAMP(*buf) = dsp->framecount * dsp->frametimemillis * GST_MSECOND;
    gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
    dsp->framecount += recv_frames;
    DBG_PRINT("ILBC SRC CREATED A BUFFER\n");
    return GST_FLOW_OK;
  }
  return GST_FLOW_WRONG_STATE;
}


/**
 * gst_dspilbcsrc_start:
 * @bsrc: GstBaseSrc
 *
 */

static gboolean
gst_dspilbcsrc_start (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dspilbcsrc_start\n");
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC(bsrc);
  SPEECH_PARAMS_DATA init_data;

  if (gst_dspaudio_ed_microphone(dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize(dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  if (gst_dspaudio_open(dsp->audio, devname) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  if(dsp->dtxmode == GST_ILBCSRC_DTX_MODE_ON) {
    init_data.audio_fmt = DSP_AFMT_ILBC_DTX;
    DBG_PRINT("DTX ON\n");
  }
  else {
    init_data.audio_fmt = DSP_AFMT_ILBC;
    DBG_PRINT("DTX OFF\n");
  }
  init_data.sample_rate = SAMPLE_RATE_8KHZ;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = dsp->audio->priority;
  init_data.frame_size = dsp->mode == GST_ILBCSRC_MODE_20 ? 160 : 240; // samples
  dsp->framesize = dsp->mode == GST_ILBCSRC_MODE_20 ? 38 : 50;
  dsp->frametimemillis = dsp->mode == GST_ILBCSRC_MODE_20 ? 20 : 30;

  DBG_PRINT("SRC FRAME SIZE: %d, init_value = %d\n", dsp->framesize, init_data.frame_size);

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(SPEECH_PARAMS_DATA))) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_play(dsp->audio);
  gst_dspaudio_update_dsp_settings(dsp->audio);
  dsp->framecount = 0;
  DBG_PRINT("gst_dspilbcsrc_start OK\n");
  return TRUE;
}


/**
 *
 *
 */


static gboolean
gst_dspilbcsrc_stop (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dspilbcsrc_stop\n");
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (bsrc);

  g_mutex_lock(dsp->audio->dsp_mutex);

  if ( dsp->audio->all_frames_read ||
       dsp->audio->read_sent )
  {
    DBG_PRINT("Waiting reply for CMD_DATA_READ\n");
    gst_dspaudio_wait_buffer(dsp->audio);
    DBG_PRINT("Got it\n");
  }

  if(gst_dspaudio_stop(dsp->audio)) {
    gst_dspaudio_read_cmd(dsp->audio, 200);
  }

  gst_dspaudio_ed_microphone(dsp->audio, FALSE);
  gst_dspaudio_close(dsp->audio);
  gst_dspaudio_aep_close(dsp->audio);
  gst_dspaudio_reset(dsp->audio);
  g_mutex_unlock(dsp->audio->dsp_mutex);

  DBG_PRINT("gst_dspilbcsrc_stop OK\n");
  return TRUE;
}


/**
 * gst_dspilbcsrc_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */

static void
gst_dspilbcsrc_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (object);

  if(prop_id == DSPAUDIO_PROP_MUTE) {
    DBG_PRINT("ILBC_SRC MUTE SETTING\n");
  }


  if(prop_id == DSPILBC_PROP_MODE) {
    dsp->mode = g_value_get_enum(value);
    dsp->framesize = dsp->mode == GST_ILBCSRC_MODE_20 ? 38 : 50;
    dsp->frametimemillis = dsp->mode == GST_ILBCSRC_MODE_20 ? 20 : 30;
    DBG_PRINT("ILBC SRC: MODE = %d\n", dsp->mode);
  }
  else if(prop_id == DSPILBC_PROP_DTX_MODE) {
    dsp->dtxmode = g_value_get_enum(value);
  }
  else if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dspilbcsrc_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */

static void
gst_dspilbcsrc_get_property(GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (object);

  if(prop_id == DSPILBC_PROP_MODE) {
    g_value_set_enum (value, dsp->mode);
  }
  else if(prop_id == DSPILBC_PROP_DTX_MODE) {
    g_value_set_enum (value, dsp->dtxmode);
  }
  else if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 *
 *
 */

static GstStateChangeReturn
gst_dspilbcsrc_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPILBCSrc *dsp = GST_DSPILBCSRC (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("ILBC SRC PLAYING TO PAUSED\n");
      gst_dspaudio_interrupt_render(dsp->audio);
      break;

    default:
      break;
  }
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  return ret;
}


