/*
 * Copyright (c) 2003, 2004 Nokia
 * Author: tsavola@movial.fi
 *
 * This program is licensed under GPL (see COPYING for details)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <netinet/in.h>
#include "protocol.h"
#include "common.h"
#include "mount.h"

#if __BYTE_ORDER == __BIG_ENDIAN
# define htonll(n)  (n)
# define ntohll(n)  (n)
#elif __BYTE_ORDER == __LITTLE_ENDIAN
# define htonll(n)  ((((uint64_t) htonl(n)) << 32LL) | htonl((n) >> 32LL))
# define ntohll(n)  ((((uint64_t) ntohl(n)) << 32LL) | ntohl((n) >> 32LL))
#endif


#if 0

#include <ctype.h>

static FILE *get_log(void)
{
	static FILE *file = NULL;

	if (!file) {
		file = fopen("/tmp/read-write.log", "w");
		if (!file) {
			abort();
		}
	}

	return file;
}

static ssize_t __write__(int fd, void *buf, size_t len)
{
	size_t i;
	unsigned char c;
	FILE *file;

	file = get_log();
	fprintf(file, "write fd=%2d len=%5d buf=", fd, len);

	for (i = 0; i < len; i++) {
		c = ((char *) buf)[i];
		if (isalnum(c)) {
			fprintf(file, " %c", c);
		} else {
			fprintf(file, "%02d", (int) c);
		}
	}

	fprintf(file, "\n");
	fflush(file);

	return write(fd, buf, len);
}

static ssize_t __read__(int fd, void *buf, size_t len)
{
	FILE *file;

	file = get_log();
	fprintf(file, "read  fd=%2d len=%5d\n", fd, len);
	fflush(file);

	return read(fd, buf, len);
}

#define write __write__
#define read  __read__

#endif


ssize_t read_ni(int fd,
		void *buf,
		size_t len)
{
	ssize_t rc;

	do {
		rc = read(fd, buf, len);
	} while (rc < 0 && errno == EINTR);

	return rc;
}

ssize_t write_ni(int fd,
		 void *buf,
		 size_t len)
{
	ssize_t rc;

	do {
		rc = write(fd, buf, len);
	} while (rc < 0 && errno == EINTR);

	return rc;
}

/**
 * Writes an integer of type int16_t into a file (in network byte order).
 * The integer should be non-negative.
 * @param fd the file descriptor
 * @param i the integer
 * @return 0 on success, -1 on error
 */
int write_int16(int fd,
		int16_t i)
{
	int16_t data = htons(i);

	if (write_buf(fd, &data, sizeof (data)) < 0) {
		return -1;
	}

	return 0;
}

int write_int32(int fd,
		int32_t i)
{
	int32_t data = htonl(i);

	if (write_buf(fd, &data, sizeof (data)) < 0) {
		return -1;
	}

	return 0;
}

int write_uint64(int fd,
		 uint64_t i)
{
	uint64_t data = htonll(i);

	if (write_buf(fd, &data, sizeof (data)) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Writes a string into a file.
 * @param fd the file descriptor
 * @param str the null-terminated string
 * @return 0 on success, -1 on error
 */
int write_str(int fd,
	      const char *str)
{
	int16_t len;

	if (!str) {
		static const char *empty = "";
		str = empty;
	}

	len = strlen(str);

	/* Write the length. */
	if (write_int16(fd, len) < 0) {
		return -1;
	}

	/* Write the string. */
	if (write_buf(fd, (void *) str, len) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Writes a string vector into a file.
 * @param fd the file descriptor
 * @param strv the string vector
 * @return 0 on success, -1 on error
 */
int write_strv(int fd,
	       const char **strv)
{
	const char **s;

	if (!strv) {
		static const char *empty[] = { NULL };
		strv = empty;
	}

	/* Write the length. */
	if (write_int16(fd, calc_vec_len((void **) strv)) < 0) {
		return -1;
	}

	/* Write the strings. */
	for (s = strv; *s; s++) {
		if (write_str(fd, *s) < 0) {
			return -1;
		}
	}

	return 0;
}

int write_mount(int fd,
		const mount_info_t *mi)
{
	if (write_int16 (fd, mi->type      ) < 0 ||
	    write_str   (fd, mi->point     ) < 0 ||
	    write_str   (fd, mi->device    ) < 0 ||
	    write_str   (fd, mi->opts      ) < 0 ||
	    write_uint64(fd, mi->device_dev)) {
		return -1;
	}

	return 0;
}

int write_mountv(int fd,
		 const mount_info_t **mounts)
{
	const mount_info_t **p;

	if (!mounts) {
		static const mount_info_t *empty[] = { NULL };
		mounts = empty;
	}

	/* Write the length. */
	if (write_int16(fd, calc_vec_len((void **) mounts)) < 0) {
		return -1;
	}

	/* Write the entries. */
	for (p = mounts; *p; ++p) {
		if (write_mount(fd, *p) < 0) {
			return -1;
		}
	}

	return 0;
}

int write_buf(int fd,
	      void *buf,
	      size_t len)
{
	size_t i = 0;

	if (len == 0) {
		return 0;
	}

	while (i < len) {
		ssize_t cnt;

		cnt = write(fd, buf + i, len - i);
		if (cnt < 0) {
			if (errno == EAGAIN || errno == EINTR) {
				continue;
			}
			return -1;
		}

		i += cnt;
	}

	return 0;
}

/**
 * Reads a complete buffer from a file.
 * @param fd a file descriptor
 * @param buf pointer to a buffer large enough
 * @param len the length of the buffer
 * @return 0 on success, -1 on i/o error
 */
int read_buf(int fd,
	     void *buf,
	     size_t len)
{
	size_t i = 0;

	if (len == 0) {
		return 0;
	}

	while (i < len) {
		ssize_t cnt;

		cnt = read(fd, buf + i, len - i);
		if (cnt < 0) {
			if (errno == EAGAIN || errno == EINTR) {
				continue;
			}
			return -1;
		}

		if (cnt == 0) {
			errno = 0;
			return -1;
		}

		i += cnt;
	}

	return 0;
}

/**
 * Reads an integer of type int16_t from a file (where it is assumed to be
 * in network byte order). At EOF errno is set to 0 and -1 is returned.
 * @param fd a file descriptor
 * @return -1 on i/o error (thus negative values should not be used)
 */
int16_t read_int16(int fd)
{
	int16_t data;

	if (read_buf(fd, &data, sizeof (data)) < 0) {
		return -1;
	}

	return ntohs(data);
}

int32_t read_int32(int fd)
{
	int32_t data;

	if (read_buf(fd, &data, sizeof (data)) < 0) {
		return -1;
	}

	return ntohl(data);
}

int read_uint64(int fd,
		uint64_t *iptr)
{
	uint64_t data;

	if (read_buf(fd, &data, sizeof (data)) < 0) {
		return -1;
	}

	*iptr = ntohll(data);

	return 0;
}

/**
 * Reads a string from a file. The returned string should be free()'d.
 * @param fd a file descriptor
 * @return NULL on error
 */
char *read_str(int fd)
{
	int16_t len;
	char *str;

	/* Read the length. */
	len = read_int16(fd);
	if (len < 0) {
		return NULL;
	}

	str = calloc(len + 1, 1);
	if (!str) {
		return NULL;
	}

	/* Read the string. */
	if (read_buf(fd, str, len) < 0) {
		free(str);
		return NULL;
	}

	return str;
}

/**
 * Reads a string vector from a file. The returned strings and the vector
 * should be set free(). The vector is null-terminated.
 * @param fd a file descriptor
 * @return NULL on error
 */
char **read_strv(int fd)
{
	ssize_t strc;
	char **strv;
	size_t i;

	/* Read the length. */
	strc = read_int16(fd);
	if (strc < 0) {
		return NULL;
	}

	strv = calloc(strc + 1, sizeof (char *));
	if (!strv) {
		return NULL;
	}

	/* Read the strings. */
	for (i = 0; i < strc; i++) {
		strv[i] = read_str(fd);
		if (!strv[i]) {
			goto _err;
		}
	}

	return strv;

_err:
	free_vec((void **) strv, NULL);
	return NULL;
}

mount_info_t *read_mount(int fd)
{
	mount_info_t *mi;

	mi = mntinfo_alloc();
	if (!mi) {
		return NULL;
	}

	if ((mi->type   = read_int16(fd)) >= 0 &&
	    (mi->point  = read_str  (fd))      &&
	    (mi->device = read_str  (fd))      &&
	    (mi->opts   = read_str  (fd))      &&
	    read_uint64(fd, &mi->device_dev) >= 0) {
		return mi;
	}

	mntinfo_free(mi);
	return NULL;
}

mount_info_t **read_mountv(int fd)
{
	ssize_t cnt;
	mount_info_t **mounts;
	size_t i;

	/* Read the length. */
	cnt = read_int16(fd);
	if (cnt < 0) {
		return NULL;
	}

	mounts = calloc(cnt + 1, sizeof (mount_info_t *));
	if (!mounts) {
		return NULL;
	}

	/* Read the strings. */
	for (i = 0; i < cnt; i++) {
		mounts[i] = read_mount(fd);
		if (!mounts[i]) {
			goto _err;
		}
	}

	return mounts;

_err:
	free_vec((void **) mounts, (free_func_t *) mntinfo_free);
	return NULL;
}

static void check_str_len(const char *str,
			  size_t maxlen,
			  const char *name)
{
	if (strlen(str) >= maxlen) {
		fprintf(stderr, "Warning: %s is too long; "
			"truncating to %d characters\n", name, maxlen - 1);
	}
}

/**
 * Writes an authentication info structure into a file.
 * @param fd a file descriptor
 * @param user the username
 * @param pwd the password
 * @return 0 on success, -1 on error
 */
int write_auth(int fd,
	       const char *user,
	       const char *pwd)
{
	auth_info_t auth;

	check_str_len(user, USER_SIZE, "username");
	memset(auth.user, '\0', USER_SIZE);
	strncpy(auth.user, user, USER_SIZE - 1);

	check_str_len(pwd, PWD_SIZE, "password");
	memset(auth.pwd, '\0', PWD_SIZE);
	strncpy(auth.pwd, pwd, PWD_SIZE - 1);

	if (write_buf(fd, &auth.user, USER_SIZE) < 0 ||
	    write_buf(fd, &auth.pwd,  PWD_SIZE ) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Reads an authentication info structure from a file.
 * @param fd a file descriptor
 * @param auth pointer to some memory...
 * @return -1 on error
 */
int read_auth(int fd,
	      auth_info_t *auth)
{
	if (read_buf(fd, &auth->user, USER_SIZE) < 0 ||
	    read_buf(fd, &auth->pwd,  PWD_SIZE ) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Writes a command info structure into a file.
 */
int write_cmd(int fd,
	      int16_t umount,
	      const char *target,
	      const mount_info_t **mounts,
	      const char **argv,
	      const char *curdir,
	      const char **env,
	      int16_t is_tty,
	      const struct winsize *ws)
{
	if (write_int16(fd, umount       ) < 0 ||
	    write_str  (fd, target       ) < 0 ||
	    write_mountv(fd, mounts      ) < 0 ||
	    write_strv (fd, argv         ) < 0 ||
	    write_str  (fd, curdir       ) < 0 ||
	    write_strv (fd, env          ) < 0 ||
	    write_int16(fd, is_tty       ) < 0 ||
	    write_int16(fd, ws->ws_row   ) < 0 ||
	    write_int16(fd, ws->ws_col   ) < 0 ||
	    write_int16(fd, ws->ws_xpixel) < 0 ||
	    write_int16(fd, ws->ws_ypixel) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Reads a command info structure from a file. The CMD structure
 * should be freed with the free_cmd() function.
 * @param fd a file descriptor
 * @param cmd pointer to some memory...
 * @return -1 on error
 */
int read_cmd(int fd,
	     cmd_info_t *cmd)
{
	if ((cmd->umount     = read_int16 (fd)) >= 0 &&
	    (cmd->target     = read_str   (fd))      &&
	    (cmd->mounts     = read_mountv(fd))      &&
	    (cmd->argv       = read_strv  (fd))      &&
	    (cmd->curdir     = read_str   (fd))      &&
	    (cmd->env        = read_strv  (fd))      &&
	    (cmd->is_tty     = read_int16 (fd)) >= 0 &&
	    (cmd->tty_row    = read_int16 (fd)) >= 0 &&
	    (cmd->tty_col    = read_int16 (fd)) >= 0 &&
	    (cmd->tty_xpixel = read_int16 (fd)) >= 0 &&
	    (cmd->tty_ypixel = read_int16 (fd)) >= 0) {
		return 0;
	}

	free_cmd(cmd);
	return -1;
}

/**
 * Frees a command info structure including its contents.
 */
void free_cmd(cmd_info_t *cmd)
{
	if (cmd->target) {
		free(cmd->target);
	}

	if (cmd->mounts) {
		free_vec((void **) cmd->mounts, (free_func_t *) mntinfo_free);
	}

	if (cmd->argv) {
		free_vec((void **) cmd->argv, NULL);
	}

	if (cmd->curdir) {
		free(cmd->curdir);
	}

	if (cmd->env) {
		free_vec((void **) cmd->env, NULL);
	}
}

/**
 * Writes a packet header into a file.
 * @param fd a file descriptor
 * @param type of the packet (one of the PTYPE-values)
 * @param size the size of buf
 * @return 0 on success, -1 on i/o error
 */
int write_phead(int fd,
		int16_t type,
		int32_t size)
{
	if (write_int16(fd, type) < 0 ||
	    write_int32(fd, size) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Writes a packet into a file.
 * @param fd a file descriptor
 * @param type of the packet (one of the PTYPE-values)
 * @param buf the contents of the packet
 * @param size the size of buf
 * @return 0 on success, -1 on i/o error
 */
int write_packet(int fd,
		 int16_t type,
		 void *buf,
		 int32_t size)
{
	if (write_phead(fd, type, size) < 0) {
		return -1;
	}

	if (size > 0 &&
	    write_buf(fd, buf, size) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Reads the header of a packet from a file so that it can be identified.
 * At EOF errno is set to 0 and -1 is returned.
 * @param fd a file descriptor
 * @param head buffer for the packet
 * @return 0 on success, -1 on i/o error
 */
int read_phead(int fd,
	       phead_t *head)
{
	head->type = read_int16(fd);
	if (head->type < 0) {
		return -1;
	}

	head->size = read_int32(fd);
	if (head->size < 0) {
		return -1;
	}

	return 0;
}

/**
 * Writes the VERSION packet to a file.
 * @param fd a file descriptor
 * @return 0 on success, -1 on i/o error
 */
int send_version(int fd)
{
	if (write_phead(fd, PTYPE_VERSION, sizeof (int16_t)) < 0 ||
	    write_int16(fd, PROTOCOL_VERSION) < 0) {
		return -1;
	}

	return 0;
}

/**
 * Reads the VERSION packet from a file.
 * @param fd a file descriptor
 * @return >0 on success, -1 on i/o error
 */
int get_version(int fd)
{
	phead_t head;

	if (read_phead(fd, &head) < 0) {
		return -1;
	}

	if (head.type != PTYPE_VERSION ||
	    head.size != sizeof (int16_t)) {
		errno = 0;
		return -1;
	}

	return read_int16(fd);
}
