/**
 * @file libgalago/galago-value.c Value wrapper API
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-value.h>
#include <libgalago/galago-assert.h>
#include <string.h>

/**
 * A wrapper for a type of values.
 */
struct _GalagoValue
{
	GalagoType type;
	GalagoType subtype;

	union
	{
		void *ptr;
		GType gtype;

	} detail;

	union
	{
		char char_data;
		unsigned char uchar_data;
		gboolean boolean_data;
		short short_data;
		unsigned short ushort_data;
		int int_data;
		unsigned int uint_data;
		long long_data;
		unsigned long ulong_data;
		char *string_data;
		void *object_data;
		void *pointer_data;
		GList *list_data;

		struct
		{
			const void *array;
			gsize size;

		} array_data;

	} data;
};

#define DEFINE_ACCESSOR(name, type, defaultval) \
void \
galago_value_set_##name(GalagoValue *value, type data) \
{ \
	g_return_if_fail(value != NULL); \
	g_return_if_fail(galago_value_get_type(value) != GALAGO_VALUE_TYPE_LIST); \
\
	value->data.name##_data = data; \
} \
\
type \
galago_value_get_##name(const GalagoValue *value) \
{ \
	g_return_val_if_fail(value != NULL, defaultval); \
	g_return_val_if_fail(galago_value_get_type(value) != GALAGO_VALUE_TYPE_LIST,\
							  defaultval); \
\
	return value->data.name##_data; \
}

#define DEFINE_ACCESSOR_MEM(name, type, defaultval, destroy, cpy) \
void \
galago_value_set_##name(GalagoValue *value, type data) \
{ \
	g_return_if_fail(value != NULL); \
	g_return_if_fail(galago_value_get_type(value) != GALAGO_VALUE_TYPE_LIST); \
\
	if (value->data.name##_data != NULL) \
		(destroy)(value->data.name##_data); \
\
	value->data.name##_data = (data == NULL ? NULL : (cpy)(data)); \
} \
\
type \
galago_value_get_##name(const GalagoValue *value) \
{ \
	g_return_val_if_fail(value != NULL, defaultval); \
	g_return_val_if_fail(galago_value_get_type(value) != GALAGO_VALUE_TYPE_LIST,\
							  defaultval); \
\
	return value->data.name##_data; \
}

GalagoValue *
galago_value_new(GalagoType type, const void *data, void *detail)
{
	GalagoValue *value;

	g_return_val_if_fail(type != GALAGO_VALUE_TYPE_UNKNOWN, NULL);
	g_return_val_if_fail(type != GALAGO_VALUE_TYPE_OBJECT,  NULL);

	value = g_new0(GalagoValue, 1);
	value->type       = type;
	value->detail.ptr = detail;

	if (data != NULL)
	{
#define CHECK_SET_TYPE(type, name, datatype) \
	case type: \
		galago_value_set_##name(value, *(datatype *)data); \
		break;

		switch (type)
		{
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_CHAR,    char,    char);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_UCHAR,   uchar,   unsigned char);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_BOOLEAN, boolean, gboolean);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_SHORT,   short,   short);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_USHORT,  ushort,  unsigned short);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_INT,     int,     int);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_UINT,    uint,    unsigned int);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_LONG,    long,    long);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_ULONG,   ulong,   unsigned long);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_STRING,  string,  const char *);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_OBJECT,  object,  void *);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_POINTER, pointer, void *);
			CHECK_SET_TYPE(GALAGO_VALUE_TYPE_LIST,    list,    GList *);

			default:
				galago_value_destroy(value);
				return NULL;
		}
	}

	return value;
}

GalagoValue *
galago_value_new_object(GType type, const GObject *obj)
{
	GalagoValue *value;

	value = g_new0(GalagoValue, 1);
	value->type   = GALAGO_VALUE_TYPE_OBJECT;
	value->detail.gtype = type;

	if (obj != NULL)
		galago_value_set_object(value, (void *)obj);

	return value;
}

GalagoValue *
galago_value_new_list(GalagoType type, GList *list, void *detail)
{
	GalagoValue *value;

	g_return_val_if_fail(type != GALAGO_VALUE_TYPE_UNKNOWN, NULL);

	value = galago_value_new(GALAGO_VALUE_TYPE_LIST, list, detail);
	value->subtype = type;

	return value;
}

GalagoValue *
galago_value_new_array(GalagoType type, const void *array, gsize size,
					   void *detail)
{
	GalagoValue *value;

	g_return_val_if_fail(type != GALAGO_VALUE_TYPE_UNKNOWN, NULL);

	value = galago_value_new(GALAGO_VALUE_TYPE_ARRAY, NULL, detail);
	value->subtype = type;

	galago_value_set_array(value, array, size);

	return value;
}

void
galago_value_destroy(GalagoValue *value)
{
	GalagoType type;

	g_return_if_fail(value != NULL);

	type = galago_value_get_type(value);

	if (type == GALAGO_VALUE_TYPE_LIST && value->data.list_data != NULL)
	{
		g_list_foreach(value->data.list_data, (GFunc)galago_value_destroy,
					   NULL);
		g_list_free(value->data.list_data);
	}
	else if (type == GALAGO_VALUE_TYPE_STRING)
	{
		g_free(value->data.string_data);
	}

	g_free(value);
}

GalagoType
galago_value_get_type(const GalagoValue *value)
{
	g_return_val_if_fail(value != NULL, GALAGO_VALUE_TYPE_UNKNOWN);

	return value->type;
}

GalagoType
galago_value_get_subtype(const GalagoValue *value)
{
	g_return_val_if_fail(value != NULL, GALAGO_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail(
		galago_value_get_type(value) == GALAGO_VALUE_TYPE_LIST ||
		galago_value_get_type(value) == GALAGO_VALUE_TYPE_ARRAY,
		GALAGO_VALUE_TYPE_UNKNOWN);

	return value->subtype;
}

GType
galago_value_get_gtype(const GalagoValue *value)
{
	g_return_val_if_fail(value != NULL, 0);
	g_return_val_if_fail(galago_value_get_type(value) ==
						 GALAGO_VALUE_TYPE_OBJECT ||
						 galago_value_get_subtype(value) ==
						 GALAGO_VALUE_TYPE_OBJECT,
						 0);

	return value->detail.gtype;
}

void
galago_value_set_list(GalagoValue *value, GList *list)
{
	GList *l;
	GList *new_list = NULL;
	GalagoType subtype;

	g_return_if_fail(value != NULL);
	g_return_if_fail(galago_value_get_type(value) == GALAGO_VALUE_TYPE_LIST);

	if (value->data.list_data != NULL)
	{
		g_list_foreach(value->data.list_data, (GFunc)galago_value_destroy,
					   NULL);
		g_list_free(value->data.list_data);
	}

	subtype = galago_value_get_subtype(value);

	for (l = list; l != NULL; l = l->next)
	{
		new_list = g_list_append(new_list,
								 galago_value_new(subtype, &l->data,
												  value->detail.ptr));
	}

	value->data.list_data = new_list;
}

GList *
galago_value_get_list(const GalagoValue *value)
{
	g_return_val_if_fail(value != NULL, NULL);
	g_return_val_if_fail(galago_value_get_type(value) == GALAGO_VALUE_TYPE_LIST,
							  NULL);

	return value->data.list_data;
}

void
galago_value_set_array(GalagoValue *value, const void *data, gsize size)
{
	g_return_if_fail(value != NULL);
	g_return_if_fail(galago_value_get_type(value) == GALAGO_VALUE_TYPE_ARRAY);

	value->data.array_data.size  = size;
	value->data.array_data.array = data;
}

void
galago_value_get_array(const GalagoValue *value, const void **ret_array,
					   gsize *ret_size)
{
	g_return_if_fail(value != NULL);
	g_return_if_fail(galago_value_get_type(value) == GALAGO_VALUE_TYPE_ARRAY);

	if (ret_array != NULL)
		*ret_array = value->data.array_data.array;

	if (ret_size != NULL)
		*ret_size = value->data.array_data.size;
}

DEFINE_ACCESSOR(char,       char,           '\0')
DEFINE_ACCESSOR(uchar,      unsigned char,  '\0')
DEFINE_ACCESSOR(boolean,    gboolean,    FALSE)
DEFINE_ACCESSOR(short,      short,          0)
DEFINE_ACCESSOR(ushort,     unsigned short, 0)
DEFINE_ACCESSOR(int,        int,            0)
DEFINE_ACCESSOR(uint,       unsigned int,   0)
DEFINE_ACCESSOR(long,       long,           0)
DEFINE_ACCESSOR(ulong,      unsigned long,  0)
DEFINE_ACCESSOR(object,     void *,         NULL)
DEFINE_ACCESSOR(pointer,    void *,         NULL)
DEFINE_ACCESSOR_MEM(string, const char *,   NULL, g_free, g_strdup)
