/**
 * @file libgalago/galago-core.c Galago Context API
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <libgalago/galago-private.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-marshal.h>
#include <string.h>

#ifdef HAVE_SIGNAL_H
# include <signal.h>
#endif

#ifdef __G_LIB_H__
# include <dbus/dbus-glib-lowlevel.h>
#endif

struct _GalagoCorePrivate
{
	/*
	 * We don't want to risk a caller reffing the core, so we keep a
	 * separate init ref count here.
	 */
	unsigned int init_ref_count;

	char *app_name;
	char *conn_obj_path;
	char *uid;

	DBusConnection *dbus_conn;

	gboolean filters_added;
	gboolean watch_all;

	GalagoInitFlags flags;

	gboolean daemon;
	gboolean registered;

	gboolean daemon_active;

	gboolean registering_connection;

	guint block_signals;
};

enum
{
	REGISTERED,
	UNREGISTERED,
	SERVICE_ADDED,
	SERVICE_REMOVED,
	PERSON_ADDED,
	PERSON_REMOVED,
	CALC_PRIORITY_ACCOUNT,
	LAST_SIGNAL
};

#define GALAGO_DAEMON_ID "#galago-daemon#"

/* galago-person.c */
GalagoAccount *_galago_person_default_calc_priority_account(
	const GalagoPerson *person);

static void _galago_core_disconnect(void);
static gboolean _galago_dbus_register_connection(void);
static void _galago_dbus_unregister_connection(void);
static void _galago_dbus_unregister_connection_finish(void);
static void _galago_dbus_core_add_service(GalagoService *service);
static void _galago_dbus_core_remove_service(GalagoService *service);
static void _galago_dbus_core_add_person(GalagoPerson *person);
static void _galago_dbus_core_remove_person(GalagoPerson *person);

G_LOCK_DEFINE_STATIC(_core_lock);
static GalagoCore *_core = NULL;

static GalagoContextOps context_ops =
{
	_galago_dbus_core_add_service,
	_galago_dbus_core_remove_service,
	_galago_dbus_core_add_person,
	_galago_dbus_core_remove_person
};

/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galago_core_destroy(GalagoObject *gobject);

static GalagoObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

#define GALAGO_TYPE_CORE (galago_core_get_type())

G_DEFINE_TYPE(GalagoCore, galago_core, GALAGO_TYPE_OBJECT);

static gboolean
_galago_accumulator_account_handled(GSignalInvocationHint *ihint,
									GValue *return_accu,
									const GValue *handler_return,
									gpointer unused)
{
	gboolean continue_emission = FALSE;
	GalagoAccount *account;

	account = g_value_get_object(handler_return);

	if (account == NULL || !GALAGO_IS_ACCOUNT(account))
	{
		account = NULL;
		continue_emission = TRUE;
	}

	g_value_set_object(return_accu, account);

	return continue_emission;
}

static void
galago_core_class_init(GalagoCoreClass *klass)
{
	GalagoObjectClass *object_class = GALAGO_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->dbus_interface = GALAGO_DBUS_CORE_INTERFACE;
	object_class->destroy = galago_core_destroy;

	klass->calc_priority_account = NULL;

	signals[REGISTERED] =
		g_signal_new("registered",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoCoreClass, registered),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	signals[UNREGISTERED] =
		g_signal_new("unregistered",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoCoreClass, unregistered),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	signals[SERVICE_ADDED] =
		g_signal_new("service_added",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoCoreClass, service_added),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__OBJECT,
					 G_TYPE_NONE, 1,
					 GALAGO_TYPE_SERVICE);

	signals[SERVICE_REMOVED] =
		g_signal_new("service_removed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoCoreClass, service_removed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__OBJECT,
					 G_TYPE_NONE, 1,
					 GALAGO_TYPE_SERVICE);

	signals[PERSON_ADDED] =
		g_signal_new("person_added",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoCoreClass, person_added),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__OBJECT,
					 G_TYPE_NONE, 1,
					 GALAGO_TYPE_SERVICE);

	signals[PERSON_REMOVED] =
		g_signal_new("person_removed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoCoreClass, person_removed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__OBJECT,
					 G_TYPE_NONE, 1,
					 GALAGO_TYPE_SERVICE);

	signals[CALC_PRIORITY_ACCOUNT] =
		g_signal_new("calc_priority_account",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST,
					 G_STRUCT_OFFSET(GalagoCoreClass, calc_priority_account),
					 _galago_accumulator_account_handled,
					 NULL,
					 galago_marshal_OBJECT__OBJECT,
					 GALAGO_TYPE_ACCOUNT, 1,
					 GALAGO_TYPE_PERSON);
}

static void
galago_core_init(GalagoCore *core)
{
	core->priv = g_new0(GalagoCorePrivate, 1);
}

static void
galago_core_destroy(GalagoObject *object)
{
	GalagoCore *core = (GalagoCore *)object;

	if (core->priv != NULL)
	{
		GalagoContext *context;

		_galago_core_disconnect();

		/*
		 * Prevent this object from being removed from the soon-to-be
		 * destroyed context after this function ends.
		 */
		galago_object_set_dbus_path(GALAGO_OBJECT(object), NULL);

		context = galago_context_get();
		galago_context_pop();
		g_object_unref(context);

		g_free(core->priv->app_name);

		g_free(core->priv);
		core->priv = NULL;
	}

	G_LOCK(_core_lock);
	_core = NULL;
	G_UNLOCK(_core_lock);

	parent_class->destroy(object);
}

/**************************************************************************
 * D-BUS Processing
 **************************************************************************/
static void
reregister_client(void)
{
	GList *l;

	if (!_galago_dbus_register_connection())
	{
		g_warning("Unable to re-establish registration");

		return;
	}

	if (!galago_is_feed())
		return;

	for (l = galago_get_people(GALAGO_LOCAL, FALSE); l != NULL; l = l->next)
	{
		GalagoPerson *person = GALAGO_PERSON(l->data);

		_galago_dbus_core_add_person(person);
		galago_dbus_object_push_full(GALAGO_OBJECT(person));
	}

	for (l = galago_get_services(GALAGO_LOCAL, FALSE); l != NULL; l = l->next)
	{
		GalagoService *service = GALAGO_SERVICE(l->data);

		_galago_dbus_core_add_service(service);
		galago_dbus_object_push_full(GALAGO_OBJECT(service));
	}
}

static DBusHandlerResult
_handle_dbus_error(DBusConnection *dbus_conn, DBusMessage *message,
				   void *user_data)
{
	const char *error_name = dbus_message_get_error_name(message);

	if (dbus_message_is_error(message, DBUS_ERROR_NO_REPLY))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	if (dbus_message_is_error(message, DBUS_ERROR_SERVICE_UNKNOWN))
	{
		if (!galago_is_daemon())
		{
			_galago_dbus_unregister_connection_finish();
		}

		if (!(_core->priv->flags & GALAGO_INIT_NO_ACTIVATION))
		{
			DBusError error;

			dbus_error_init(&error);

			if (!dbus_bus_start_service_by_name(_core->priv->dbus_conn,
												GALAGO_DBUS_SERVICE,
												0, NULL, &error))
			{
				g_warning("Received ServiceDoesNotExist, and cannot "
						  "re-activate daemon. Disconnecting for now.");
				dbus_error_free(&error);
			}
		}

		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	else if (dbus_message_is_error(message, DBUS_ERROR_LIMITS_EXCEEDED))
	{
		const char *str;
		DBusMessageIter iter;

		dbus_message_iter_init(message, &iter);
		dbus_message_iter_get_basic(&iter, &str);

		g_critical("Received LimitsExceeded: %s", str);

		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	g_warning("D-BUS Error: %s", error_name);

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gboolean
_dbus_fdo_filter_func(DBusConnection *dbus_conn, DBusMessage *message,
					  void *user_data)
{
	if (dbus_message_is_signal(message, DBUS_INTERFACE_DBUS,
							   "NameOwnerChanged"))
	{
		DBusError error;
		const char *dbus_service;
		const char *old_owner;
		const char *new_owner;

		dbus_error_init(&error);

		if (dbus_message_get_args(message, &error,
								  DBUS_TYPE_STRING, &dbus_service,
								  DBUS_TYPE_STRING, &old_owner,
								  DBUS_TYPE_STRING, &new_owner,
								  DBUS_TYPE_INVALID) &&
			!strcmp(dbus_service, GALAGO_DBUS_SERVICE) &&
			!galago_is_daemon())
		{
			gboolean old_owner_good = (old_owner && *old_owner != '\0');
			gboolean new_owner_good = (new_owner && *new_owner != '\0');

			if (old_owner_good)
			{
				_galago_dbus_unregister_connection_finish();
			}

			if (new_owner_good)
			{
				reregister_client();
			}
		}
	}
	else
	{
		return FALSE;
	}

	return TRUE;
}

static DBusHandlerResult
filter_func(DBusConnection *dbus_conn, DBusMessage *message, void *user_data)
{
	DBusMessageIter iter;
	const char *path;

	if (dbus_message_get_type(message) == DBUS_MESSAGE_TYPE_ERROR)
		return _handle_dbus_error(dbus_conn, message, user_data);

	if (_dbus_fdo_filter_func(dbus_conn, message, user_data))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	path = dbus_message_get_path(message);

	if (path == NULL ||
		galago_context_get_object(path) == NULL)
	{
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	dbus_message_iter_init(message, &iter);

#define MATCH_ATTRIBUTABLE_OBJECTS(signalname) \
	(dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE, \
							(signalname)) || \
	 dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE, \
							(signalname)))

	/*
	 * Core
	 */
	if (dbus_message_is_signal(message, GALAGO_DBUS_CORE_INTERFACE,
							   "ServiceAdded"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_SERVICE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_CORE_INTERFACE,
									"ServiceRemoved"))
	{
		GalagoService *service;

		service = galago_dbus_message_iter_get_object(&iter,
													  GALAGO_TYPE_SERVICE);

		g_object_unref(service);
	}
	/*
	 * Attributes
	 *
	 * This applies to Person and Account objects.
	 */
	else if (MATCH_ATTRIBUTABLE_OBJECTS("AttributeSet"))
	{
		GalagoObject *object = galago_context_get_object(path);

		if (object != NULL)
		{
			DBusMessageIter value_iter;
			const char *name;
			int arg_type;

			dbus_message_iter_get_basic(&iter, &name);
			dbus_message_iter_next(&iter);

			dbus_message_iter_recurse(&iter, &value_iter);
			arg_type = dbus_message_iter_get_arg_type(&iter);

			switch (arg_type)
			{
				case DBUS_TYPE_STRING:
				{
					const char *value;
					dbus_message_iter_get_basic(&value_iter, &value);
					galago_object_set_attr_string(object, name, value);
					break;
				}

				case DBUS_TYPE_BOOLEAN:
				{
					gboolean value;
					dbus_message_iter_get_basic(&value_iter, &value);
					galago_object_set_attr_bool(object, name, value);
					break;
				}

				case DBUS_TYPE_UINT32:
				{
					dbus_uint32_t value;
					dbus_message_iter_get_basic(&value_iter, &value);
					galago_object_set_attr_int(object, name, value);
					break;
				}

				default:
					g_warning("Invalid property type %d received from "
							  "AttributeSet", arg_type);
					break;
			}
		}
	}
	else if (MATCH_ATTRIBUTABLE_OBJECTS("AttributeRemoved"))
	{
		GalagoObject *object = galago_context_get_object(path);

		if (object != NULL)
		{
			const char *name;
			dbus_message_iter_get_basic(&iter, &name);
			galago_object_remove_attribute(object, name);
		}
	}
	/*
	 * Service
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_SERVICE_INTERFACE,
									"AccountAdded"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_ACCOUNT);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_SERVICE_INTERFACE,
									"AccountRemoved"))
	{
		GalagoAccount *account;

		account = galago_dbus_message_iter_get_object(&iter,
													  GALAGO_TYPE_ACCOUNT);

		g_object_unref(account);
	}
	/*
	 * Account
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"PresenceCreated"))
	{
		GalagoAccount *account;
		GalagoPresence *presence;

		account = GALAGO_ACCOUNT(galago_context_get_object(path));
		presence = galago_dbus_message_iter_get_object(&iter,
													   GALAGO_TYPE_PRESENCE);
		g_assert(galago_account_get_presence(account, FALSE) == presence);

		_galago_account_presence_created(account, presence);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"PresenceDeleted"))
	{
		GalagoAccount *account;

		account = GALAGO_ACCOUNT(galago_context_get_object(path));

		if (account != NULL)
			_galago_account_presence_deleted(account);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"Connected"))
	{
		GalagoAccount *account;

		account = GALAGO_ACCOUNT(galago_context_get_object(path));

		if (account != NULL)
			galago_account_set_connected(account, TRUE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"Disconnected"))
	{
		GalagoAccount *account;

		account = GALAGO_ACCOUNT(galago_context_get_object(path));

		if (account != NULL)
			galago_account_set_connected(account, FALSE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"DisplayNameChanged"))
	{
		GalagoAccount *account;
		const char *display_name;

		account = GALAGO_ACCOUNT(galago_context_get_object(path));

		if (account != NULL)
		{
			display_name = galago_dbus_message_iter_get_string_or_nil(&iter);
			galago_account_set_display_name(account, display_name);
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"AvatarSet"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_IMAGE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_ACCOUNT_INTERFACE,
									"AvatarUnset"))
	{
		GalagoAccount *account;

		account = GALAGO_ACCOUNT(galago_context_get_object(path));

		if (account != NULL)
			galago_account_set_avatar(account, NULL);
	}
	/*
	 * Presence
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PRESENCE_INTERFACE,
									"IdleUpdated"))
	{
		GalagoPresence *presence;

		presence = GALAGO_PRESENCE(galago_context_get_object(path));

		if (presence != NULL)
		{
			gboolean idle;
			dbus_uint32_t idle_start_time;

			dbus_message_iter_get_basic(&iter, &idle);
			dbus_message_iter_next(&iter);
			dbus_message_iter_get_basic(&iter, &idle_start_time);

			galago_presence_set_idle(presence, idle, idle_start_time);
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PRESENCE_INTERFACE,
									"StatusAdded"))
	{
		GalagoPresence *presence;
		GalagoStatus *status;

		presence = GALAGO_PRESENCE(galago_context_get_object(path));

		if (presence != NULL)
		{
			const char *status_id;

			status = galago_dbus_message_iter_get_object(&iter,
														 GALAGO_TYPE_STATUS);

			status_id = galago_status_get_id(status);

			if (!galago_presence_has_status(presence, status_id))
				galago_presence_add_status(presence, status);
			else
				g_object_unref(status);
		}
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PRESENCE_INTERFACE,
									"StatusRemoved"))
	{
		GalagoPresence *presence;
		const char *status_id;

		presence = GALAGO_PRESENCE(galago_context_get_object(path));

		if (presence != NULL)
		{
			dbus_message_iter_get_basic(&iter, &status_id);

			galago_presence_remove_status(presence, status_id);
		}
	}
	/*
	 * Person
	 */
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE,
									"PhotoSet"))
	{
		galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_IMAGE);
	}
	else if (dbus_message_is_signal(message, GALAGO_DBUS_PERSON_INTERFACE,
									"PhotoUnset"))
	{
		GalagoPerson *person;

		person = GALAGO_PERSON(galago_context_get_object(path));

		if (person != NULL)
			galago_person_set_photo(person, NULL);
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/**************************************************************************
 * Core API
 **************************************************************************/
static gboolean
_galago_core_connect(void)
{
	DBusError error;

	dbus_error_init(&error);

	_core->priv->dbus_conn = dbus_bus_get(DBUS_BUS_SESSION, &error);

	if (_core->priv->dbus_conn == NULL)
	{
		g_error("Unable to connect to session bus: %s", error.message);
		dbus_error_free(&error);

		return FALSE;
	}

	dbus_connection_setup_with_g_main(_core->priv->dbus_conn, NULL);
	dbus_connection_set_exit_on_disconnect(_core->priv->dbus_conn, FALSE);

	if (_core->priv->daemon)
	{
		dbus_error_free(&error);

		return TRUE;
	}

	if (!(_core->priv->flags & GALAGO_INIT_NO_ACTIVATION))
	{
		if (!dbus_bus_start_service_by_name(_core->priv->dbus_conn,
											GALAGO_DBUS_SERVICE,
											0, NULL, &error))
		{
			g_warning("Unable to activate Galago service: %s", error.message);
			dbus_error_free(&error);
			dbus_error_init(&error);
		}
	}

	if (!dbus_connection_add_filter(_core->priv->dbus_conn,
									filter_func, NULL, NULL))
	{
		g_error("Unable to create core D-BUS handler");
		dbus_error_free(&error);

		return FALSE;
	}

	dbus_bus_add_match(_core->priv->dbus_conn,
					   "type='signal',"
					   "interface='" DBUS_INTERFACE_DBUS "',"
					   "sender='" DBUS_SERVICE_DBUS "'",
					   &error);

	if (dbus_error_is_set(&error))
	{
		g_error("Unable to subscribe to signals: %s", error.message);

		dbus_error_free(&error);

		return FALSE;
	}

	dbus_error_free(&error);

	_core->priv->filters_added = TRUE;

	if (dbus_bus_name_has_owner(_core->priv->dbus_conn,
								GALAGO_DBUS_SERVICE, NULL))
	{
		_galago_dbus_register_connection();
	}

	return TRUE;
}

static void
_galago_core_disconnect(void)
{
	if (_core->priv->dbus_conn != NULL)
	{
		if (!_core->priv->daemon && galago_is_registered())
		{
			_galago_dbus_unregister_connection();
		}

		if (_core->priv->dbus_conn != NULL)
		{
			if (_core->priv->filters_added)
			{
				dbus_connection_remove_filter(_core->priv->dbus_conn,
											  filter_func, NULL);
			}

			dbus_connection_dispatch(_core->priv->dbus_conn);

			/*
			 * NOTE: There seems to be a big problem when uninitting and
			 *       then initting Galago. It has to do with GalagoService,
			 *       I think, but it's proving extremely difficult to track
			 *       that down. For now, we're just going to prevent
			 *       closing and unreffing, as this seems to fix that
			 *       particular issue.
			 *
			 *       See bug #36.
			 */
#if 0
			dbus_connection_close(_core->priv->dbus_conn);
			dbus_connection_unref(_core->priv->dbus_conn);
#endif
			_core->priv->dbus_conn = NULL;
		}
	}

	_core->priv->filters_added = FALSE;
	_core->priv->registered    = FALSE;
}

GalagoAccount *
_galago_core_calc_priority_account(const GalagoPerson *person)
{
	GalagoAccount *account = NULL;

	g_signal_emit(_core, signals[CALC_PRIORITY_ACCOUNT], 0, person, &account);

	if (account == NULL)
		account = _galago_person_default_calc_priority_account(person);

	return account;
}

static void
_exit_galago(void)
{
	/* Forcefully uninit galago. */
	while (_core != NULL)
		galago_uninit();
}

#ifdef HAVE_SIGNAL_H
static void
_sigint_cb(int unused)
{
	signal(SIGINT, SIG_DFL);

	_exit_galago();

	raise(SIGINT);
}
#endif

static gpointer
register_exit_handlers(gpointer unused)
{
	g_atexit(_exit_galago);

#ifdef HAVE_SIGNAL_H
	if (signal(SIGINT, _sigint_cb) == SIG_ERR)
	{
		g_warning("Unable to register SIGINT signal handler");
	}
#endif /* HAVE_SIGNAL_H_ */

	return NULL;
}

gboolean
galago_init(const char *name, GalagoInitFlags flags)
{
	GalagoContext *context;
	static GOnce first_init = G_ONCE_INIT;

	g_return_val_if_fail(name  != NULL, FALSE);
	g_return_val_if_fail(*name != '\0', FALSE);

	if (_core != NULL)
	{
		_core->priv->init_ref_count++;

		return TRUE;
	}

	g_type_init();

	context = galago_context_new();
	galago_context_set_ops(context, &context_ops);

	galago_context_push(context);

	G_LOCK(_core_lock);
	_core = g_object_new(GALAGO_TYPE_CORE, NULL);
	_core->priv->init_ref_count = 1;
	G_UNLOCK(_core_lock);

	galago_object_set_dbus_path(GALAGO_OBJECT(_core), GALAGO_DBUS_CORE_OBJECT);
	_core->priv->app_name = g_strdup(name);
	_core->priv->flags = flags;

	if (!strcmp(name, GALAGO_DAEMON_ID))
		_core->priv->daemon = TRUE;

	if (!_galago_core_connect())
		_galago_core_disconnect();

	g_once(&first_init, register_exit_handlers, NULL);

	return TRUE;
}

void
galago_uninit(void)
{
	if (!galago_is_initted())
		return;

	_core->priv->init_ref_count--;

	if (_core->priv->init_ref_count == 0)
		galago_object_destroy(GALAGO_OBJECT(_core));
}

gboolean
galago_is_initted(void)
{
	return (_core != NULL);
}

gboolean
galago_is_connected(void)
{
	return (galago_is_initted() && _core->priv->dbus_conn != NULL &&
			(_core->priv->registering_connection || _core->priv->uid != NULL ||
			 galago_is_daemon()) &&
			dbus_connection_get_is_connected(_core->priv->dbus_conn));
}

gboolean
galago_is_registered(void)
{
	return _core->priv->registered;
}

gboolean
galago_is_daemon_active(void)
{
	return (galago_is_daemon() || _core->priv->daemon_active);
}

gboolean
galago_is_daemon(void)
{
	return (_core->priv->daemon);
}

DBusConnection *
galago_get_dbus_conn(void)
{
	g_return_val_if_fail(galago_is_initted(), NULL);

	return _core->priv->dbus_conn;
}

const char *
galago_get_uid(void)
{
	g_return_val_if_fail(galago_is_initted(),   NULL);
	g_return_val_if_fail(galago_is_connected(), NULL);

	return _core->priv->uid;
}

const char *
galago_get_client_obj_path(void)
{
	g_return_val_if_fail(galago_is_initted(),   NULL);
	g_return_val_if_fail(galago_is_connected(), NULL);

	return _core->priv->conn_obj_path;
}

gboolean
galago_is_feed(void)
{
	g_return_val_if_fail(galago_is_initted(),   FALSE);
	g_return_val_if_fail(galago_is_connected(), FALSE);

	return (_core->priv->flags & GALAGO_INIT_FEED);
}

#define ADD_SIGNAL_MATCH(iface) \
	dbus_bus_add_match(dbus_conn, \
					   "type='signal'," \
					   "interface='" iface "'," \
					   "sender='" GALAGO_DBUS_SERVICE "'", \
					   &error); \
	if (dbus_error_is_set(&error)) \
	{ \
		g_error("Unable to subscribe to %s signal: %s", \
				(iface), error.message); \
		goto exit; \
	}
#define REMOVE_SIGNAL_MATCH(iface) \
	dbus_bus_remove_match(dbus_conn, \
					   "type='signal'," \
					   "interface='" iface "'," \
					   "sender='" GALAGO_DBUS_SERVICE "'", \
					   &error); \
	if (dbus_error_is_set(&error)) \
	{ \
		g_error("Unable to unsubscribe from %s signal: %s", \
				(iface), error.message); \
		goto exit; \
	}

void
galago_set_watch_all(gboolean watch_all)
{
	DBusError error;
	DBusConnection *dbus_conn;

	g_return_if_fail(galago_is_initted());

	if (_core->priv->watch_all == watch_all)
		return;

	if (!galago_is_connected())
		return;

	dbus_conn = galago_get_dbus_conn();

	_core->priv->watch_all = watch_all;

	dbus_error_init(&error);

	if (watch_all)
	{
		ADD_SIGNAL_MATCH(GALAGO_DBUS_CORE_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_SERVICE_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_ACCOUNT_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_IMAGE_INTERFACE);
		ADD_SIGNAL_MATCH(GALAGO_DBUS_PRESENCE_INTERFACE);
	}
	else
	{
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_CORE_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_SERVICE_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_ACCOUNT_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_IMAGE_INTERFACE);
		REMOVE_SIGNAL_MATCH(GALAGO_DBUS_PRESENCE_INTERFACE);
	}

exit:
	dbus_error_free(&error);
}

#undef ADD_SIGNAL_MATCH
#undef REMOVE_SIGNAL_MATCH

gboolean
galago_get_watch_all(void)
{
	g_return_val_if_fail(galago_is_initted(),   FALSE);
	g_return_val_if_fail(galago_is_connected(), FALSE);

	return _core->priv->watch_all;
}

GalagoService *
galago_get_service(const char *id, GalagoOrigin origin, gboolean query)
{
	GalagoService *service;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(id != NULL,          NULL);

	galago_context_push(galago_object_get_context(GALAGO_OBJECT(_core)));

	service = galago_context_get_service(id, origin);

	if (query && service == NULL && origin == GALAGO_REMOTE &&
		!galago_is_daemon() && galago_is_connected())
	{
		service = galago_dbus_send_message_with_reply(GALAGO_OBJECT(_core),
			"GetService",
			galago_value_new_object(GALAGO_TYPE_SERVICE, NULL),
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &id, NULL),
			NULL);
	}

	galago_context_pop();

	return service;
}

GList *
galago_get_services(GalagoOrigin origin, gboolean query)
{
	GList *services;

	galago_context_push(galago_object_get_context(GALAGO_OBJECT(_core)));
	services = galago_context_get_services(origin);

	if (query && origin == GALAGO_REMOTE &&
		!galago_is_daemon() && galago_is_connected())
	{
		GList *temp;

		temp = galago_dbus_send_message_with_reply(GALAGO_OBJECT(_core),
			"GetServices",
			galago_value_new_list(GALAGO_VALUE_TYPE_OBJECT, NULL,
								  (void *)GALAGO_TYPE_SERVICE),
			NULL);
		g_list_free(temp);

		services = galago_context_get_services(origin);
	}

	galago_context_pop();

	return services;
}

GalagoPerson *
galago_get_person(const char *id, GalagoOrigin origin, gboolean query)
{
	GalagoPerson *person;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(id != NULL,          NULL);

	galago_context_push(galago_object_get_context(GALAGO_OBJECT(_core)));

	person = galago_context_get_person(id, origin);

	if (query && person == NULL && origin == GALAGO_REMOTE &&
		!galago_is_daemon() && galago_is_connected())
	{
		person = galago_dbus_send_message_with_reply(GALAGO_OBJECT(_core),
			"GetPerson",
			galago_value_new_object(GALAGO_TYPE_PERSON, NULL),
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &id, NULL),
			NULL);
	}

	galago_context_pop();

	return person;
}

GList *
galago_get_people(GalagoOrigin origin, gboolean query)
{
	GList *people;

	galago_context_push(galago_object_get_context(GALAGO_OBJECT(_core)));
	people = galago_context_get_people(origin);

	if (query && origin == GALAGO_REMOTE &&
		!galago_is_daemon() && galago_is_connected())
	{
		GList *temp;

		temp = galago_dbus_send_message_with_reply(GALAGO_OBJECT(_core),
			"GetPeople",
			galago_value_new_list(GALAGO_VALUE_TYPE_OBJECT, NULL,
								  (void *)GALAGO_TYPE_PERSON),
			NULL);
		g_list_free(temp);

		people = galago_context_get_people(origin);
	}

	galago_context_pop();

	return people;
}

GalagoPerson *
galago_get_me(GalagoOrigin origin, gboolean query)
{
	GalagoPerson *me;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(GALAGO_ORIGIN_IS_VALID(origin), NULL);

	galago_context_push(galago_object_get_context(GALAGO_OBJECT(_core)));

	me = galago_get_person(GALAGO_ME_ID, origin, FALSE);

	if (me == NULL)
	{
		switch (origin)
		{
			case GALAGO_LOCAL:
				return galago_create_person(GALAGO_ME_ID);

			case GALAGO_REMOTE:
				if (query && !galago_is_daemon() && galago_is_connected())
				{
					me = galago_dbus_send_message_with_reply(
						GALAGO_OBJECT(_core),
						"GetMe",
						galago_value_new_object(GALAGO_TYPE_PERSON, NULL),
						NULL);
				}
				break;

			default:
				g_assert_not_reached();
		}
	}

	galago_context_pop();

	return me;
}

GalagoCore *
galago_get_core(void)
{
	return _core;
}

/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static gboolean
_galago_dbus_register_connection(void)
{
	GList *list, *return_list = NULL;
	gboolean is_feed;

	return_list = g_list_append(return_list,
		galago_value_new(GALAGO_VALUE_TYPE_STRING, NULL, NULL));
	return_list = g_list_append(return_list,
		galago_value_new(GALAGO_VALUE_TYPE_STRING, NULL, NULL));

	_core->priv->registering_connection = TRUE;

	is_feed = galago_is_feed();

	list = galago_dbus_send_message_with_reply_list(
		GALAGO_OBJECT(_core), "Register", return_list,
		galago_value_new(GALAGO_VALUE_TYPE_STRING,  &_core->priv->app_name, NULL),
		galago_value_new(GALAGO_VALUE_TYPE_BOOLEAN, &is_feed,         NULL),
		NULL);

	_core->priv->registering_connection = FALSE;

	if (list == NULL)
	{
		g_warning("Unable to register local Galago connection.");

		return FALSE;
	}

	/* TODO: Do safety checks. */
	_core->priv->uid           = list->data;
	_core->priv->conn_obj_path = list->next->data;
	galago_context_set_obj_path_prefix(_core->priv->conn_obj_path);

	g_list_free(list);

	_core->priv->daemon_active = TRUE;
	galago_set_watch_all(TRUE);
	_core->priv->registered = TRUE;

	g_signal_emit(_core, signals[REGISTERED], 0);

	return TRUE;
}

static void
_galago_dbus_unregister_connection(void)
{
	const char *obj_path = galago_get_client_obj_path();

	if (!galago_is_connected() || obj_path == NULL)
		return;

	galago_dbus_send_message(GALAGO_OBJECT(_core), "Unregister",
		galago_value_new(GALAGO_VALUE_TYPE_STRING, &obj_path, NULL),
		NULL);

	dbus_connection_flush(_core->priv->dbus_conn); /* Keep this here! */

	_galago_dbus_unregister_connection_finish();
}

static void
_galago_dbus_unregister_connection_finish(void)
{
	galago_context_clear_objects(GALAGO_REMOTE);
	galago_set_watch_all(FALSE);

	g_free(_core->priv->uid);
	g_free(_core->priv->conn_obj_path);

	_core->priv->uid = NULL;
	_core->priv->conn_obj_path = NULL;
	_core->priv->daemon_active = FALSE;
	_core->priv->registered = FALSE;

	g_signal_emit(_core, signals[UNREGISTERED], 0);
}

static void
_galago_dbus_core_add_service(GalagoService *service)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	const char *obj_path;
	DBusError error;

	g_signal_emit(_core, signals[SERVICE_ADDED], 0, service);

	if (!galago_is_connected() || !galago_is_feed() ||
		GALAGO_OBJECT_IS_REMOTE(service))
	{
		return;
	}

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_get_client_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "AddService");
	g_return_if_fail(message != NULL);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(service));

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(galago_get_dbus_conn(),
													  message, -1, &error);
	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		g_warning("Error sending AddService: %s", error.message);
		return;
	}

	g_assert(reply != NULL);

	dbus_message_iter_init(reply, &iter);
	dbus_message_iter_get_basic(&iter, &obj_path);
	galago_object_set_dbus_path(GALAGO_OBJECT(service), obj_path);
	dbus_message_unref(reply);
}

static void
_galago_dbus_core_remove_service(GalagoService *service)
{
	DBusMessage *message;
	DBusMessageIter iter;

	g_signal_emit(_core, signals[SERVICE_REMOVED], 0, service);

	if (!galago_is_connected() || !galago_is_feed() ||
		GALAGO_OBJECT_IS_REMOTE(service))
	{
		return;
	}

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_get_client_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "RemoveService");

	g_return_if_fail(message != NULL);

	dbus_message_set_no_reply(message, TRUE);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(service));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

static void
_galago_dbus_core_add_person(GalagoPerson *person)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	const char *obj_path, *uid;
	DBusError error;

	g_signal_emit(_core, signals[PERSON_ADDED], 0, person);

	if (!galago_is_connected() || !galago_is_feed() ||
		GALAGO_OBJECT_IS_REMOTE(person))
	{
		return;
	}

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_get_client_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "AddPerson");

	g_return_if_fail(message != NULL);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(person));

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(galago_get_dbus_conn(),
													  message, -1, &error);
	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		g_warning("Error sending AddPerson: %s", error.message);
		return;
	}

	g_assert(reply != NULL);

	dbus_message_iter_init(reply, &iter);
	dbus_message_iter_get_basic(&iter, &obj_path);
	dbus_message_iter_next(&iter);
	dbus_message_iter_get_basic(&iter, &uid);

	galago_object_set_dbus_path(GALAGO_OBJECT(person), obj_path);
	_galago_person_set_id(person, uid);

	dbus_message_unref(reply);
}

static void
_galago_dbus_core_remove_person(GalagoPerson *person)
{
	DBusMessage *message;
	DBusMessageIter iter;

	g_signal_emit(_core, signals[PERSON_REMOVED], 0, person);

	if (!galago_is_connected() || !galago_is_feed() ||
		GALAGO_OBJECT_IS_REMOTE(person))
	{
		return;
	}

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   galago_get_client_obj_path(),
										   GALAGO_DBUS_CORE_INTERFACE,
										   "RemovePerson");

	g_return_if_fail(message != NULL);

	dbus_message_set_no_reply(message, TRUE);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(person));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}
