/**
 * @file libgalago/galago-context.c Galago Context API
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-core.h>
#include <string.h>

struct _GalagoContextPrivate
{
	GalagoContextOps *ops;

	char *obj_path_prefix;

	GHashTable *services_table;
	GHashTable *people_table;
	GHashTable *obj_tree;

	struct
	{
		GList *services;
		GList *people;

	} local;

	struct
	{
		GList *services;
		GList *people;

	} remote;
};

typedef struct
{
	char *id;
	GalagoOrigin origin;

} ServiceCacheKey;

typedef struct
{
	char *id;
	char *session_id;
	GalagoOrigin origin;

} PersonCacheKey;


/**************************************************************************
 * Object initialization
 **************************************************************************/
static void galago_context_destroy(GalagoObject *gobject);

G_LOCK_DEFINE_STATIC(_contexts_lock);
static GList *contexts = NULL;

static GalagoObjectClass *parent_class = NULL;

G_DEFINE_TYPE(GalagoContext, galago_context, GALAGO_TYPE_OBJECT);

static void
galago_context_class_init(GalagoContextClass *klass)
{
	GalagoObjectClass *object_class = GALAGO_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->destroy = galago_context_destroy;
}

static guint
service_cache_key_hash(gconstpointer key)
{
	return g_str_hash(((ServiceCacheKey *)key)->id);
}

static guint
person_cache_key_hash(gconstpointer key)
{
	return g_str_hash(((PersonCacheKey *)key)->session_id);
}

static gboolean
service_cache_key_equal(gconstpointer a, gconstpointer b)
{
	ServiceCacheKey *key1 = (ServiceCacheKey *)a;
	ServiceCacheKey *key2 = (ServiceCacheKey *)b;

#if 0
	g_assert(key1->id != NULL || key1->session_id != NULL);
	g_assert(key2->id != NULL || key2->session_id != NULL);
#endif

	return key1->origin == key2->origin && g_str_equal(key1->id, key2->id);
}

static gboolean
person_cache_key_equal(gconstpointer a, gconstpointer b)
{
	PersonCacheKey *key1 = (PersonCacheKey *)a;
	PersonCacheKey *key2 = (PersonCacheKey *)b;

#if 0
	g_assert(key1->id != NULL || key1->session_id != NULL);
	g_assert(key2->id != NULL || key2->session_id != NULL);
#endif

	return key1->origin == key2->origin &&
	       ((key1->id != NULL && key2->id != NULL &&
			 g_str_equal(key1->id, key2->id)) ||
			(key1->session_id != NULL && key2->session_id != NULL &&
			 g_str_equal(key1->session_id, key2->session_id)));
}

static void
service_cache_key_destroy(void *ptr)
{
	ServiceCacheKey *key = (ServiceCacheKey *)ptr;

	g_free(key->id);
	g_free(key);
}

static void
person_cache_key_destroy(void *ptr)
{
	PersonCacheKey *key = (PersonCacheKey *)ptr;

	g_free(key->id);
	g_free(key->session_id);
	g_free(key);
}

static void
galago_context_init(GalagoContext *context)
{
	context->priv = g_new0(GalagoContextPrivate, 1);

	context->priv->services_table =
		g_hash_table_new_full(service_cache_key_hash, service_cache_key_equal,
							  service_cache_key_destroy, NULL);

	context->priv->people_table =
		g_hash_table_new_full(person_cache_key_hash, person_cache_key_equal,
							  person_cache_key_destroy, NULL);

	context->priv->obj_tree =
		g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
}

static void
galago_context_destroy(GalagoObject *object)
{
	GalagoContext *context = GALAGO_CONTEXT(object);

	if (context->priv != NULL)
	{
		g_list_foreach(context->priv->local.services,
					   (GFunc)galago_object_destroy, NULL);
		g_list_foreach(context->priv->remote.services,
					   (GFunc)galago_object_destroy, NULL);
		g_list_foreach(context->priv->local.people,
					   (GFunc)galago_object_destroy, NULL);
		g_list_foreach(context->priv->remote.people,
					   (GFunc)galago_object_destroy, NULL);

		g_hash_table_destroy(context->priv->services_table);
		g_hash_table_destroy(context->priv->people_table);
		g_hash_table_destroy(context->priv->obj_tree);

		g_free(context->priv->obj_path_prefix);
		g_free(context->priv);
		context->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

GalagoContext *
galago_context_new(void)
{
	return GALAGO_CONTEXT(g_object_new(GALAGO_TYPE_CONTEXT,
									   "origin", GALAGO_LOCAL,
									   NULL));
}

void
galago_context_push(GalagoContext *context)
{
	g_return_if_fail(context != NULL);
	g_return_if_fail(GALAGO_IS_CONTEXT(context));

	G_LOCK(_contexts_lock);
	contexts = g_list_prepend(contexts, context);
	G_UNLOCK(_contexts_lock);
}

void
galago_context_pop(void)
{
	GalagoContext *context;

	G_LOCK(_contexts_lock);

	context = galago_context_get();

	if (context != NULL)
		contexts = g_list_remove(contexts, context);

	G_UNLOCK(_contexts_lock);
}

GalagoContext *
galago_context_get(void)
{
	if (contexts == NULL)
		return NULL;

	return GALAGO_CONTEXT(contexts->data);
}

void
galago_context_set_obj_path_prefix(const char *prefix)
{
	GalagoContext *context;

	g_return_if_fail(prefix != NULL);

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	g_free(context->priv->obj_path_prefix);
	context->priv->obj_path_prefix = (prefix == NULL ? NULL : g_strdup(prefix));
}

const char *
galago_context_get_obj_path_prefix(void)
{
	GalagoContext *context;

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	return context->priv->obj_path_prefix;
}

void
galago_context_set_ops(GalagoContext *context, GalagoContextOps *ops)
{
	g_return_if_fail(context != NULL);

	context->priv->ops = ops;
}

void
galago_context_add_service(GalagoService *service)
{
	ServiceCacheKey *key;
	GalagoContext *context;

	g_return_if_fail(galago_is_initted());
	g_return_if_fail(service != NULL);
	g_return_if_fail(GALAGO_IS_SERVICE(service));

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	key         = g_new0(ServiceCacheKey, 1);
	key->id     = g_ascii_strdown(galago_service_get_id(service), -1);
	key->origin = galago_object_get_origin(GALAGO_OBJECT(service));

	if (galago_context_get_service(key->id, key->origin) != NULL)
	{
		switch (key->origin)
		{
			case GALAGO_LOCAL:
				g_warning("A local service with ID %s has already been added.",
						  key->id);
				break;

			case GALAGO_REMOTE:
				g_warning("A remote service with ID %s has already been added.",
						  key->id);
				break;

			default:
				g_assert_not_reached();
		}

		service_cache_key_destroy(key);

		return;
	}

	g_hash_table_insert(context->priv->services_table, key, service);

	switch (key->origin)
	{
		case GALAGO_LOCAL:
			context->priv->local.services =
				g_list_append(context->priv->local.services, service);
			break;

		case GALAGO_REMOTE:
			context->priv->remote.services =
				g_list_append(context->priv->remote.services, service);
			break;

		default:
			g_assert_not_reached();
	}

	if (context->priv->ops != NULL &&
		context->priv->ops->service_added != NULL)
	{
		context->priv->ops->service_added(service);
	}
}

void
galago_context_remove_service(GalagoService *service)
{
	GalagoContext *context;
	ServiceCacheKey key;

	g_return_if_fail(galago_is_initted());
	g_return_if_fail(service != NULL);
	g_return_if_fail(GALAGO_IS_SERVICE(service));

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	key.id     = g_ascii_strdown(galago_service_get_id(service), -1);
	key.origin = galago_object_get_origin(GALAGO_OBJECT(service));

	switch (key.origin)
	{
		case GALAGO_LOCAL:
			context->priv->local.services =
				g_list_remove(context->priv->local.services, service);
			break;

		case GALAGO_REMOTE:
			context->priv->remote.services =
				g_list_remove(context->priv->remote.services, service);
			break;

		default:
			g_assert_not_reached();
	}

	g_hash_table_remove(context->priv->services_table, &key);

	g_free(key.id);

	if (context->priv->ops != NULL &&
		context->priv->ops->service_removed != NULL)
	{
		context->priv->ops->service_removed(service);
	}
}

GalagoService *
galago_context_get_service(const char *id, GalagoOrigin origin)
{
	GalagoContext *context;
	GalagoService *service;
	ServiceCacheKey key;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(id != NULL, NULL);
	g_return_val_if_fail(GALAGO_ORIGIN_IS_VALID(origin), NULL);

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	key.id     = g_ascii_strdown(id, -1);
	key.origin = origin;

	service = g_hash_table_lookup(context->priv->services_table, &key);

	g_free(key.id);

	return service;
}

GList *
galago_context_get_services(GalagoOrigin origin)
{
	GalagoContext *context;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(GALAGO_ORIGIN_IS_VALID(origin), NULL);

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	switch (origin)
	{
		case GALAGO_LOCAL:
			return context->priv->local.services;

		case GALAGO_REMOTE:
			return context->priv->remote.services;

		default:
			g_assert_not_reached();
			return NULL;
	}
}

static void
add_person_to_table(GalagoPerson *person)
{
	GalagoContext *context = galago_context_get();
	const char *id = galago_person_get_id(person);
	const char *session_id = galago_person_get_session_id(person);
	PersonCacheKey *key;

	key = g_new0(PersonCacheKey, 1);
	key->id         = (id == NULL ? NULL : g_ascii_strdown(id, -1));
	key->session_id = g_strdup(session_id);
	key->origin     = galago_object_get_origin(GALAGO_OBJECT(person));
	g_hash_table_replace(context->priv->people_table, key, person);
}

static void
remove_person_from_table(GalagoPerson *person)
{
	PersonCacheKey key;
	GalagoContext *context = galago_context_get();
	const char *id = galago_person_get_id(person);

	key.id         = (id == NULL ? NULL : g_ascii_strdown(id, -1));
	key.session_id = g_strdup(galago_person_get_session_id(person));
	key.origin     = galago_object_get_origin(GALAGO_OBJECT(person));
	g_hash_table_remove(context->priv->people_table, &key);
	g_free(key.id);
	g_free(key.session_id);
}

static void
person_id_changed_cb(GalagoPerson *person)
{
	const char *id = galago_person_get_id(person);
	const char *old_id = g_object_get_data(G_OBJECT(person),
										   "_galago_old_id");

	if (id == old_id || (old_id != NULL && !strcmp(id, old_id)))
		return;

	remove_person_from_table(person);
	add_person_to_table(person);

	g_object_set_data_full(G_OBJECT(person), "_galago_old_id",
						   (id == NULL ? NULL : g_strdup(id)), g_free);
}

void
galago_context_add_person(GalagoPerson *person)
{
	GalagoContext *context;
	const char *session_id;
	GalagoOrigin origin;
	unsigned int signal_id;

	g_return_if_fail(galago_is_initted());
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	origin     = galago_object_get_origin(GALAGO_OBJECT(person));
	session_id = galago_person_get_session_id(person);

	if (session_id != NULL)
	{
		if (galago_context_get_person_with_session_id(session_id, origin)
			!= NULL)
		{
			switch (origin)
			{
				case GALAGO_LOCAL:
					g_warning("A local person with seesion ID %s has already "
							  "been added.", session_id);
					break;

				case GALAGO_REMOTE:
					g_warning("A remote person with session ID %s has already "
							  "been added.", session_id);
					break;

				default:
					g_assert_not_reached();
			}

			return;
		}
	}

	switch (origin)
	{
		case GALAGO_LOCAL:
			context->priv->local.people =
				g_list_append(context->priv->local.people, person);
			break;

		case GALAGO_REMOTE:
			context->priv->remote.people =
				g_list_append(context->priv->remote.people, person);
			break;

		default:
			g_assert_not_reached();
	}

	signal_id = g_signal_connect(G_OBJECT(person), "notify::id",
								 G_CALLBACK(person_id_changed_cb), NULL);
	g_object_set_data(G_OBJECT(person), "_galago_id_signal_id",
					  GINT_TO_POINTER(signal_id));

	add_person_to_table(person);

	if (context->priv->ops != NULL &&
		context->priv->ops->person_added != NULL)
	{
		context->priv->ops->person_added(person);
	}
}

void
galago_context_remove_person(GalagoPerson *person)
{
	GalagoContext *context;
	unsigned int signal_id;

	g_return_if_fail(galago_is_initted());
	g_return_if_fail(person != NULL);
	g_return_if_fail(GALAGO_IS_PERSON(person));

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	switch (galago_object_get_origin(GALAGO_OBJECT(person)))
	{
		case GALAGO_LOCAL:
			context->priv->local.people =
				g_list_remove(context->priv->local.people, person);
			break;

		case GALAGO_REMOTE:
			context->priv->remote.people =
				g_list_remove(context->priv->remote.people, person);
			break;

		default:
			g_assert_not_reached();
	}

	remove_person_from_table(person);

	signal_id = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(person),
												  "_galago_id_signal_id"));
	if (signal_id > 0)
	{
		g_signal_handler_disconnect(person, signal_id);
		g_object_set_data(G_OBJECT(person), "_galago_id_signal_id", NULL);
	}

	g_object_set_data(G_OBJECT(person), "_galago_id_signal_id", NULL);

	if (context->priv->ops != NULL &&
		context->priv->ops->person_removed != NULL)
	{
		context->priv->ops->person_removed(person);
	}
}

GalagoPerson *
galago_context_get_person(const char *id, GalagoOrigin origin)
{
	GalagoContext *context;
	GalagoPerson *person;
	PersonCacheKey key;

	g_return_val_if_fail(galago_is_initted(), FALSE);
	g_return_val_if_fail(id != NULL, NULL);
	g_return_val_if_fail(GALAGO_ORIGIN_IS_VALID(origin), NULL);

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	key.id         = g_ascii_strdown(id, -1);
	key.session_id = g_ascii_strdown(id, -1); // XXX
	key.origin     = origin;

	person = g_hash_table_lookup(context->priv->people_table, &key);

	g_free(key.id);
	g_free(key.session_id); // XXX

	return person;
}

GalagoPerson *
galago_context_get_person_with_session_id(const char *session_id,
										  GalagoOrigin origin)
{
	GalagoContext *context;
	GalagoPerson *person;
	PersonCacheKey key;

	g_return_val_if_fail(galago_is_initted(), FALSE);
	g_return_val_if_fail(session_id != NULL, NULL);
	g_return_val_if_fail(GALAGO_ORIGIN_IS_VALID(origin), NULL);

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	key.id         = NULL;
	key.session_id = g_ascii_strdown(session_id, -1);
	key.origin     = origin;

	person = g_hash_table_lookup(context->priv->people_table, &key);

	g_free(key.session_id);

	return person;
}

GList *
galago_context_get_people(GalagoOrigin origin)
{
	GalagoContext *context;

	g_return_val_if_fail(galago_is_initted(), NULL);
	g_return_val_if_fail(GALAGO_ORIGIN_IS_VALID(origin), NULL);

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	switch (origin)
	{
		case GALAGO_LOCAL:
			return context->priv->local.people;

		case GALAGO_REMOTE:
			return context->priv->remote.people;

		default:
			g_assert_not_reached();
			return NULL;
	}
}

void
galago_context_add_object(GalagoObject *obj)
{
	GalagoContext *context;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(obj));
	g_return_if_fail(galago_object_get_dbus_path(obj) != NULL);
	g_return_if_fail(galago_is_initted());

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	g_hash_table_insert(context->priv->obj_tree,
						g_strdup(galago_object_get_dbus_path(obj)), obj);
}

void
galago_context_remove_object(GalagoObject *obj)
{
	GalagoContext *context;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(obj));
	g_return_if_fail(galago_object_get_dbus_path(obj) != NULL);
	g_return_if_fail(galago_is_initted());

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	g_hash_table_remove(context->priv->obj_tree,
						galago_object_get_dbus_path(obj));
}

void
galago_context_clear_objects(GalagoOrigin origin)
{
	GalagoContext *context;

	g_return_if_fail(galago_is_initted());
	g_return_if_fail(GALAGO_ORIGIN_IS_VALID(origin));

	context = galago_context_get();
	g_return_if_fail(context != NULL);

	switch (origin)
	{
		case GALAGO_LOCAL:
			g_list_foreach(context->priv->local.services,
						   (GFunc)galago_object_destroy, NULL);
			g_list_foreach(context->priv->local.people,
						   (GFunc)galago_object_destroy, NULL);

			context->priv->local.services = NULL;
			context->priv->local.people   = NULL;
			break;

		case GALAGO_REMOTE:
			g_list_foreach(context->priv->remote.services,
						   (GFunc)galago_object_destroy, NULL);
			g_list_foreach(context->priv->remote.people,
						   (GFunc)galago_object_destroy, NULL);

			context->priv->remote.services = NULL;
			context->priv->remote.people   = NULL;
			break;

		default:
			g_assert_not_reached();
	}
}

GalagoObject *
galago_context_get_object(const char *path)
{
	GalagoContext *context;

	g_return_val_if_fail(path != NULL,        NULL);
	g_return_val_if_fail(galago_is_initted(), NULL);

	context = galago_context_get();
	g_return_val_if_fail(context != NULL, NULL);

	return GALAGO_OBJECT(g_hash_table_lookup(context->priv->obj_tree, path));
}
