/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __GST_DSP_VIDEO_H__
#define __GST_DSP_VIDEO_H__

#ifdef DEBUG
#define DBG_PRINT(...) g_print(__VA_ARGS__)
#else
#define DBG_PRINT(...) ((void)(0))
#endif

// System
#include <math.h>

#include <sys/time.h>
#include <time.h>

// Gstreamer
#include <glib.h>
#include <gst/gst.h>

#include <dsp/video_dsptask.h>

#define DSP_XPOS_DEFAULT         174
#define DSP_YPOS_DEFAULT         60
#define DSP_SCRNWIDTH_DEFAULT    602
#define DSP_SCRNHEIGHT_DEFAULT   362
#define DSP_ROTATION_DEFAULT     0
#define DSP_FLIPPING_DEFAULT     0

#define DSP_XPOS_MAX          1024
#define DSP_YPOS_MAX          1024
#define DSP_SCRNWIDTH_MAX     800
#define DSP_SCRNHEIGHT_MAX    480

#define DSP_ROTATION_MAX     270
#define DSP_FLIPPING_MAX     3


enum
{
  DSPVIDEO_PROP_0,
  DSPVIDEO_PROP_XPOS,
  DSPVIDEO_PROP_YPOS,
  DSPVIDEO_PROP_SCRNWIDTH,
  DSPVIDEO_PROP_SCRNHEIGHT,
  DSPVIDEO_PROP_ROTATION,
  DSPVIDEO_PROP_FLIPPING,
  DSPVIDEO_PROP_AUDIO_STREAMID,
  DSPVIDEO_PROP_FULLSCREEN,
  DSPVIDEO_PROP_VIDEOONLY,
  DSPVIDEO_PROP_FRAME_BY_FRAME,
  DSPVIDEO_PROP_ARM_TIMESTAMPS,
  DSPVIDEO_PROP_LAST
};


G_BEGIN_DECLS

typedef struct _GstDSPVideo GstDSPVideo;

struct _GstDSPVideo {
  gchar          *device;
  GMutex         *dsp_mutex;
  GMutex         *sync_mutex;
  int            fd;
  int            syncfd;
  guint          dsp_buffer_size;
  gdouble        aspect_ratio;

  // Properties
  guint          xpos;
  guint          ypos;
  guint          scrn_width;
  guint          scrn_height;
  guint          rotation;
  guint          flipping;
  guint          audio_stream_id;
  gboolean       fullscreen;
  gboolean       videoonly;
  gboolean       waiting_for_eos;
  gboolean       stream_id_set;
  gdouble        pixel_aspect_ratio;
  gboolean       frame_by_frame;
  gboolean       arm_timestamps;

  // Stream information
  guint          width;
  guint          height;
  guint          bpp;
  guint          color_format;
  gdouble        fps;

  // DSP communication related data
  gboolean       write_pending;
  guint          writes_done;
  guint          writereqs_got;
  guint          mode;
  gint           read_index;
  gint           write_index;
  guint          error_status;
  guint          error_cmd;

  // for testing, timing measurements etc. purposes
  int            log;
  struct timeval        time;
};


GstDSPVideo *
gst_dspvideo_new();

void
gst_dspvideo_destroy(GstDSPVideo *dsp);

gboolean
gst_dspvideo_open(GstDSPVideo *dsp, gchar *device);

gboolean
gst_dspvideo_close_dsp(GstDSPVideo *dsp);

gboolean
gst_dspvideo_close(GstDSPVideo *dsp);

GstClockTime
gst_dspvideo_get_time(GstDSPVideo *dsp);

gboolean
gst_dspvideo_avsync_initialize(GstDSPVideo *dsp);

gboolean
gst_dspvideo_initialize(GstDSPVideo *dsp, void *data, guint dataLen);

gboolean
gst_dspvideo_setparams(GstDSPVideo *dsp);

gboolean
gst_dspvideo_endstream(GstDSPVideo * dsp);

//gboolean
//gst_dspvideo_setup_vpop(GstDSPVideo *dsp);

gboolean
gst_dspvideo_play(GstDSPVideo *dsp);

gboolean
gst_dspvideo_discont(GstDSPVideo *dsp);

gboolean
gst_dspvideo_pause(GstDSPVideo *dsp);

gboolean
gst_dspvideo_pause_sync(GstDSPVideo *dsp);

gboolean
gst_dspvideo_stop(GstDSPVideo *dsp);

gboolean
gst_dspvideo_wait_eof( GstDSPVideo * dsp );

gboolean
gst_dspvideo_set_property (GstDSPVideo *dsp, guint prop_id, const GValue *value);

gboolean
gst_dspvideo_get_property (GstDSPVideo *dsp, guint prop_id, GValue *value);

void
gst_dspvideo_install_properties (GObjectClass *klass);

gboolean
gst_dspvideo_set_mirror(GstDSPVideo *dsp);

gboolean
gst_dspvideo_set_rotation(GstDSPVideo *dsp);

gboolean
gst_dspvideo_check_read(GstDSPVideo *dsp);

gboolean
gst_dspvideo_read_pending(GstDSPVideo *dsp,
                          gboolean need_write,
                          gboolean need_cmd,
                          gboolean block);

gboolean
gst_dspvideo_play_sync(GstDSPVideo * dsp, gint strm_id);

gboolean
gst_dspvideo_peer_query(GstPad *pad, GstQuery *query);

G_END_DECLS


#endif

