/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

/* System includes */
#include <unistd.h>
#include <sys/mman.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* GST includes */
#include "gstdspmpeg4sink.h"
#include <gst/gst.h>

/* DSP includes */
#include "omapdsp.h"
#include <dsp/interface_common.h>
#include <dsp/video_dsptask.h>



static void gst_dspmpeg4_class_init (GstDSPMPEG4SinkClass * klass);
static void gst_dspmpeg4_base_init (gpointer g_class);
static void gst_dspmpeg4_init (GstDSPMPEG4Sink * dspmpeg4);

//static gboolean gst_dspmpeg4_send_event (GstElement *element, GstEvent * event);
static gboolean gst_dspmpeg4_basesink_event (GstBaseSink *sink, GstEvent * event);
static gboolean gst_dspmpeg4_initialize(GstDSPMPEG4Sink *dsp);
static GstCaps * gst_dspmpeg4_get_caps (GstBaseSink *sink);
static gboolean gst_dspmpeg4_set_caps (GstBaseSink *sink, GstCaps *caps);
static const GstQueryType * gst_dspmpeg4_get_query_types(GstElement * element);
static gboolean gst_dspmpeg4_query (GstElement * element, GstQuery * query);
static GstFlowReturn gst_dspmpeg4_render_fbf (GstBaseSink * bsink, GstBuffer * buf);
static GstFlowReturn gst_dspmpeg4_preroll (GstBaseSink * bsink, GstBuffer * buf);
static guint32 gst_dspmpeg4_takebit(guint8 * buf, guint pos, guint bits);
static gboolean gst_dspmpeg4_get_par_info(GstDSPMPEG4Sink * dspmpeg4, GstBuffer * buf);

static void gst_dspmpeg4_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_dspmpeg4_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

static GstStateChangeReturn
gst_dspmpeg4_change_state (GstElement * element, GstStateChange transition);
static void gst_dspmpeg4_finalize (GObject * object);


GstElementDetails gst_dspmpeg4_details = GST_ELEMENT_DETAILS ("DSP MPEG4/H.263 Sink",
    "Decoder/Sink/Video",
    "Raw MPEG4 and H.263 video sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

static gchar *device = "/dev/dsptask/mpeg4dec";


static GstStaticPadTemplate dspmpeg4_factory =
  GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (

        "video/x-divx, "
        "width = (int) [ 0, 4096 ], "
        "height = (int) [ 0, 4096 ], "
        "divxversion = (int) [ 4, 5 ];"

        "video/x-h263, "
        "width = (int) [ 0, 4096 ], "
        "height = (int) [ 0, 4096 ] "
        )
);

static GstVideoSinkClass *parent_class = NULL;

/**
 *
 *
 */

GType
gst_dspmpeg4sink_get_type (void)
{
  static GType dspmpeg4_type = 0;

  if (!dspmpeg4_type) {
    static const GTypeInfo dspmpeg4_info = {
      sizeof (GstDSPMPEG4SinkClass),
      gst_dspmpeg4_base_init,
      NULL,
      (GClassInitFunc) gst_dspmpeg4_class_init,
      NULL,
      NULL,
      sizeof (GstDSPMPEG4Sink),
      0,
      (GInstanceInitFunc) gst_dspmpeg4_init,
    };
    dspmpeg4_type =
        g_type_register_static (GST_TYPE_VIDEO_SINK, "GstDSPMPEG4Sink", &dspmpeg4_info,
        0);

//     static const GInterfaceInfo iface_info = {
//       (GInterfaceInitFunc) gst_dspmpeg4_interface_init,
//       NULL,
//       NULL,
//     };
//     static const GInterfaceInfo overlay_info = {
//       (GInterfaceInitFunc) gst_dspmpeg4_xoverlay_init,
//       NULL,
//       NULL,
//     };

//     g_type_add_interface_static (dspmpeg4_type, GST_TYPE_IMPLEMENTS_INTERFACE,
//         &iface_info);
//     g_type_add_interface_static (dspmpeg4_type, GST_TYPE_X_OVERLAY,
//         &overlay_info);

  }

  return dspmpeg4_type;
}


/**
 *
 *
 */

static void
gst_dspmpeg4_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspmpeg4_factory));
  gst_element_class_set_details (gstelement_class, &gst_dspmpeg4_details);

}


/**
 *
 *
 */

static void
gst_dspmpeg4_class_init (GstDSPMPEG4SinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbasesink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesink_class = (GstBaseSinkClass *) klass;

  parent_class = g_type_class_ref (GST_TYPE_VIDEO_SINK);

  gobject_class->set_property = gst_dspmpeg4_set_property;
  gobject_class->get_property = gst_dspmpeg4_get_property;
  gobject_class->finalize = gst_dspmpeg4_finalize;

  gstelement_class->change_state =
      GST_DEBUG_FUNCPTR (gst_dspmpeg4_change_state);
  gstelement_class->get_query_types = 
      GST_DEBUG_FUNCPTR(gst_dspmpeg4_get_query_types);
  gstelement_class->query = GST_DEBUG_FUNCPTR(gst_dspmpeg4_query);
//  gstelement_class->send_event = gst_dspmpeg4_send_event;
//   gstelement_class->get_formats = gst_dspmpeg4sink_get_formats;

  gstbasesink_class->get_caps = GST_DEBUG_FUNCPTR (gst_dspmpeg4_get_caps);
  gstbasesink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dspmpeg4_set_caps);
  gstbasesink_class->render = GST_DEBUG_FUNCPTR (gst_dspmpeg4_render_fbf);
  gstbasesink_class->event = GST_DEBUG_FUNCPTR (gst_dspmpeg4_basesink_event);
  gstbasesink_class->preroll = GST_DEBUG_FUNCPTR (gst_dspmpeg4_preroll);

  // Use dspvideo utility to install default properties
  gst_dspvideo_install_properties(G_OBJECT_CLASS (klass));


}


/**
 *
 *
 */

static void
gst_dspmpeg4_init (GstDSPMPEG4Sink * dspmpeg4)
{
  GstBaseSink *bsink = GST_BASE_SINK(dspmpeg4);
  dspmpeg4->video = gst_dspvideo_new();
  dspmpeg4->fbf_queue = g_queue_new();
  dspmpeg4->video->width = 176;
  dspmpeg4->video->height = 144;
  dspmpeg4->eos = FALSE;
  dspmpeg4->sinkpad = bsink->sinkpad;
  
}


/**
 *
 *
 */
static GstCaps *
gst_dspmpeg4_get_caps (GstBaseSink *sink)
{
  return gst_caps_copy (
    gst_pad_get_pad_template_caps (GST_VIDEO_SINK_PAD(sink)));
}



/**
 *
 * 
 */
static gboolean
gst_dspmpeg4_set_caps (GstBaseSink *sink, GstCaps *caps)
{
  GstStructure *structure;
  const gchar *mime;
  gint mpegversion = 0;
  GstDSPMPEG4Sink *dsp = GST_DSPMPEG4SINK(sink);

  structure = gst_caps_get_structure (caps, 0);
  if (!structure) { 
    GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
    return FALSE; 
  }
  mime = gst_structure_get_name (structure);
  DBG_PRINT("dspmpeg4: mime for stream: %s\n", mime);
  // Check DivX and "ordinary" MPEG mimetypes
  if (strcmp (mime, "video/x-divx") == 0 ) {

    if(gst_structure_has_field(structure,"divxversion")) {
      gst_structure_get_int(structure, "divxversion", &mpegversion);
    }
    else {
      DBG_PRINT("DSPMPEG4: connect: no divxversion version, refused\n");
      GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
      return FALSE;
    }

    DBG_PRINT("dspmpeg4: Divxversion is %d\n", mpegversion);
    if ( ( mpegversion < 4 ) ||
         ( mpegversion > 5 ) ) {
      DBG_PRINT("dspmpeg4: Invalid DivX version (Only supported is DivX4/5)\n");
      GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
      return FALSE;
    }
  }
  // Check H.263 mimetype
  else if(strcmp (mime, "video/x-h263") == 0 ) {
    DBG_PRINT("mpeg4: H.263 detected\n");
    // Do nothing
  }
  else {
    DBG_PRINT ("Wrong mimetype %s provided, we only support %s",
                 mime, "video/x-divx and video/x-h263");
    GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
    return FALSE;
  }

  // Ensure that MPEG version is 4
  /*
  if(mpegversion != 4) {
  GST_WARNING("DSPMPEG4: connect: illegal mpeg version, refused\n");
  return FALSE;
}
  */

  if(gst_structure_has_field(structure,"width")) {
    gst_structure_get_int(structure, "width", &dsp->video->width);
  }
  else {
    DBG_PRINT("DSPMPEG4: no width, caps refused\n");
    GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
    return FALSE;
  }

  if(gst_structure_has_field(structure,"height")) {
    gst_structure_get_int(structure, "height", &dsp->video->height);
  }
  else {
    DBG_PRINT("DSPMPEG4: no height, caps refused\n");
    GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
    return FALSE;
  }
  if ( (dsp->video->width < 16) ||
        (dsp->video->width > 352) ||
        (dsp->video->width % 16 ) ) {
    DBG_PRINT("dspmpeg4: Unsupported width (%d)\n", dsp->video->width);
    GST_ELEMENT_ERROR (dsp, STREAM, DECODE, (NULL), (NULL));
    return FALSE;
  } else {
      DBG_PRINT("dspmpeg4: width ok\n");
  }

  if ( (dsp->video->height < 16) ||
       (dsp->video->height > 288) ||
       (dsp->video->height % 16 ) ) {
    DBG_PRINT("dspmpeg4: Unsupported height (%d)\n", dsp->video->height);
    GST_ELEMENT_ERROR (dsp, STREAM, DECODE, (NULL), (NULL));
    return FALSE;
  } else {
    DBG_PRINT("dspmpeg4: height ok\n");
  }

  // Aspect ratio, x/y
  if ( ( dsp->video->width != 0 ) &&
       ( dsp->video->height != 0 ) ) {
    dsp->video->aspect_ratio = (gdouble) dsp->video->width / dsp->video->height;
    DBG_PRINT("dspmpeg4: Aspect ratio: %lf \n", dsp->video->aspect_ratio );
  }

  if(gst_structure_has_field(structure,"framerate")) {
    gint fps_n, fps_d;
    gdouble fps = 0.0;
    gst_structure_get_fraction(structure, "framerate", &fps_n, &fps_d);
    if ( fps_d > 0 ) 
      fps = (gdouble)fps_n / (gdouble)fps_d;
    if ( ( fps <= MPEG4_MIN_FPS ) ||
         ( fps > MPEG4_MAX_FPS  ) ) {
      DBG_PRINT("dspmpeg4: FPS %f either too high or too low!\n",fps);
      GST_ELEMENT_ERROR (dsp, STREAM, FORMAT, (NULL), (NULL));
      return FALSE;
    } else {
       DBG_PRINT("dspmpeg4: FPS %f ok\n", fps);
    }
  }

  DBG_PRINT("dspmpeg4: caps accepted!\n");
//  dsp->caps_parsed = TRUE;
  return TRUE;

}

/**
 *
 * This method handles events coming from basesink
 */

static gboolean gst_dspmpeg4_basesink_event (GstBaseSink *sink, GstEvent * event)
{
  DBG_PRINT("dspmpeg4: got event from basesink: %d\n", GST_EVENT_TYPE (event));
  GstDSPMPEG4Sink * dspmpeg4 = GST_DSPMPEG4SINK (sink);
  gboolean update = FALSE;
  gdouble rate = 0.0;
  GstFormat format = GST_FORMAT_UNDEFINED;
  gint64 start = 0, stop = 0, time = 0;

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  switch (type) {
  case GST_EVENT_EOS:
    DBG_PRINT("dspmpeg4: EOS EVENT\n");
    dspmpeg4->eos = TRUE;
    break;
  case GST_EVENT_NEWSEGMENT:
    gst_event_parse_new_segment (event, &update, &rate, &format,
                                 &start, &stop, &time);
    if ( format == GST_FORMAT_TIME ){
      DBG_PRINT("dspmpeg4: NEWSEGMENT EVENT (time: %"
                GST_TIME_FORMAT")\n", GST_TIME_ARGS(time));
      dspmpeg4->base_time = time;
    }
    else {
      GST_WARNING("dspmpeg4: seeking using format %d not supported",format);
    }
    break;
  case GST_EVENT_FLUSH_START:
    DBG_PRINT("dspmpeg4: FLUSH START EVENT\n");
    while ( !g_queue_is_empty(dspmpeg4->fbf_queue) ){
      gst_buffer_unref(g_queue_pop_head(dspmpeg4->fbf_queue));
      DBG_PRINT("dspmpeg4: flushing buffer\n");
    }
    if (dspmpeg4->discont_sent) {
      break;
    }
    gst_dspvideo_discont(dspmpeg4->video);
    dspmpeg4->discont_sent = TRUE;
    dspmpeg4->video->write_index = 0;
    break;
  default:
    break;
  }

  return TRUE;
}

/**
 *
 *
 */

static const GstQueryType *
gst_dspmpeg4_get_query_types(GstElement * element)
{
  static const GstQueryType gst_dspmpeg4_query_types[] =
  {
    GST_QUERY_POSITION,
    GST_QUERY_DURATION,
    0
  };

  return gst_dspmpeg4_query_types;
}

/**
 *
 *  This method overrides basesink query handling
 */

static gboolean
gst_dspmpeg4_query (GstElement * element, GstQuery * query)
{
  gboolean res = FALSE;

  GstDSPMPEG4Sink *dspmpeg4 = GST_DSPMPEG4SINK (element);

  switch (GST_QUERY_TYPE (query)) {
    case GST_QUERY_POSITION:
    {
      GstFormat format;
      GstClockTime pos = 0;

      gst_query_parse_position(query, &format, NULL);
      if ( format == GST_FORMAT_TIME ){
        // Ask position from dsp. 
        pos = gst_dspvideo_get_time(dspmpeg4->video);
        pos += dspmpeg4->base_time;
        gst_query_set_position(query, format, pos);
        DBG_PRINT("dspmpeg4: queried position from dsp:%"G_GUINT64_FORMAT"\n",pos);
        res = TRUE;
      }
      else {
        DBG_PRINT("dspmpeg4: position query format %d not handled.\n",format);
      }
      break;
    }
    case GST_QUERY_DURATION:
      res = gst_dspvideo_peer_query (dspmpeg4->sinkpad, query);
      break;
    default:
 //     res = gst_dspvideo_peer_query (GstPad *pad, GstQuery *query)
      DBG_PRINT("dspmpeg4: query type %d not handled.\n", GST_QUERY_TYPE(query));
      break;
  }
  return res;
}

/**
 * gst_dspmpeg4_fill_mmap_fbf:
 *
 * Fills given buffer data (one frame) to arm dsp mmap buffer.
 */
static void
gst_dspmpeg4_fill_mmap_fbf(GstDSPMPEG4Sink *dspmpeg4, GstBuffer *buf) {

  guint32 start_code = GST_READ_UINT32_BE(GST_BUFFER_DATA(buf));
  if ( start_code != 0x000001b0 && start_code != 0x00000100 ) {
    DBG_PRINT("MPEG4: start code %x, prepending with four 0 bytes\n",
              start_code);
    memset(&(dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index]),0,4);
    dspmpeg4->video->write_index += 4;
  } else {
    DBG_PRINT("MPEG4: start code %x, sending as-is\n",start_code);
  }
  DBG_PRINT("MPEG4: frame-by-frame, copying buf with size %d\n",
            GST_BUFFER_SIZE(buf));
  memcpy(&(dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index]), 
          GST_BUFFER_DATA(buf), GST_BUFFER_SIZE(buf));
  dspmpeg4->video->write_index += GST_BUFFER_SIZE(buf);
  // dsp reads two-byte words, so append 0x00 byte if buffer size not even
  if ( GST_BUFFER_SIZE(buf) % 2 ){
    DBG_PRINT("MPEG4: frame size is odd, appending 0x00 byte\n");
    dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index] = 0x00;
    dspmpeg4->video->write_index++;
  }

  // Append four 0x00 bytes after each frame to prevent 
  // dsp hang experienced with advanced simple profile clip.
  dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index] = 0x00;
  dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index+1] = 0x00;
  dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index+2] = 0x00;
  dspmpeg4->mmap_buf_addr[dspmpeg4->video->write_index+3] = 0x00;
  dspmpeg4->video->write_index += 4;
  
  return;

}


/**
 *
 *
 */

static GstFlowReturn
gst_dspmpeg4_render_fbf (GstBaseSink * bsink, GstBuffer * buf)
{
  GstDSPMPEG4Sink * dspmpeg4 = GST_DSPMPEG4SINK (bsink);
  GstFlowReturn ret = GST_FLOW_ERROR;
  VIDEO_DATA_WRITE v_cmd;
  guint wait_count = 0;

  DBG_PRINT("MPEG4 RENDER ENTER: %d bytes\n", GST_BUFFER_SIZE(buf));

  while (!dspmpeg4->video->stream_id_set && 
         !dspmpeg4->video->videoonly) {
    // FIXME we should receive stream id or videoonly indication 
    //       _before_ entering render.  
    DBG_PRINT("dspmpeg4: waiting for stream id or videoonly indication\n");
    g_usleep(100000);
    wait_count++;
    if ( wait_count > 5 ) {
      DBG_PRINT("dspmpeg4: stream id or videoonly not received, bail out.\n");
      g_queue_push_tail(dspmpeg4->fbf_queue, buf);
      gst_buffer_ref(buf);
      return GST_FLOW_OK;
    }
  } 

  if( dspmpeg4->video->mode == DSP_MODE_PAUSED ) {
    gst_dspvideo_play( dspmpeg4->video );
  }

  g_queue_push_tail(dspmpeg4->fbf_queue,buf);
  gst_buffer_ref(buf);

  // Feed all queued buffers to dsp.
  while ( !g_queue_is_empty(dspmpeg4->fbf_queue) ) {

    if (dspmpeg4->paused == TRUE){
      DBG_PRINT("dspmpeg4: Paused during render.\n");
      ret = GST_FLOW_OK;
      break;
    }
    g_mutex_lock(dspmpeg4->video->dsp_mutex);
    buf = g_queue_pop_head(dspmpeg4->fbf_queue);
    gst_dspvideo_read_pending(dspmpeg4->video, TRUE, FALSE, TRUE); // Blocking one... :-/
    dspmpeg4->video->write_pending = TRUE;

    if(dspmpeg4->video->mode != DSP_MODE_ERROR) {

      // Fill the MMAP buffer with video data
      gst_dspmpeg4_fill_mmap_fbf(dspmpeg4, buf);
      dspmpeg4->prev_timestamp = GST_BUFFER_TIMESTAMP(buf);
      guint32 ts = 0;
      if ( dspmpeg4->prev_timestamp < dspmpeg4->base_time ) {
        dspmpeg4->prev_timestamp = dspmpeg4->base_time;
        ts = 0;
      } else {
        ts = (guint32) ((dspmpeg4->prev_timestamp - dspmpeg4->base_time) / GST_MSECOND);
      }

      DBG_PRINT("MPEG4: Writing one frame, timestamp %"GST_TIME_FORMAT", %d ms\n",
                GST_TIME_ARGS(dspmpeg4->prev_timestamp),ts);
      v_cmd.mmap_buf_info = dspmpeg4->video->write_index / sizeof( short int );
      v_cmd.time_stamp = (ts >> 16 & 0x0000FFFF);
      v_cmd.time_stamp |= (ts << 16 & 0xFFFF0000);
      gst_buffer_unref(buf);

      if ( dspmpeg4->eos ) 
        v_cmd.dsp_cmd = DSP_CMD_EOF;        
      else 
        v_cmd.dsp_cmd = DSP_CMD_DATA_WRITE;
      v_cmd.mmap_buf_info = dspmpeg4->video->write_index / sizeof( short int );

      if(write(dspmpeg4->video->fd, &v_cmd, 4 * sizeof(short int)) < 0) {
        g_warning("dspmpeg4:DSP data write returned -1\n");
        GST_ELEMENT_ERROR (dspmpeg4, RESOURCE, WRITE, (NULL), ("DSP_CMD_DATA_WRITE -1"));
        g_mutex_unlock(dspmpeg4->video->dsp_mutex);
        ret = GST_FLOW_ERROR;
        break;
      }
      else {
        DBG_PRINT("dspmpeg4: Resetting write_index, was %d\n", dspmpeg4->video->write_index);
        dspmpeg4->video->write_index = 0;
        dspmpeg4->video->writes_done++;
        dspmpeg4->video->write_pending = FALSE;
        dspmpeg4->discont_sent = FALSE;
      }
      ret = GST_FLOW_OK;
    }
    else {
      DBG_PRINT("MPEG4: ERROR\n");
      GST_ELEMENT_ERROR (dspmpeg4, STREAM, FORMAT, (NULL), (NULL));
      ret = GST_FLOW_ERROR;
      g_mutex_unlock(dspmpeg4->video->dsp_mutex);
      break;
    }
    // We cannot free the lock earlier than here. Otherwise there is a chance
    // that DATA_WRITE gets sent to DSP even PAUSE has already been sent
    g_mutex_unlock(dspmpeg4->video->dsp_mutex);
  }
  return ret;
}


/**
 *
 *
 */

static GstFlowReturn
gst_dspmpeg4_preroll (GstBaseSink * bsink, GstBuffer * buf){

  DBG_PRINT("dspmpeg4: preroll\n");
  GstDSPMPEG4Sink *dspmpeg4 = GST_DSPMPEG4SINK(bsink);

  if ( dspmpeg4->par_info_handled == FALSE &&
       gst_dspmpeg4_get_par_info(dspmpeg4, buf) ){
    DBG_PRINT("dspmpeg4: PAR information found\n");
    gst_dspvideo_setparams(dspmpeg4->video);
  }

  // check par info from the first buffer only
  dspmpeg4->par_info_handled = TRUE;

  return GST_FLOW_OK;

}


static inline guint32 gst_dspmpeg4_convert_to_header(guint8 * buf) {
  return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}

/**
 *
 * Get wanted amount of bits from wanted position in the buf
 *
 *@param start address of buffer
 *@param pos bit position in the buffer
 *@param bits Amount of bits wanted. Max 32
 *
 *@return the bits
 */
 
static guint32
gst_dspmpeg4_takebit(guint8 * buf, guint pos, guint bits)
{
  guint32 res = 0;
  gint i = 0;
  
  for ( i=0; i<bits; i++ ) {
    res <<= 1;
    res |= (buf[pos>>3] >> (7 - (pos & 7))) & 1;
    pos++;
  }

  return res;
}


/**
 *
 * Scan for Pixel Aspect Ratio information from VOB headers.
 *
 * @param element dspmpeg4 element
 * @param buf buffer from which the par information is searched
 *
 * @return If TRUE, PAR info found
 *
 */
static gboolean
gst_dspmpeg4_get_par_info(GstDSPMPEG4Sink * dspmpeg4, GstBuffer * buf)
{
  gint i = 0;
  gint pos = 0;  
  guint32 header = 0;
  guint par = 0;
  guint par_w = 0;
  guint par_h = 0;
  gboolean ret = FALSE;
  gint size = 0;

  if ( dspmpeg4 == NULL || buf == NULL ) {
    return ret;
  }

  size = GST_BUFFER_SIZE(buf);

  DBG_PRINT("Looking for VOL startcode...\n");

  while ( i < size ) {
    header = gst_dspmpeg4_convert_to_header(GST_BUFFER_DATA(buf) + i);
    if ( ( header & 0xfffffff0 ) == 0x00000120 ) {
      DBG_PRINT("mpeg4: Got VOL startcode\n");
      break;
    }
    i++;
  }

  if ( ( header & 0xfffffff0 ) != 0x00000120 ) {
    DBG_PRINT("No VOL header found\n");
    return ret;
  }
  
  pos = i * 8;      

  // Jump over startcode, random acc. vol. and obj. type indication
  pos += (32+9);
  header = gst_dspmpeg4_takebit(GST_BUFFER_DATA(buf), pos, 1);
  // Obj. layer identifier present?
  if ( header ) {
    pos += 8;
  } else {
    pos++;
  }
  
  // Next is the PAR info
  par = gst_dspmpeg4_takebit(GST_BUFFER_DATA(buf), pos, 4);
  // Extended PAR?
  if ( par == 0x000F ) {        
    pos += 4;
    par_w = gst_dspmpeg4_takebit(GST_BUFFER_DATA(buf), pos, 8);
    pos += 8;
    par_h = gst_dspmpeg4_takebit(GST_BUFFER_DATA(buf), pos, 8);
  }

  if ( par ) {
    switch (par) {
      case 0x0001:
        DBG_PRINT("PAR 1:1\n");
        dspmpeg4->video->pixel_aspect_ratio = 1.0;
        break;
      case 0x0002:
        DBG_PRINT("PAR 12:11\n");
        dspmpeg4->video->pixel_aspect_ratio = 12.0 / 11.0;        
        break;
      case 0x0003:
        DBG_PRINT("PAR 10:11\n");
        dspmpeg4->video->pixel_aspect_ratio = 10.0 / 11.0;
        break;
      case 0x0004:
        DBG_PRINT("PAR 16:11\n");
        dspmpeg4->video->pixel_aspect_ratio = 16.0 / 11.0;
        break;
      case 0x0005:
        DBG_PRINT("PAR 40:33\n");
        dspmpeg4->video->pixel_aspect_ratio = 40.0 / 33.0;
        break;
      case 0x000F:
        DBG_PRINT("Extended PAR, where ratio is %d:%d\n",
                  par_w, par_h);
        dspmpeg4->video->pixel_aspect_ratio = (gdouble) par_w / (gdouble) par_h;
        break;
      default:
        break;
    }
    ret = TRUE;
  } else {
    DBG_PRINT("No PAR info!\n");
  }
  
  return ret;
}


/**
 *
 *
 */

static gboolean
gst_dspmpeg4_initialize(GstDSPMPEG4Sink *dsp)
{
  DSP_CMD_STATUS cmd;
  VIDEO_INIT_STATUS status;
  short int tmp;

  DBG_PRINT("mpeg4: init\n");

  while(gst_dspvideo_check_read(dsp->video)) {
    read(dsp->video->fd, &tmp, sizeof(short int));
    DBG_PRINT("mpeg4: from DSP: %d\n", tmp );
  }

  cmd.dsp_cmd = DSP_CMD_INIT;

  DBG_PRINT("mpeg4: writing init cmd\n");
  write(dsp->video->fd, &cmd, sizeof(short int));
  //DBG_PRINT("mpeg4: Reading init status\n");
  read(dsp->video->fd, &status, sizeof(VIDEO_INIT_STATUS));
  if(status.init_status != DSP_OK) {
    dsp->video->mode = DSP_MODE_ERROR;
    g_warning("failed to initialize dsp MPEG4 node, error = 0x%02X\n",
    status.init_status);
    return FALSE;
  }

  DBG_PRINT("mmap size from init: %d\n", status.mmap_buffer_size );
  dsp->video->dsp_buffer_size = status.mmap_buffer_size;
  dsp->write_size = status.mmap_buffer_size << 1;


  DBG_PRINT("DSP BUFSIZE = %d\n", dsp->video->dsp_buffer_size );
  DBG_PRINT("WRITE_SIZE = %d\n", dsp->write_size);

  DBG_PRINT("mmapping the device to memory\n");
  if ( ( dsp->mmap_buf_addr = (unsigned char *)mmap((void *)0,
         dsp->write_size * sizeof(unsigned char),
         PROT_READ | PROT_WRITE, MAP_SHARED, dsp->video->fd, 0) ) < 0) {
    g_warning("\nError mapping mmap buffer.\n\n");
    return FALSE;
  } else {
    DBG_PRINT("mmap successfull\n");
  }

  // Read the junk out...
  while(gst_dspvideo_check_read(dsp->video)) {
    read(dsp->video->fd, &tmp, sizeof(short int));
  }

  if(gst_dspvideo_avsync_initialize(dsp->video) == FALSE) {
    g_warning("AVSYNC INIT FAIL\n");
    return FALSE;
  }

  cmd.dsp_cmd = DSP_CMD_VIDEO_RESET_LOCKEDRATIO;
  write(dsp->video->fd, &cmd, sizeof(short int));
  read(dsp->video->fd, &cmd, sizeof(DSP_CMD_STATUS));
  if ( cmd.status != DSP_OK ) {
    DBG_PRINT("locked ratio reset fail!\n");
    return FALSE;
  }
  else DBG_PRINT("Locked ratio successfully reset.\n");

  dsp->video->write_index = 0;
  dsp->video->read_index = 0;
  dsp->video->writes_done = 0;
  dsp->max = 33000;

#ifdef DEBUG
  dsp->seek = 0;
  dsp->p_ctr = 0;
  dsp->d_count = 0;
#endif

  dsp->video->mode = DSP_MODE_INITIALIZED;
  return TRUE;
}


/**
 *
 *
 */

static GstStateChangeReturn
gst_dspmpeg4_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;
  GstDSPMPEG4Sink *dspmpeg4;
  short int tmp;

  dspmpeg4 = GST_DSPMPEG4SINK (element);

  switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
      DBG_PRINT("mpeg4: null to ready\n");
      if (gst_dspvideo_open(dspmpeg4->video, device) == FALSE) {
        // GST_ELEMENT_ERROR (element, CORE, STATE_CHANGE, (NULL), (NULL));
        return GST_STATE_CHANGE_FAILURE;
      }
      dspmpeg4->paused = FALSE;
      dspmpeg4->base_time = 0;
      dspmpeg4->par_info_handled = FALSE;
      dspmpeg4->video->write_pending = FALSE;
      dspmpeg4->video->frame_by_frame = TRUE;
      dspmpeg4->video->arm_timestamps = TRUE;
      dspmpeg4->discont_sent = TRUE;

      if(!gst_dspmpeg4_initialize(dspmpeg4)) {
        DBG_PRINT("Cannot initialize DSP!\n");
        dspmpeg4->video->mode = DSP_MODE_ERROR;
        return GST_STATE_CHANGE_FAILURE;
      }
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      DBG_PRINT("mpeg4: ready to pause\n");
      if(dspmpeg4->video->mode == DSP_MODE_INITIALIZED) {
        gst_dspvideo_setparams(dspmpeg4->video);

/*        if(gst_dspvideo_play(dspmpeg4->video) == FALSE) {
          DBG_PRINT("Cannot START the playback\n");
          GST_ELEMENT_ERROR (dspmpeg4, STREAM, FORMAT, (NULL), (NULL));
          return GST_FLOW_ERROR;
        }*/
      }
      dspmpeg4->video->mode = DSP_MODE_PAUSED;

      // FIXME: Is it correct to set this here?
      // Should MM server set this as a property instead???
      GST_BASE_SINK(element)->sync = FALSE;
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      DBG_PRINT("mpeg4: pause to play\n");
      dspmpeg4->paused = FALSE;
      break;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    return ret;

 switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("mpeg4: play to pause\n");
      dspmpeg4->paused = TRUE;
      gst_dspvideo_pause(dspmpeg4->video);
      break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      DBG_PRINT("mpeg4: pause to ready\n");
      gst_dspvideo_endstream(dspmpeg4->video);
      break;
    case GST_STATE_CHANGE_READY_TO_NULL:
      DBG_PRINT("mpeg4: ready to null\n");
      gst_dspvideo_close_dsp(dspmpeg4->video);
      munmap( dspmpeg4->mmap_buf_addr, dspmpeg4->write_size * sizeof(unsigned char) );
      while(gst_dspvideo_check_read(dspmpeg4->video)) {
        read(dspmpeg4->video->fd, &tmp, sizeof(short int));
      }
      gst_dspvideo_close(dspmpeg4->video);
      break;
    default:
      break;
  }

  return ret;
}


/**
 *
 *
 */

static void
gst_dspmpeg4_set_property (GObject * object, guint prop_id, const GValue * value,
                           GParamSpec * pspec)
{
  GstDSPMPEG4Sink *dsp = GST_DSPMPEG4SINK (object);

  if(!gst_dspvideo_set_property(dsp->video, prop_id, value)) {
    GST_WARNING("TRYING TO SET ILLEGAL PROPERTY\n");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 *
 *
 */

static void
gst_dspmpeg4_get_property (GObject * object, guint prop_id, GValue * value,
                           GParamSpec * pspec)
{
  GstDSPMPEG4Sink *dsp = GST_DSPMPEG4SINK (object);

  if(!gst_dspvideo_get_property(dsp->video, prop_id, value)) {
    GST_WARNING("TRYING TO GET ILLEGAL PROPERTY\n");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }

}

/**
 *
 *
 */

static void
gst_dspmpeg4_finalize (GObject * object)
{
  DBG_PRINT("gst_dspmpeg4_finalize\n");
  GstDSPMPEG4Sink *dspmpeg4 = GST_DSPMPEG4SINK (object);
  if ( dspmpeg4->video ){
    gst_dspvideo_destroy(dspmpeg4->video);
    dspmpeg4->video = NULL;
  }
  while ( !g_queue_is_empty(dspmpeg4->fbf_queue) ){
    gst_buffer_unref(g_queue_pop_head(dspmpeg4->fbf_queue));
  }
  g_queue_free(dspmpeg4->fbf_queue);
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

/**
 *
 *
 */

static gboolean
plugin_init (GstPlugin * plugin)
{

  return gst_element_register (plugin, "dspmpeg4sink", GST_RANK_PRIMARY,
      GST_TYPE_DSPMPEG4SINK);
}

/**
 *
 *
 */

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "dspmpeg4sink",
    "DSP MPEG4 video sink",
    plugin_init, VERSION, "LGPL", "dspmpeg4sink", "Nokia Corporation");
