/************************************************************
*
*	CyberLink for C
*
*	Copyright (C) Satoshi Konno 2005
*
*	File: clock_main.c
*
*	Revision:
*       05/11/05
*               - first release.
*
************************************************************/

#include <cybergarage/upnp/cupnp.h>

#include <termios.h>
#include <unistd.h>
#include <argp.h>
#include <cybergarage/util/clog.h>
#include "clock_device.h"

#if defined(TENGINE)
#if defined(PROCESS_BASE) /* Process based */
#include <basic.h>
#define MBEG	int main( W ac, TC *av[] )
#define MEND	0
#define DIR	"/SYS/bin/"
#else /* T-Kernel based */
#include <basic.h>
#include <tk/tkernel.h>
#include <sys/pinfo.h>
#define MBEG	ER main( INT ac, UB *av[] )
#define MEND	E_SYS
#define DIR	"/SYS/bin/"
#endif
#endif

const char *argp_program_version = "Eventapp (upnp clock fork)";
const char *argp_program_bug_address = "<mikael.saarenpaa@invalid>";

struct arguments
{
        int cache_control;
};

static struct argp_option options[] =
{
        { "log-defaults", 'l', 0, 0, "Add default outputs: -e stderr -w stderr-i stdout -d stdout" },
        { "log-error", 'e', "stdout|stderr|FILE", 0, "Add error output" },
        { "log-warning", 'w', "stdout|stderr|FILE", 0, "Add warning output" },
        { "log-info", 'i', "stdout|stderr|FILE", 0, "Add info output" },
        { "log-debug", 'd', "stdout|stderr|FILE", 0, "Add all debug output" },
        { "log-debug-1", 'f', "stdout|stderr|FILE", 0, "Add level 1 (basic) debug output" },
        { "log-debug-2", 'g', "stdout|stderr|FILE", 0, "Add level 2 (stack) debug output" },
        { "log-debug-3", 'h', "stdout|stderr|FILE", 0, "Add level 3 (testing) debug output" },
        { "log-debug-4", 'j', "stdout|stderr|FILE", 0, "Add level 4 (low level) debug output" },
        { "log-debug-5", 'k', "stdout|stderr|FILE", 0, "Add level 5 (low level) debug output" },
	{ "cache-control", 'c', "INTEGER", 0, "Set cache control value" },
        {0}
};

static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *arguments = state->input;

        switch (key)
        {
                case 'l':
                        cg_log_add_target("stdout", ( SEV_INFO | SEV_DEBUG_L1 | SEV_DEBUG_L2 | SEV_DEBUG_L3 ) );
                        cg_log_add_target("stderr", ( SEV_ERROR | SEV_WARNING ) );
                        break;
                case 'e':
                        cg_log_add_target(arg, SEV_ERROR );
                        break;
                case 'w':
                        cg_log_add_target(arg, SEV_WARNING );
                        break;
                case 'i':
                        cg_log_add_target(arg, SEV_INFO );
                        break;
                case 'd':
                        cg_log_add_target(arg, ( SEV_DEBUG_L1 | SEV_DEBUG_L2 |SEV_DEBUG_L3 ) );
                        break;
                case 'f':
                        cg_log_add_target(arg, SEV_DEBUG_L1 );
                        break;
                case 'g':
                        cg_log_add_target(arg, SEV_DEBUG_L2 );
                        break;
                case 'h':
                        cg_log_add_target(arg, SEV_DEBUG_L3 );
                        break;
                case 'j':
                        cg_log_add_target(arg, SEV_DEBUG_L4 );
                        break;
                case 'k':
                        cg_log_add_target(arg, SEV_DEBUG_L5 );
                        break;
		case 'c':
			arguments->cache_control = atoi(arg);
			break;
                default:
                        return ARGP_ERR_UNKNOWN;
        }

        return 0;
}

static char args_doc[] = "";
static char doc[] = "eventapp -- A program to test ClinkC Upnp stack.";

static struct argp argp = {options, parse_opt, args_doc, doc};

#define USAGE_STRING "Press:\v\t\
			Space\tto generate an event,\n\t\
			m\tto specify messages size,\n\t\
			e\tto specify event count to execute in bysy loop,\n\t\
			i\tto indicate IP address change to stack,\n\t\
			q\tto quit.\n"

CgUpnpDevice *clockDev;

////////////////////////////////////////////////////////// 
//  InitApp
////////////////////////////////////////////////////////// 

BOOL InitApp( int cacheControl )
{
	clockDev = upnp_clock_device_new();
        if ( clockDev == NULL ) {
                printf( "upnp_clock_device_new() failed\n" );
                return FALSE;
        }
        cg_upnp_device_setleasetime( clockDev, cacheControl );
	return cg_upnp_device_start(clockDev);
}

void ExitApp()
{
	cg_upnp_device_stop(clockDev);
	cg_upnp_device_delete(clockDev);
}

////////////////////////////////////////////////////////// 
//  main
////////////////////////////////////////////////////////// 

#if defined(TENGINE)
MBEG
#else
int main( int argc, char* argv[] )
#endif
{
	struct termios old_tio, new_tio;
        int m_length, event_count, stop = 0;
	struct arguments arguments;

	arguments.cache_control = 1800;

	argp_parse(&argp, argc, argv, 0, 0, &arguments);	

        printf( "Using CACHE-CONTROL value: %d sec\n", arguments.cache_control );

	if (InitApp( arguments.cache_control ) == FALSE) {
		fprintf(stderr, "Couldn't create this clock device !!\n");
		exit(0);
	}
				
	int ch = 0;

	tcgetattr(STDIN_FILENO ,&old_tio);
	new_tio=old_tio;
	new_tio.c_lflag &=(~ICANON & ~ECHO);
	tcsetattr(STDIN_FILENO ,TCSANOW,&new_tio);

	puts(USAGE_STRING);	

        /* upnp_clock_device_update(clockDev); */

	while (!stop)
	{
        	ch = getchar();

		switch (ch)
		{
			case 'q':
				stop = 1;
				break;
				
			case 'm':
				printf("Give message length (Use -1 for default (%d)): ", MAX_MESSAGE_LENGTH);
				
				tcsetattr(STDIN_FILENO ,TCSANOW,&old_tio);
				if ( 1 == scanf("%i", &m_length) )
				{
					if ( -1 == m_length ) m_length = MAX_MESSAGE_LENGTH;
					upnp_clock_device_update_max( clockDev, m_length );
					tcsetattr(STDIN_FILENO ,TCSANOW,&new_tio);

					break;
				}
				tcsetattr(STDIN_FILENO ,TCSANOW,&new_tio);				

				puts("Parse error, using default message length");
				upnp_clock_device_update_max( clockDev, MAX_MESSAGE_LENGTH );
				break;
				
			case 'e':
				printf("Give event count to send in busy loop: ", MAX_MESSAGE_LENGTH);
				
				tcsetattr(STDIN_FILENO ,TCSANOW,&old_tio);
				if ( 1 == scanf("%i", &event_count) )
				{
					int i;
					struct timeval start_time, end_time, elapsed_time;
					
					puts("Updating clock device:");
					gettimeofday(&start_time, NULL);
					for (i=0; i<event_count; i++) {
						upnp_clock_device_update( clockDev );
					}
					gettimeofday(&end_time, NULL);
					elapsed_time.tv_usec = 
						(start_time.tv_usec < end_time.tv_usec)?
						end_time.tv_usec-start_time.tv_usec:
						end_time.tv_sec--, (10000+end_time.tv_usec) - start_time.tv_usec;
					elapsed_time.tv_sec = end_time.tv_sec - start_time.tv_sec;
					
					printf("Done in %d seconds and %d micro seconds.\n", 
							elapsed_time.tv_sec,
							elapsed_time.tv_usec );
					
					tcsetattr(STDIN_FILENO ,TCSANOW,&new_tio);
										
					break;
				}
				tcsetattr(STDIN_FILENO ,TCSANOW,&new_tio);				

				puts("Parse error, no updates this time...");
				
				break;

			case 'i':
                                printf("Indicating IP address change to stack...\n");
				cg_upnp_device_ipchanged(clockDev);
                                printf("Done (Indicating IP address change to stack.)\n");
				break;

			case 32:
				upnp_clock_device_update(clockDev);
				break;

			case 10:
				/* Just bypassing new lines... */
				break;
			
			default:
				puts( "Unknown command!" );
				puts( USAGE_STRING );
		}
	}

	tcsetattr(STDIN_FILENO ,TCSANOW,&old_tio);
	ExitApp();
	return(0);
}
