/******************************************************************
*
*	CyberUtil for C
*
*	Copyright (C) Satoshi Konno 2005
*
*       Copyright (C) 2006 Nokia Corporation. All rights reserved.
*
*       This is licensed under BSD-style license with patent exclusion,
*       see file COPYING.
*
*	File: cthread.cpp
*
*	Revision:
*
*	01/17/05
*		- first revision
*	09/11/05
*		- Thanks for Visa Smolander <visa.smolander@nokia.com>
*		- Changed cg_thread_start() to set the flag at first.
*	10/31/05
*		- Added a signal handler to block all signals to posix threads
*
******************************************************************/

#include <signal.h>
#include <cybergarage/util/cthread.h>
#include <cybergarage/util/clog.h>
#include <string.h>

/* Private function prototypes */
static void sig_handler(int sign);

/****************************************
* Thread Function
****************************************/

#if defined(WIN32) && !defined(ITRON)
static DWORD WINAPI Win32ThreadProc(LPVOID lpParam)
{
	cg_log_debug_l4("Entering...\n");

	CgThread *thread = (CgThread *)lpParam;
	if (thread->action != NULL)
		thread->action(thread);
	return 0;

	cg_log_debug_l4("Leaving...\n");
}
#elif defined(BTRON)
static VOID BTronTaskProc(W param)
{
	cg_log_debug_l4("Entering...\n");

	CgThread *thread = (CgThread *)param;
	if (thread->action != NULL)
		thread->action(thread);
	ext_tsk();

	cg_log_debug_l4("Leaving...\n");
}
#elif defined(ITRON)
static TASK ITronTaskProc(int param)
{
	cg_log_debug_l4("Entering...\n");

	T_RTSK rtsk;
	CgThread *thread;
	if (ref_tsk(TSK_SELF, &rtsk) != E_OK)
		return;
	thread = (CgThread *)rtsk.exinf;
	if (thread->action != NULL)
		thread->action(thread);
	exd_tsk();

	cg_log_debug_l4("Leaving...\n");
}
#elif defined(TENGINE) && !defined(PROCESS_BASE)
static VOID TEngineTaskProc(INT stacd, VP param)
{
	cg_log_debug_l4("Entering...\n");

	CgThread *thread = (CgThread *)param;
	if (thread->action != NULL)
		thread->action(thread);
	tk_exd_tsk();

	cg_log_debug_l4("Leaving...\n");
}
#elif defined(TENGINE) && defined(PROCESS_BASE)
static VOID TEngineProcessBasedTaskProc(W param)
{
	cg_log_debug_l4("Entering...\n");

	CgThread *thread = (CgThread *)param;
	if (thread->action != NULL)
		thread->action(thread);
	b_ext_tsk();

	cg_log_debug_l4("Leaving...\n");
}
#else

static void *PosixThreadProc(void *param)
{
	cg_log_debug_l4("Entering...\n");

	sigset_t set;
	struct sigaction actions;
	CgThread *thread = (CgThread *)param;

	/* SIGINT is used in thread deletion routine
	 * to force accept and recvmsg to return during thread
	 * termination process. */
	sigfillset(&set);
	sigdelset(&set, SIGQUIT);
	pthread_sigmask(SIG_SETMASK, &set, NULL);
	
	memset(&actions, 0, sizeof(actions));
	sigemptyset(&actions.sa_mask);
	actions.sa_flags = 0;
	actions.sa_handler = sig_handler;
	sigaction(SIGQUIT, &actions, NULL);

	if (thread->action != NULL) 
		thread->action(thread);
	
	return 0;

	cg_log_debug_l4("Leaving...\n");
}
#endif

/****************************************
* cg_thread_new
****************************************/

CgThread *cg_thread_new()
{
	cg_log_debug_l4("Entering...\n");

	CgThread *thread = (CgThread *)malloc(sizeof(CgThread));

	if ( NULL != thread )
	{
		cg_list_node_init((CgList *)thread);
		
		thread->runnableFlag = FALSE;
		thread->action = NULL;
		thread->userData = NULL;
	}

	return thread;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_delete
****************************************/

BOOL cg_thread_delete(CgThread *thread)
{
	cg_log_debug_l4("Entering...\n");

	if (thread->runnableFlag == TRUE) 
		cg_thread_stop(thread);

	cg_thread_remove(thread);
	
	free(thread);
	return TRUE;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_start
****************************************/

BOOL cg_thread_start(CgThread *thread)
{
	cg_log_debug_l4("Entering...\n");

	/**** Thanks for Visa Smolander (09/11/2005) ****/
	thread->runnableFlag = TRUE;
	
#if defined(WIN32) && !defined(ITRON)
	thread->hThread = CreateThread(NULL, 0, Win32ThreadProc, (LPVOID)thread, 0, &thread->threadID);
#elif defined(BTRON)
	P_STATE pstate;
	prc_sts(0, &pstate, NULL);
	thread->taskID = cre_tsk(BTronTaskProc, pstate.priority, (W)thread);
	if (thread->taskID < 0) {
		thread->runnableFlag = FALSE;
		return FALSE;
	}
#elif defined(ITRON)
	T_CTSK ctsk = {TA_HLNG,  (VP_INT)thread, ITronTaskProc, 6, 512, NULL, NULL};
	thread->taskID = acre_tsk(&ctsk);
	if (thread->taskID < 0) {
		thread->runnableFlag = FALSE;
		return FALSE;
	}
	if (sta_tsk(thread->taskID, 0) != E_OK) {
		thread->runnableFlag = FALSE;
		del_tsk(thread->taskID);
		return FALSE;
	}
#elif defined(TENGINE) && !defined(PROCESS_BASE)
	T_CTSK ctsk = {(VP)thread, TA_HLNG, TEngineTaskProc,10, 2048};
	thread->taskID = tk_cre_tsk(&ctsk);
	if (thread->taskID < E_OK) {
		thread->runnableFlag = FALSE;
		return FALSE;
	}
	if (tk_sta_tsk(thread->taskID, 0) < E_OK) {
		thread->runnableFlag = FALSE;
		tk_del_tsk(thread->taskID);
		return FALSE;
	}
#elif defined(TENGINE) && defined(PROCESS_BASE)
	P_STATE pstate;
	b_prc_sts(0, &pstate, NULL);
	thread->taskID = b_cre_tsk(TEngineProcessBasedTaskProc, pstate.priority, (W)thread);
	if (thread->taskID < 0) {
		thread->runnableFlag = FALSE;
		return FALSE;
	}
#else
	pthread_attr_t thread_attr;
	pthread_attr_init(&thread_attr);
	pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_JOINABLE);
	if (pthread_create(&thread->pThread, &thread_attr, PosixThreadProc, thread) != 0) {
		thread->runnableFlag = FALSE;
		pthread_attr_destroy(&thread_attr);
		return FALSE;
	}
	pthread_attr_destroy(&thread_attr);
#endif

	return TRUE;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_stop
****************************************/

BOOL cg_thread_stop(CgThread *thread)
{
	return cg_thread_stop_with_cond(thread, NULL);
}

BOOL cg_thread_stop_with_cond(CgThread *thread, CgCond *cond)
{
	cg_log_debug_l4("Entering...\n");

	if (thread->runnableFlag == TRUE) {
		thread->runnableFlag = FALSE;
		if (cond != NULL) {
			cg_cond_signal(cond);
		}
#if defined(WIN32) && !defined(ITRON)
		TerminateThread(thread->hThread, 0);
		WaitForSingleObject(thread->hThread, INFINITE);
#elif defined(BTRON)
		ter_tsk(thread->taskID);
#elif defined(ITRON)
		ter_tsk(thread->taskID);
		del_tsk(thread->taskID);
#elif defined(TENGINE) && !defined(PROCESS_BASE)
		tk_ter_tsk(thread->taskID);
		tk_del_tsk(thread->taskID);
#elif defined(TENGINE) && defined(PROCESS_BASE)
		b_ter_tsk(thread->taskID);
#else
		pthread_kill(thread->pThread, SIGQUIT);
		pthread_join(thread->pThread, NULL);
#endif
	}
	return TRUE;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_restart
****************************************/

BOOL cg_thread_restart(CgThread *thread)
{
	cg_log_debug_l4("Entering...\n");

	cg_thread_stop(thread);
	cg_thread_start(thread);
	return TRUE;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_isrunnable
****************************************/

BOOL cg_thread_isrunnable(CgThread *thread)
{
	cg_log_debug_l4("Entering...\n");

	return thread->runnableFlag;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_setaction
****************************************/

void cg_thread_setaction(CgThread *thread, CG_THREAD_FUNC func)
{
	cg_log_debug_l4("Entering...\n");

	thread->action = func;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_setuserdata
****************************************/

void cg_thread_setuserdata(CgThread *thread, void *value)
{
	cg_log_debug_l4("Entering...\n");
	
	thread->userData = value;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_thread_getuserdata
****************************************/

void *cg_thread_getuserdata(CgThread *thread)
{
	cg_log_debug_l4("Entering...\n");

	return thread->userData;

	cg_log_debug_l4("Leaving...\n");
}

/* Private helper functions */

static void sig_handler(int sign)
{
	cg_log_debug_s("Got signal %d.\n", sign);
	return;
}
