/**
 * @file libgalago/galago-value.h Value wrapper API
 * @ingroup Utility
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_VALUE_H_
#define _GALAGO_VALUE_H_

#include <libgalago/galago-object.h>

typedef struct _GalagoValue GalagoValue;

/**
 * Specific value types.
 */
typedef enum
{
	GALAGO_VALUE_TYPE_UNKNOWN = 0,  /**< Unknown type.                     */
	GALAGO_VALUE_TYPE_CHAR,         /**< Character.                        */
	GALAGO_VALUE_TYPE_UCHAR,        /**< Unsigned character.               */
	GALAGO_VALUE_TYPE_BOOLEAN,      /**< Boolean.                          */
	GALAGO_VALUE_TYPE_SHORT,        /**< Short integer.                    */
	GALAGO_VALUE_TYPE_USHORT,       /**< Unsigned short integer.           */
	GALAGO_VALUE_TYPE_INT,          /**< Integer.                          */
	GALAGO_VALUE_TYPE_UINT,         /**< Unsigned integer.                 */
	GALAGO_VALUE_TYPE_LONG,         /**< Long integer.                     */
	GALAGO_VALUE_TYPE_ULONG,        /**< Unsigned long integer.            */
	GALAGO_VALUE_TYPE_STRING,       /**< String.                           */
	GALAGO_VALUE_TYPE_OBJECT,       /**< Object pointer.                   */
	GALAGO_VALUE_TYPE_POINTER,      /**< Generic pointer.                  */
	GALAGO_VALUE_TYPE_LIST,         /**< A list of values.                 */
	GALAGO_VALUE_TYPE_ARRAY         /**< An array of values.               */

} GalagoType;

G_BEGIN_DECLS

/**************************************************************************/
/** @name Value API                                                       */
/**************************************************************************/
/*@{*/

/**
 * Creates a new GalagoValue.
 *
 * @param type   The type.
 * @param data   The optional data to set.
 * @param detail Extra detail, type-specific. Currently unused.
 *
 * @return The new value.
 */
GalagoValue *galago_value_new(GalagoType type, const void *data, void *detail);

/**
 * Creates a new, special GalagoValue that takes an object type and an
 * optional object.
 *
 * @param type   The GType.
 * @param obj    The object.
 *
 * @return The new value.
 */
GalagoValue *galago_value_new_object(GType type, const GObject *obj);

/**
 * Creates a new, special GalagoValue that takes only a list of values
 * of the specified type.
 *
 * If @a type is GALAGO_VALUE_TYPE_OBJECT, @a detail must be the class type.
 *
 * @param type   The type of list values.
 * @param list   The optional default list of values.
 * @param detail Extra detail, type-specific.
 *
 * @return The new value.
 */
GalagoValue *galago_value_new_list(GalagoType type, GList *list,
								   void *detail);

/**
 * Creates a new, special GalagoValue that takes only an array of values
 * of the specified type.
 *
 * If @a type is GALAGO_VALUE_TYPE_OBJECT, @a detail must be the class type.
 *
 * @param type   The type of list values.
 * @param array  The optional default array of values.
 * @param size   The size of the array.
 * @param detail Extra detail, type-specific.
 *
 * @return The new value.
 */
GalagoValue *galago_value_new_array(GalagoType type, const void *array,
									gsize size, void *detail);

/**
 * Destroys a GalagoValue.
 *
 * @param value The value to destroy.
 */
void galago_value_destroy(GalagoValue *value);

/**
 * Returns a value's type.
 *
 * @return The value's type.
 */
GalagoType galago_value_get_type(const GalagoValue *value);

/**
 * Returns a value's subtype.
 *
 * This is only usable for lists and arrays.
 *
 * @return The value's list type.
 */
GalagoType galago_value_get_subtype(const GalagoValue *value);

/**
 * Returns the object's GType, if the value's type is GALAGO_VALUE_TYPE_OBJECT.
 *
 * @param value The value.
 *
 * @return The object's GType, or NULL.
 */
GType galago_value_get_gtype(const GalagoValue *value);

/**
 * Sets the value's character data.
 *
 * @param value The value.
 * @param data The character data.
 */
void galago_value_set_char(GalagoValue *value, char data);

/**
 * Sets the value's unsigned character data.
 *
 * @param value The value.
 * @param data The unsigned character data.
 */
void galago_value_set_uchar(GalagoValue *value, unsigned char data);

/**
 * Sets the value's boolean data.
 *
 * @param value The value.
 * @param data The boolean data.
 */
void galago_value_set_boolean(GalagoValue *value, gboolean data);

/**
 * Sets the value's short integer data.
 *
 * @param value The value.
 * @param data The short integer data.
 */
void galago_value_set_short(GalagoValue *value, short data);

/**
 * Sets the value's unsigned short integer data.
 *
 * @param value The value.
 * @param data The unsigned short integer data.
 */
void galago_value_set_ushort(GalagoValue *value, unsigned short data);

/**
 * Sets the value's integer data.
 *
 * @param value The value.
 * @param data The integer data.
 */
void galago_value_set_int(GalagoValue *value, int data);

/**
 * Sets the value's unsigned integer data.
 *
 * @param value The value.
 * @param data The unsigned integer data.
 */
void galago_value_set_uint(GalagoValue *value, unsigned int data);

/**
 * Sets the value's long integer data.
 *
 * @param value The value.
 * @param data The long integer data.
 */
void galago_value_set_long(GalagoValue *value, long data);

/**
 * Sets the value's unsigned long integer data.
 *
 * @param value The value.
 * @param data The unsigned long integer data.
 */
void galago_value_set_ulong(GalagoValue *value, unsigned long data);

/**
 * Sets the value's string data.
 *
 * @param value The value.
 * @param data The string data.
 */
void galago_value_set_string(GalagoValue *value, const char *data);

/**
 * Sets the value's object data.
 *
 * @param value The value.
 * @param data The object data.
 */
void galago_value_set_object(GalagoValue *value, void *data);

/**
 * Sets the value's pointer data.
 *
 * @param value The value.
 * @param data The pointer data.
 */
void galago_value_set_pointer(GalagoValue *value, void *data);

/**
 * Sets the value's list of values.
 *
 * @param value The value.
 * @param data The list of values.
 */
void galago_value_set_list(GalagoValue *value, GList *data);

/**
 * Sets the value's array of values.
 *
 * @param value The value.
 * @param data  The array of values.
 * @param size  The size of the array.
 */
void galago_value_set_array(GalagoValue *value, const void *data,
							gsize size);

/**
 * Returns the value's character data.
 *
 * @param value The value.
 *
 * @return The character data.
 */
char galago_value_get_char(const GalagoValue *value);

/**
 * Returns the value's unsigned character data.
 *
 * @param value The value.
 *
 * @return The unsigned character data.
 */
unsigned char galago_value_get_uchar(const GalagoValue *value);

/**
 * Returns the value's boolean data.
 *
 * @param value The value.
 *
 * @return The boolean data.
 */
gboolean galago_value_get_boolean(const GalagoValue *value);

/**
 * Returns the value's short integer data.
 *
 * @param value The value.
 *
 * @return The short integer data.
 */
short galago_value_get_short(const GalagoValue *value);

/**
 * Returns the value's unsigned short integer data.
 *
 * @param value The value.
 *
 * @return The unsigned short integer data.
 */
unsigned short galago_value_get_ushort(const GalagoValue *value);

/**
 * Returns the value's integer data.
 *
 * @param value The value.
 *
 * @return The integer data.
 */
int galago_value_get_int(const GalagoValue *value);

/**
 * Returns the value's unsigned integer data.
 *
 * @param value The value.
 *
 * @return The unsigned integer data.
 */
unsigned int galago_value_get_uint(const GalagoValue *value);

/**
 * Returns the value's long integer data.
 *
 * @param value The value.
 *
 * @return The long integer data.
 */
long galago_value_get_long(const GalagoValue *value);

/**
 * Returns the value's unsigned long integer data.
 *
 * @param value The value.
 *
 * @return The unsigned long integer data.
 */
unsigned long galago_value_get_ulong(const GalagoValue *value);

/**
 * Returns the value's string data.
 *
 * @param value The value.
 *
 * @return The string data.
 */
const char *galago_value_get_string(const GalagoValue *value);

/**
 * Returns the value's object data.
 *
 * @param value The value.
 *
 * @return The object data.
 */
void *galago_value_get_object(const GalagoValue *value);

/**
 * Returns the value's pointer data.
 *
 * @param value The value.
 *
 * @return The pointer data.
 */
void *galago_value_get_pointer(const GalagoValue *value);

/**
 * Returns the value's list of values.
 *
 * @param value The value.
 *
 * @return The list of values.
 */
GList *galago_value_get_list(const GalagoValue *value);

/**
 * Returns the value's array of values.
 *
 * @param value     The value.
 * @param ret_array The returned array.
 * @param ret_size  The returned size of the array.
 */
void galago_value_get_array(const GalagoValue *value,
							const void **ret_array,
							gsize *ret_size);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_VALUE_H_ */
