/**
 * @file libgalago/galago-status.h Galago Status API
 * @ingroup Presence
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_STATUS_H_
#define _GALAGO_STATUS_H_

typedef struct _GalagoStatus        GalagoStatus;
typedef struct _GalagoStatusClass   GalagoStatusClass;
typedef struct _GalagoStatusPrivate GalagoStatusPrivate;

typedef enum
{
	GALAGO_STATUS_UNSET = 0,
	GALAGO_STATUS_OFFLINE,
	GALAGO_STATUS_AVAILABLE,
	GALAGO_STATUS_AWAY,
	GALAGO_STATUS_EXTENDED_AWAY,
	GALAGO_STATUS_HIDDEN

} GalagoStatusType;

#include <libgalago/galago-presence.h>
#include <libgalago/galago-object.h>

/**************************************************************************/
/** @name Common Status Definitions                                       */
/**************************************************************************/
/*@{*/

#define GALAGO_STATUS_ID_AVAILABLE       "available"
#define GALAGO_STATUS_ID_AWAY            "away"
#define GALAGO_STATUS_ID_BRB             "brb"
#define GALAGO_STATUS_ID_BUSY            "busy"
#define GALAGO_STATUS_ID_DND             "dnd"
#define GALAGO_STATUS_ID_EXTENDED_AWAY   "xa"
#define GALAGO_STATUS_ID_HIDDEN          "hidden"
#define GALAGO_STATUS_ID_OFFLINE         "offline"

/*@}*/

/**************************************************************************/
/** @name Common Status Attributes                                        */
/**************************************************************************/
/*@{*/

/**
 * A message attribute, typically used for custom away and available messages.
 */
#define GALAGO_STATUS_ATTR_MESSAGE "message"

/*@}*/

/**************************************************************************/
/** @name Status API                                                      */
/**************************************************************************/
/*@{*/

/**
 * A status in a presence.
 */
struct _GalagoStatus
{
	GalagoObject parent_object;

	GalagoStatusPrivate *priv;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * GalagoStatus's class.
 */
struct _GalagoStatusClass
{
	GalagoObjectClass parent_class;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

#define GALAGO_TYPE_STATUS (galago_status_get_type())

/** Cast @c obj to a GalagoStatus. */
#define GALAGO_STATUS(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_STATUS, \
									GalagoStatus))

/** Cast @c klass to a GalagoStatus. */
#define GALAGO_STATUS_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_STATUS, \
								 GalagoStatusClass))

/** Check if @c obj is a GalagoStatus. */
#define GALAGO_IS_STATUS(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_STATUS))

/** Check if @c klass is a GalagoStatusClass. */
#define GALAGO_IS_STATUS_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_STATUS))

/** Get the class for the GalagoStatus @c obj. */
#define GALAGO_STATUS_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_STATUS, \
								   GalagoStatusClass))

G_BEGIN_DECLS

/**************************************************************************/
/** @name Status API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoStatus's GType.
 *
 * @return The 's GType.
 */
GType galago_status_get_type(void);

/**
 * Creates a new status.
 *
 * If @a exclusive is TRUE, the status will be exclusive. Only one
 * exclusive status can be set at a time. If another exclusive status
 * is set, the previously set exclusive status will be removed.
 *
 * If @a exclusive is FALSE, the status will not be removed unless
 * manually removed.
 *
 * @param type      The type of status.
 * @param id        The status ID.
 * @param name      The name of the status.
 * @param exclusive TRUE if the status is exclusive.
 *
 * @return The status.
 */
GalagoStatus *galago_status_new(GalagoStatusType type, const char *id,
								const char *name, gboolean exclusive);

/**
 * Duplicates a status.
 *
 * @param status The status to duplicate.
 *
 * @return The duplicate status.
 */
GalagoStatus *galago_status_duplicate(const GalagoStatus *status);

/**
 * Sets the parent presence of a status.
 *
 * @internal This is used internally.
 *
 * @param status   The status.
 * @param presence The presence.
 */
void galago_status_set_presence(GalagoStatus *status,
								GalagoPresence *presence);

/**
 * Returns the parent presence of a status.
 *
 * @param status The status.
 *
 * @return The presence.
 */
GalagoPresence *galago_status_get_presence(const GalagoStatus *status);

/**
 * Returns the primitive type of a status.
 *
 * @param status The status.
 *
 * @return The status's primitive type.
 */
GalagoStatusType galago_status_get_primitive(const GalagoStatus *status);

/**
 * Returns the status's ID.
 *
 * @param status The status.
 *
 * @return The status's ID.
 */
const char *galago_status_get_id(const GalagoStatus *status);

/**
 * Returns the status's name.
 *
 * @param status The status.
 *
 * @return The status's name.
 */
const char *galago_status_get_name(const GalagoStatus *status);

/**
 * Returns whether or not a status is exclusive.
 *
 * @param status The status.
 *
 * @return TRUE if the status is exclusive, or FALSE.
 */
gboolean galago_status_is_exclusive(const GalagoStatus *status);

/**
 * Returns whether or not a status is considered available.
 *
 * @param status The status.
 *
 * @return TRUE if the status is available, or FALSE.
 */
gboolean galago_status_is_available(const GalagoStatus *status);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_STATUS_H_ */
