/**
 * @file libgalago/galago-service.h Galago Service API
 * @ingroup Presence
 *
 * A #GalagoService represents an online service to which users can
 * connect that keeps track of the user's presence. Examples are instant
 * messaging services (such as Jabber) or video chat (such as SIP).
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_SERVICE_H_
#define _GALAGO_SERVICE_H_

typedef struct _GalagoService        GalagoService;
typedef struct _GalagoServiceClass   GalagoServiceClass;
typedef struct _GalagoServicePrivate GalagoServicePrivate;

/*
 * Service flags
 */
typedef enum
{
	/** Preserve spaces during normalization */
	GALAGO_PRESERVE_SPACES = 1 << 0,

	/** Preserve case during normalization */
	GALAGO_PRESERVE_CASE   = 1 << 1,

	/** Strip a slash and everything after it during normalization. */
	GALAGO_STRIP_SLASH     = 1 << 2
} GalagoServiceFlags;

#include <libgalago/galago-account.h>
#include <libgalago/galago-object.h>

/**
 * A service capable of having accounts and presences.
 */
struct _GalagoService
{
	GalagoObject parent_object;

	GalagoServicePrivate *priv;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * A service's class.
 */
struct _GalagoServiceClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*account_added)(GalagoService *service, GalagoAccount *account);
	void (*account_removed)(GalagoService *service, GalagoAccount *account);

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

#define GALAGO_TYPE_SERVICE (galago_service_get_type())

/** Cast @c obj to a GalagoService. */
#define GALAGO_SERVICE(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_SERVICE, \
									GalagoService))

/** Cast @c klass to a GalagoService. */
#define GALAGO_SERVICE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_SERVICE, \
								 GalagoServiceClass))

/** Check if @c obj is a GalagoService. */
#define GALAGO_IS_SERVICE(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_SERVICE))

/** Check if @c klass is a GalagoServiceClass. */
#define GALAGO_IS_SERVICE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_SERVICE))

/** Get the class for the GalagoService @c obj. */
#define GALAGO_SERVICE_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_SERVICE, \
								   GalagoServiceClass))

#define GALAGO_DBUS_SERVICE_INTERFACE "org.freedesktop.Galago.Service"

G_BEGIN_DECLS

/**************************************************************************/
/** @name Common Service Definitions                                      */
/**************************************************************************/
/*@{*/

#define GALAGO_SERVICE_ID_AIM          "aim"
#define GALAGO_SERVICE_ID_GADUGADU     "gadugadu"
#define GALAGO_SERVICE_ID_GROUPWISE    "groupwise"
#define GALAGO_SERVICE_ID_ICQ          "icq"
#define GALAGO_SERVICE_ID_IRC          "irc"
#define GALAGO_SERVICE_ID_JABBER       "jabber"
#define GALAGO_SERVICE_ID_MSN          "msn"
#define GALAGO_SERVICE_ID_NAPSTER      "napster"
#define GALAGO_SERVICE_ID_SILC         "silc"
#define GALAGO_SERVICE_ID_TREPIA       "trepia"
#define GALAGO_SERVICE_ID_YAHOO        "yahoo"
#define GALAGO_SERVICE_ID_ZEPHYR       "zephyr"

/*@}*/

/**************************************************************************/
/** @name Service API                                                     */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoService's GType.
 *
 * @return The GType.
 */
GType galago_service_get_type(void);

/**
 * Returns a service's ID.
 *
 * @param service The service.
 *
 * @return The service's ID.
 */
const char *galago_service_get_id(const GalagoService *service);

/**
 * Returns a service's name.
 *
 * @param service The service.
 *
 * @return The service's name.
 */
const char *galago_service_get_name(const GalagoService *service);

/**
 * Returns a service's flags.
 *
 * @param service The service.
 *
 * @return The flags.
 */
GalagoServiceFlags galago_service_get_flags(const GalagoService *service);

/**
 * Creates an account belonging to this service.
 *
 * If the account already exists, the existing account will be returned.
 *
 * @param service  The service.
 * @param person   The person the account belongs to.
 * @param username The account username.
 *
 * @return The account.
 */
GalagoAccount *galago_service_create_account(GalagoService *service,
											 GalagoPerson *person,
											 const char *username);

/**
 * Returns the account with the specified username from a service.
 *
 * @param service  The service.
 * @param username The account's username.
 * @param query    TRUE if a remote query should be done if there is no
 *                 local account found, or FALSE.
 *
 * @return The account, if found, or NULL.
 */
GalagoAccount *galago_service_get_account(const GalagoService *service,
										  const char *username,
										  gboolean query);

/**
 * Returns a list of all accounts in the service.
 *
 * This may emit an account-added signal for every object that returns. If
 * your code connects to this signal and calls galago_service_get_accounts()
 * as a result, you will want to add a lock so that you don't end up with
 * unwanted side-effects.
 *
 * @param service The service.
 * @param query   TRUE if a remote query should be done if there is no
 *                local account found, or FALSE.
 *
 * @return A list of all accounts in the service.
 */
GList *galago_service_get_accounts(const GalagoService *service,
								   gboolean query);

/**
 * Normalizes an account username based on the service's normalization
 * flags.
 *
 * @param service  The service.
 * @param username The username to normalize.
 *
 * @return The string. This must be freed.
 */
char *galago_service_normalize(const GalagoService *service,
							   const char *username);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_SERVICE_H_ */
