/**
 * @file libgalago/galago-object.h Galago Base Object
 * @ingroup Utility
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_OBJECT_H_
#define _GALAGO_OBJECT_H_

#include <glib.h>
#include <glib-object.h>

#ifndef _DOXYGEN_
#define DBUS_API_SUBJECT_TO_CHANGE
#endif
#include <dbus/dbus.h>

typedef struct _GalagoObjectClass   GalagoObjectClass;
typedef struct _GalagoObject        GalagoObject;
typedef struct _GalagoObjectPrivate GalagoObjectPrivate;

/**
 * An object.
 */
struct _GalagoObject
{
	GObject parent_object;
	guint32 flags;

	GalagoObjectPrivate *priv;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * A class.
 */
struct _GalagoObjectClass
{
	GObjectClass parent_class;

	const char *dbus_interface;
	const char *dbus_signature;
	gboolean supports_attrs;

	/* Virtual functions */
	void (*dbus_message_append)(DBusMessageIter *iter,
								const GalagoObject *object);
	void *(*dbus_message_get)(DBusMessageIter *iter);
	void (*dbus_push_full)(GalagoObject *object);
	gchar *(*dbus_get_signature)(void);
	void (*set_attribute)(GalagoObject *object, const char *name,
						  GValue *value);
	gboolean (*remove_attribute)(GalagoObject *object, const char *name);
	const GValue *(*get_attribute)(const GalagoObject *object,
								   const char *name);
	GList *(*get_attributes)(const GalagoObject *object);

	/* Signals */
	void (*destroy)(GalagoObject *object);

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

#define GALAGO_TYPE_OBJECT (galago_object_get_type())

/** Cast @c obj to a GalagoObject. */
#define GALAGO_OBJECT(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_OBJECT, GalagoObject))

/** Cast @c klass to a GalagoObject. */
#define GALAGO_OBJECT_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_OBJECT, \
								 GalagoObjectClass))

/** Check if @c obj is a GalagoObject. */
#define GALAGO_IS_OBJECT(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_OBJECT))

/** Check if @c klass is a GalagoObjectClass. */
#define GALAGO_IS_OBJECT_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_OBJECT))

/** Get the class for the GalagoObject @c obj. */
#define GALAGO_OBJECT_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_OBJECT, \
								   GalagoObjectClass))

/**
 * Object flags.
 */
typedef enum
{
	/** The object is currently being destroyed */
	GALAGO_OBJECT_IN_DESTRUCTION = 1,
	GALAGO_OBJECT_RESERVED_1     = 2,
	GALAGO_OBJECT_RESERVED_2     = 4

} GalagoObjectFlags;

/** Check if @c obj has @c flag set. */
#define GALAGO_OBJECT_HAS_FLAG(obj, flag) \
	((GALAGO_OBJECT_FLAGS(obj) & (flag)) == (flag))

/** Get the flags for @c obj. */
#define GALAGO_OBJECT_FLAGS(obj) (GALAGO_OBJECT(obj)->flags)

/** Set the flags for @c obj whch are set in @c flags. */
#define GALAGO_OBJECT_SET_FLAGS(obj, flag) \
	G_STMT_START{ (GALAGO_OBJECT_FLAGS(obj) |= (flag)); } G_STMT_END

/** Unset the flags for @c obj whch are set in @c flags. */
#define GALAGO_OBJECT_UNSET_FLAGS(obj, flag) \
	G_STMT_START{ (GALAGO_OBJECT_FLAGS(obj) &= ~(flag)); } G_STMT_END


/*
 * Object origin.
 */
typedef enum
{
	/** Local object */
	GALAGO_LOCAL = 0,

	/** Remote object */
	GALAGO_REMOTE

} GalagoOrigin;

#define GALAGO_ORIGIN_IS_VALID(origin) \
	((origin) == GALAGO_LOCAL || (origin) == GALAGO_REMOTE)

#define GALAGO_OBJECT_IS_LOCAL(obj) \
	(galago_object_get_origin(GALAGO_OBJECT(obj)) == GALAGO_LOCAL)
#define GALAGO_OBJECT_IS_REMOTE(obj) \
	(galago_object_get_origin(GALAGO_OBJECT(obj)) == GALAGO_REMOTE)

#include <libgalago/galago-context-base.h>

G_BEGIN_DECLS

/**************************************************************************/
/** @name Object API                                                      */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoObject's GType.
 *
 * @return The GType.
 */
GType galago_object_get_type(void);

/**
 * Destroys an object.
 *
 * This will discard all references and emit the "destroy" signal.
 *
 * @param object The object to destroy.
 */
void galago_object_destroy(GalagoObject *object);

/**
 * Returns the D-BUS signature of the object type.
 *
 * @param type The object GType.
 *
 * @return The signature string, or NULL.
 */
const gchar *galago_object_type_get_dbus_signature(GType type);

/**
 * Sets the D-BUS object path of an object.
 *
 * @param object   The object.
 * @param obj_path The object path.
 */
void galago_object_set_dbus_path(GalagoObject *object, const gchar *obj_path);

/**
 * Returns the D-BUS object path of an object.
 *
 * @param object The object.
 *
 * @return The object path.
 */
const gchar *galago_object_get_dbus_path(const GalagoObject *object);

/**
 * Sets whether or not this object is watched for events.
 *
 * @param object The object.
 * @param watch  TRUE if this object should be watched, or FALSE.
 */
void galago_object_set_watch(GalagoObject *object, gboolean watch);

/**
 * Returns whether or not an object is watched for events.
 *
 * @param object The object.
 *
 * @return TRUE if this object is being watched, or FALSE.
 */
gboolean galago_object_is_watched(const GalagoObject *object);

/**
 * Returns the object's origin.
 *
 * @param object The object.
 *
 * @return The object's origin
 */
GalagoOrigin galago_object_get_origin(const GalagoObject *object);

/**
 * Returns the object's context.
 *
 * @param object The object.
 *
 * @return The object's context.
 */
GalagoContext *galago_object_get_context(const GalagoObject *object);

/**
 * Sets a string attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute to set.
 * @param value  The value of the attribute.
 */
void galago_object_set_attr_string(GalagoObject *object, const char *name,
								   const char *value);

/**
 * Sets a boolean attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute to set.
 * @param value  The value of the attribute.
 */
void galago_object_set_attr_bool(GalagoObject *object, const char *name,
								 gboolean value);

/**
 * Sets an integer attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute to set.
 * @param value  The value of the attribute.
 */
void galago_object_set_attr_int(GalagoObject *object, const char *name,
								gint32 value);

/**
 * Sets a double attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute to set.
 * @param value  The value of the attribute.
 */
void galago_object_set_attr_double(GalagoObject *object, const char *name,
								   gdouble value);

/**
 * Sets an attribute on an object.
 *
 * This is limited to string, boolean, and integer value types.
 *
 * @param object The object.
 * @param name   The name of the attribute to set.
 * @param value  The value of the attribute.
 */
void galago_object_set_attribute(GalagoObject *object, const char *name,
								 GValue *value);

/**
 * Removes an attribute on an object.
 *
 * @param object The object
 * @param name   The name of the attribute to remove
 *
 * @return TRUE if the attribute was removed, or FALSE.
 */
gboolean galago_object_remove_attribute(GalagoObject *object,
										const char *name);

/**
 * Returns the value of a string attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute.
 *
 * @return The attribute value, or NULL.
 */
const char *galago_object_get_attr_string(const GalagoObject *object,
										  const char *name);

/**
 * Returns the value of a boolean attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute.
 *
 * @return The attribute value.
 */
gboolean galago_object_get_attr_bool(const GalagoObject *object,
									 const char *name);

/**
 * Returns the value of an integer attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute.
 *
 * @return The attribute value.
 */
gint32 galago_object_get_attr_int(const GalagoObject *object,
								  const char *name);

/**
 * Returns the value of a double attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute.
 *
 * @return The attribute value.
 */
gdouble galago_object_get_attr_double(const GalagoObject *object,
									  const char *name);

/**
 * Returns the value of an attribute on an object.
 *
 * @param object The object.
 * @param name   The name of the attribute.
 *
 * @return The attribute value, or NULL.
 */
const GValue *galago_object_get_attribute(const GalagoObject *object,
										  const char *name);

/**
 * Returns whether or not an object has a specific attribute set.
 *
 * @param object The object.
 * @param name   The name of the attribute.
 *
 * @return TRUE if the attribute is set, or FALSE.
 */
gboolean galago_object_get_has_attribute(const GalagoObject *object,
										 const char *name);

/**
 * Returns the list of attributes in an object, represented by GalagoKeyValue
 * structs.
 *
 * @param object The object.
 *
 * @return The attributes in the object. Each one is a GalagoKeyValue.
 */
GList *galago_object_get_attributes(const GalagoObject *object);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_OBJECT_H_ */
