/**
 * @file libgalago/galago-context.h Galago Context API
 * @ingroup Presence
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CONTEXT_H_
#define _GALAGO_CONTEXT_H_

/*
 * NOTE: The base GalagoContext struct has been split out into
 *       galago-context-base.h in order to break a fairly nasty header
 *       file dependency hell problem that was caused by the fact that
 *       GalagoObject requires GalagoContext, and GalagoContext makes use of
 *       things that inherit GalagoObject. Eventually, it all just broke. This
 *       only matters inside of here, since nothing outside libgalago requests
 *       specific header files.
 */

#include <libgalago/galago-context-base.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>

G_BEGIN_DECLS

/**************************************************************************/
/** @name Galago Context API                                              */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoContext's GType.
 *
 * @return The GType.
 */
GType galago_context_get_type(void);

/**
 * Creates a new context.
 *
 * @return The new context.
 */
GalagoContext *galago_context_new(void);

/**
 * Pushes a context onto the stack.
 *
 * @param context The context to push.
 */
void galago_context_push(GalagoContext *context);

/**
 * Pops a context off the stack.
 */
void galago_context_pop(void);

/**
 * Returns the current context.
 *
 * @return The current context.
 */
GalagoContext *galago_context_get(void);

/**
 * Sets the base object path prefix for this context.
 *
 * @param prefix The object path prefix.
 */
void galago_context_set_obj_path_prefix(const char *prefix);

/**
 * Returns the base object path prefix for this context.
 *
 * @return The object path prefix.
 */
const char *galago_context_get_obj_path_prefix(void);

/**
 * Returns the service with the specified ID.
 *
 * @param id     The service ID.
 * @param origin The object's origin.
 *
 * @return The service, if found, or NULL.
 */
GalagoService *galago_context_get_service(const char *id, GalagoOrigin origin);

/**
 * Returns a list of all services.
 *
 * @param origin The object's origin.
 *
 * @return The list of services, if found, or NULL.
 */
GList *galago_context_get_services(GalagoOrigin origin);

/**
 * Returns the person with the specified ID.
 *
 * @param id     The person ID.
 * @param origin The object's origin.
 *
 * @return The person, if found, or NULL.
 */
GalagoPerson *galago_context_get_person(const char *id, GalagoOrigin origin);

/**
 * Returns the person with the specified session ID.
 *
 * @param session_id The person's session ID.
 * @param origin     The object's origin.
 *
 * @return The person, if found, or NULL.
 */
GalagoPerson *galago_context_get_person_with_session_id(const char *session_id,
														GalagoOrigin origin);

/**
 * Returns a list of all people.
 *
 * @param origin The object's origin.
 *
 * @return The list of people, if found, or NULL.
 */
GList *galago_context_get_people(GalagoOrigin origin);

/**
 * Finds an object with the specified path.
 *
 * @param path The D-BUS object path.
 *
 * @return The object if found, or NULL.
 */
GalagoObject *galago_context_get_object(const char *path);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_CONTEXT_H_ */
