/* -*- Mode: C; c-basic-offset: 8; indent-tabs-mode: t -*- */
/**
  @file ic-api.c
  
  osso-ic-oss Internet Connectivity library
  Copyright (C) 2005 Nokia Corporation

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License version 2.1 as
  published by the Free Software Foundation.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <dbus/dbus.h>
#include <gconf/gconf-client.h>

#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <pthread.h>
#include <netdb.h>
#include <sys/socket.h>
#include <errno.h>

#include <config.h>
#include <osso-ic.h>
#include <osso-ic-dbus.h>
#include <osso-ic-gconf.h>

static char *osso_iap_name = NULL;
static dbus_bool_t osso_iap_disconnect_pending = FALSE;
static dbus_bool_t osso_iap_disconnect_filtered = FALSE;
static osso_iap_cb_t osso_cb = NULL;

static const char * const domain_names[] = {
	"PF_UNSPEC", "PF_UNIX", "PF_INET", "PF_AX25", "PF_IPX",
	"PF_APPLETALK", "PF_NETROM", "PF_BRIDGE", "PF_ATMPVC",
	"PF_X25", "PF_INET6", "PF_ROSE", "PF_DECnet", "PF_NETBEUI",
	"PF_SECURITY", "PF_KEY", "PF_NETLINK", "PF_PACKET", "PF_ASH",
	"PF_ECONET", "PF_ATMSVC", "PF_SNA", "PF_IRDA", "PF_PPPOX",
	"PF_WANPIPE", "PF_BLUETOOTH"
};

static const char * const type_names[] = {
	"?", "SOCK_STREAM", "SOCK_DGRAM", "SOCK_RAW", "SOCK_RDM",
	"SOCK_SEQ_PACKET", "?", "?", "?", "?", "SOCK_PACKET"
};

#define tablesize(table) (sizeof(table)/sizeof(table[0]))

#define SAFE_LOOKUP(i,table) \
	(((i) < 0 || (i) >= tablesize(table)) ? "?" : (table[i]))

static fd_set open_sockets;
static pthread_mutex_t mutex = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;


#define DBUS_SYSTEM_BUS_DEFAULT_ADDRESS "unix:path=/var/run/dbus/system_bus_socket"

static inline void disconnected()
{
	if (osso_iap_name) {
		free(osso_iap_name);
		osso_iap_name = NULL;
		osso_iap_disconnect_pending = FALSE;
		osso_iap_disconnect_filtered = FALSE;
	}
}

static DBusHandlerResult handle_icd_message(
	DBusConnection *c,
	DBusMessage *msg,
	void *user_data)
{
	char *iap_name, *info, *state;

	if (!dbus_message_is_signal(msg, ICD_DBUS_INTERFACE, 
				    "status_changed"))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	/* Get info from the message */
	if (!dbus_message_get_args(msg, NULL,
				   DBUS_TYPE_STRING, &iap_name,
				   DBUS_TYPE_STRING, &info,
				   DBUS_TYPE_STRING, &state,
				   DBUS_TYPE_INVALID))
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

#ifdef DEBUG	
	fprintf(stderr, "Received status_changed to %s notification for IAP %s\n",
		state, iap_name);
#endif

	/* IAP died? */
	if (osso_iap_name != NULL &&
	    strcmp(iap_name, osso_iap_name) == 0 &&
	    strcmp(state, "IDLE") == 0) {
		if (osso_iap_disconnect_pending == FALSE) {
			struct iap_event_t ev;
			ev.type = OSSO_IAP_DISCONNECTED;
			ev.iap_name = iap_name;

			if (osso_cb)
				osso_cb(&ev, user_data);
			disconnected();
		} else {
			osso_iap_disconnect_filtered = TRUE;
		}
	}

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusConnection *get_connection(void *arg)
{
	static DBusConnection *connection = NULL;

	static struct DBusObjectPathVTable icd_vtable = {
		.message_function = &handle_icd_message
	};

	if (connection == NULL) {
		connection = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);

		/* We also need to listen ICd events */
		dbus_bus_add_match(connection,
				   "type='signal',"
				   "interface='" ICD_DBUS_INTERFACE "',"
				   "path='" ICD_DBUS_PATH "'",
				   NULL);

		if (dbus_connection_register_object_path(
				connection, ICD_DBUS_PATH,
				&icd_vtable, arg) == FALSE) {
			dbus_connection_disconnect(connection);
			dbus_connection_unref(connection);
			connection = NULL;
			return NULL;
		}
	}

	return connection;
}

static void report_error(DBusError *error, void *arg)
{
	struct iap_event_t ev;

	if (strcasecmp(error->name, ICD_DBUS_ERROR_INVALID_IAP) == 0)
		ev.u.error_code = OSSO_ERROR_INVALID_IAP;
	else    ev.u.error_code = OSSO_ERROR;

	ev.type = OSSO_IAP_ERROR;
	ev.iap_name = error->message;

	if (osso_cb)
		osso_cb(&ev, arg);
}

static gint send_message(DBusMessage *msg,
			 void (*handler)(DBusPendingCall *pending, void *arg),
			 void *arg, int timeout, int block)
{
	DBusConnection *conn;
	DBusPendingCall *pending;
	gint ret = OSSO_OK;

	conn = get_connection(NULL);
	if (handler != NULL) {
		if (dbus_connection_send_with_reply(conn,
						    msg, &pending, timeout)) {
			dbus_pending_call_set_notify(pending, handler, arg, NULL);
			if (block)
				dbus_pending_call_block(pending);
			dbus_pending_call_unref(pending);
		} else {
			ret = OSSO_ERROR;
		}
	} else {
		if (!dbus_connection_send(conn, msg, NULL))
			ret = OSSO_ERROR;
	}

	return ret;
}

static void connected_handler(DBusPendingCall *pending,
			      void *arg)
{
	DBusMessage *reply;
	DBusError error;
	struct iap_event_t ev;
	const char *iap_name;

	memset(&ev, 0, sizeof(ev));
	disconnected();

	dbus_error_init(&error);
	reply = dbus_pending_call_steal_reply(pending);

	if (dbus_set_error_from_message(&error, reply))
		goto error_exit;

	if (!dbus_message_get_args(reply, &error,
				   DBUS_TYPE_STRING, &iap_name,
				   DBUS_TYPE_INVALID))
		goto error_exit;

	osso_iap_name = strdup(iap_name);

	ev.type = OSSO_IAP_CONNECTED;
	ev.iap_name = osso_iap_name;
	if (osso_cb)
		osso_cb(&ev, arg);

	dbus_message_unref(reply);
	return;

error_exit:
	report_error(&error, arg);
	dbus_error_free(&error);
	dbus_message_unref(reply);
}

static void disconnected_handler(DBusPendingCall *pending,
				 void *arg)
{
	DBusMessage *reply;
	DBusError error;
	struct iap_event_t ev;

	memset(&ev, 0, sizeof(ev));
	dbus_error_init(&error);
	reply = dbus_pending_call_steal_reply(pending);

	if (dbus_set_error_from_message(&error, reply))
		goto error_exit;

	if (!dbus_message_get_args(reply, &error,
				   DBUS_TYPE_STRING, &ev.iap_name,
				   DBUS_TYPE_INVALID))
		goto error_exit;

	ev.type = OSSO_IAP_DISCONNECTED;
	if (osso_cb)
		osso_cb(&ev, arg);
	disconnected();

	dbus_message_unref(reply);
	return;

error_exit:
	if (osso_iap_disconnect_filtered) {
		ev.type = OSSO_IAP_DISCONNECTED;
		ev.iap_name = osso_iap_name;

		if (osso_cb)
			osso_cb(&ev, NULL);
		disconnected();
	}

	report_error(&error, arg);
	dbus_error_free(&error);
	dbus_message_unref(reply);
}

static void statistics_handler(DBusPendingCall *pending,
			       void *arg)
{
	DBusMessage *reply;
	DBusError error;
	struct iap_event_t ev;

	memset(&ev, 0, sizeof(ev));
	dbus_error_init(&error);
	reply = dbus_pending_call_steal_reply(pending);

	if (dbus_set_error_from_message(&error, reply)) {
		report_error(&error, arg);
		dbus_error_free(&error);
		dbus_message_unref(reply);
		return;
	}

	ev.type = OSSO_IAP_STATISTICS;
	if (!dbus_message_get_args(
		reply, &error,
		DBUS_TYPE_STRING, &ev.iap_name,
		DBUS_TYPE_UINT32, &ev.u.statistics.time_active,
		DBUS_TYPE_UINT32, &ev.u.statistics.signal_strength,
		DBUS_TYPE_UINT32, &ev.u.statistics.rx_packets,
		DBUS_TYPE_UINT32, &ev.u.statistics.tx_packets,
		DBUS_TYPE_UINT32, &ev.u.statistics.rx_bytes,
		DBUS_TYPE_UINT32, &ev.u.statistics.tx_bytes,
		DBUS_TYPE_INVALID)) {
		report_error(&error, arg);
		dbus_error_free(&error);
		dbus_message_unref(reply);
		return;
	}

	if (osso_cb)
		osso_cb(&ev, arg);

	dbus_message_unref(reply);
}

gint osso_iap_connect(const char *iap, dbus_uint32_t flags, void *arg)
{
	DBusMessage *msg;
	gint ret;

	/* start listening for signals emitted by icm */
	if (get_connection(NULL) == NULL)
		return OSSO_ERROR;

	msg = dbus_message_new_method_call(
		ICD_DBUS_SERVICE,
		ICD_DBUS_PATH,
		ICD_DBUS_INTERFACE,
		"connect");
	if (msg == NULL)
		return OSSO_ERROR;

	if (!dbus_message_append_args(msg,
				      DBUS_TYPE_STRING, &iap,
				      DBUS_TYPE_UINT32, &flags,
				      DBUS_TYPE_INVALID)) {
		dbus_message_unref(msg);
		return OSSO_ERROR;
	}
	ret = send_message(msg, connected_handler, arg, 3*60*1000, 0);
	dbus_message_unref(msg);

	return ret;
}

gint osso_iap_cb(osso_iap_cb_t callback)
{
	/* register application */
	osso_cb = callback;

	return OSSO_OK;
}

gint osso_iap_disconnect(const char *iap, void *arg)
{
	DBusMessage *msg;
	gint ret;

	msg = dbus_message_new_method_call(
		ICD_DBUS_SERVICE,
		ICD_DBUS_PATH,
		ICD_DBUS_INTERFACE,
		"disconnect");
	if (msg == NULL)
		return OSSO_ERROR;

	if (!dbus_message_append_args(msg,
				      DBUS_TYPE_STRING, &iap,
				      DBUS_TYPE_INVALID)) {
		dbus_message_unref(msg);
		return OSSO_ERROR;
	}
	ret = send_message(msg, disconnected_handler, arg, 30*1000, 0);
	dbus_message_unref(msg);

	if (ret == OSSO_OK)
		osso_iap_disconnect_pending = TRUE;

	return ret;
}

gint osso_iap_get_statistics(const char *iap, void *arg)
{
	DBusMessage *msg;
	gint ret;

	msg = dbus_message_new_method_call(
		ICD_DBUS_SERVICE,
		ICD_DBUS_PATH,
		ICD_DBUS_INTERFACE,
		"get_statistics");
	if (msg == NULL)
		return OSSO_ERROR;

	if (!dbus_message_append_args(msg,
				      DBUS_TYPE_STRING, &iap,
				      DBUS_TYPE_INVALID)) {
		dbus_message_unref(msg);
		return OSSO_ERROR;
	}
	ret = send_message(msg, statistics_handler, arg, 1000, 0);
	dbus_message_unref(msg);

	return ret;
}

GSList *osso_iap_get_configured_iaps(void)
{
	GConfClient *client;
	GSList *list, *entry;

	client = gconf_client_get_default();
	if (client == NULL)
		return NULL;

	list = gconf_client_all_dirs(client, ICD_GCONF_PATH, NULL);
	g_object_unref(client);

	/* GConf return absolute directory names, so we just
	 * rewrite the strings to be relative directory names */
	for (entry = list; entry; entry = entry->next) {
		gchar *tmp;

		tmp = gconf_unescape_key(strrchr(entry->data, '/')+1, -1);
		g_free(entry->data);
		entry->data = tmp;
	}

	return list;
}

static DBusMessage *send_message_blocking(DBusMessage *msg,
				 int timeout)
{
	DBusConnection *conn;
	DBusMessage *reply;

	conn = get_connection(NULL);
	reply = dbus_connection_send_with_reply_and_block(conn,
							  msg,
							  timeout,
							  NULL);
	return reply;
}

static void signal_handler(int signum, siginfo_t *info, void *ctx)
{
	int i;

	if (info->si_code != SI_QUEUE || 
	    info->si_value.sival_int != 0xC0DE) {
		fprintf(stderr, "Spurious signal (0x%x)",
			info->si_value.sival_int);
		return;
	}

	for (i = 0; i < FD_SETSIZE; i++) {
		if (FD_ISSET(i, &open_sockets))
			shutdown(i, SHUT_RDWR);
	}
}
static int connect_iap_blocking(const char *iap)
{
	DBusMessage *msg;
	DBusMessage *reply;
	const char *iap_name;
	dbus_uint32_t flags = 0x0;
	dbus_uint32_t pid = getpid();

	msg = dbus_message_new_method_call(
		ICD_DBUS_SERVICE,
		ICD_DBUS_PATH,
		ICD_DBUS_INTERFACE,
		"connect");
	if (msg == NULL)
		return -1;

	if (!dbus_message_append_args(msg,
				      DBUS_TYPE_STRING, &iap,
				      DBUS_TYPE_UINT32, &flags,
				      DBUS_TYPE_UINT32, &pid,
				      DBUS_TYPE_INVALID)) {
		dbus_message_unref(msg);
		return -1;
	}
	reply = send_message_blocking(msg, 3*60*1000);
	dbus_message_unref(msg);

	if (reply == NULL)
		return -1;

	if (!dbus_message_get_args(reply, NULL,
				   DBUS_TYPE_STRING, &iap_name,
				   DBUS_TYPE_INVALID)) {
		dbus_message_unref(reply);
		return -1;
	}

	osso_iap_name = strdup(iap_name);
	dbus_message_unref(reply);

	return 0;
}

static void init(void)
{
	struct sigaction act;

	FD_ZERO(&open_sockets);

	act.sa_sigaction = &signal_handler;
	act.sa_flags = SA_SIGINFO;
	sigemptyset(&act.sa_mask);
	sigaction(SIGRTMAX-1, &act, NULL);
}

int osso_socket(int domain, int type, int protocol)
{
	int rc, i, n;
	struct protoent *pent;
	static int initialized;

	if (!initialized) {
		init();
		initialized = 1;
	}
	
	rc = socket(domain, type, protocol);

	if (rc < 0)
		return rc;

	if (domain == PF_INET || domain == PF_INET6) {
		pthread_mutex_lock(&mutex);
		pent = getprotobynumber(protocol);
#ifdef DEBUG		
		printf("Tracking osso_socket(%d=%s, %d=%s, %d=%s) = %d\n",
		       domain, SAFE_LOOKUP(domain, domain_names),
		       type, SAFE_LOOKUP(type, type_names),
		       protocol, pent?pent->p_name:"?",
		       rc);
#endif
		endprotoent();

		/* Track this socket */
		FD_SET(rc, &open_sockets);

		n = 0;
		for (i = 0; i < FD_SETSIZE; i++)
			n += FD_ISSET(i, &open_sockets);

		if (n == 1 && osso_iap_name == NULL) {
			/* First socked created -> connect */
			connect_iap_blocking(OSSO_IAP_ANY);

			if (osso_iap_name == NULL) {
				close(rc);

				FD_CLR(rc, &open_sockets);
				errno = EHOSTUNREACH;
				rc = -1;
			}
		}
		pthread_mutex_unlock(&mutex);
	}

	return rc;
}

int osso_close(int fd)
{
	int rc;

	rc = close(fd);
	if (rc < 0)
		return rc;
	
	if (FD_ISSET(fd, &open_sockets)) {
		FD_CLR(fd, &open_sockets);

#if 0
		n = 0;
		for (i = 0; i < FD_SETSIZE; i++)
			n += FD_ISSET(i, &open_sockets);

		if (n == 0 && osso_cb == NULL && osso_iap_name != NULL) {
			/* Last socket closed and API not used -> disconnect */
			printf("disconnecting IAP... ");
			fflush(stdout);
			osso_iap_disconnect(iap_name, NULL);
			printf("done.\n");
		}
#endif
	}

	return rc;
}
