/**
  @file btpair-signal.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <glib.h>

#include "log.h"
#include "btpair-error.h"
#include "btpair-engine.h"
#include "btpair-signal.h"

extern GMainLoop *mainloop;

static int signal_pipe[2];

static void handle_signal(int sig, PairingContext *ctx) {
    report("Signal %d received: %s.", sig, strsignal(sig));
    pair_cancel(ctx);
    g_main_loop_quit(mainloop);
}

static void signal_handler(int sig) {   
    send(signal_pipe[1], &sig, sizeof(sig), MSG_DONTWAIT);
}   

static gboolean signal_cb(GIOChannel *chan, GIOCondition cond, PairingContext *ctx) {
    int fd;

    if (cond != G_IO_IN) {
        error("signal_cb: cond != G_IO_IN");
        handle_signal(SIGTERM, ctx);
        g_io_channel_unref(chan);
        return FALSE;
    }

    fd = g_io_channel_unix_get_fd(chan);
    g_assert(fd >= 0);

    for (;;) {
        int sig;

        if (recv(fd, &sig, sizeof(sig), MSG_DONTWAIT) != sizeof(sig))
            break;

        handle_signal(sig, ctx);
    }

    return TRUE;
}

void bind_unix_signals(PairingContext *ctx) {
    GIOChannel *gio;

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, signal_pipe) < 0)
        die("socketpair: %s", g_strerror(errno));

    gio = g_io_channel_unix_new(signal_pipe[0]);
    g_io_add_watch(gio, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP,
                   (GIOFunc)signal_cb, ctx);

    if (signal(SIGINT, signal_handler) == SIG_ERR)
        die("signal(SIGINT) failed");
    if (signal(SIGTERM, signal_handler) == SIG_ERR)
        die("signal(SIGTERM) failed");
}

