/**
  @file btname-op.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "dbus.h"
#include "btname-op.h"

#define DEV_PATH "/org/bluez/hci0"

gboolean get_name(char *name, int name_len) {
    DBusConnection *conn;
    DBusMessage *msg, *reply;
    DBusError err;
    const char *str;

    conn = get_dbus_connection();
    if (!conn) {
        error("Unable to connect to system bus");
        return FALSE;
    }

    msg = dbus_message_new_method_call("org.bluez", DEV_PATH,
                                       "org.bluez.Adapter", "GetName");

    dbus_error_init(&err);
    reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);

    dbus_message_unref(msg);

    if (dbus_error_is_set(&err)) {
        error("Got error reply for GetName: %s", err.message);
        dbus_error_free(&err);
        return FALSE;
    }

    dbus_error_init(&err);
    if (!dbus_message_get_args(reply, &err,
                               DBUS_TYPE_STRING, &str,
                               DBUS_TYPE_INVALID)) {
        error("get_args failed: %s", err.message);
        dbus_error_free(&err);
        dbus_message_unref(reply);
        return FALSE;
    }

    dbus_message_unref(reply);

    strncpy(name, str, name_len);

    return TRUE;
}

gboolean set_name(const char *name) {
    DBusConnection *conn;
    DBusError err;
    DBusMessage *msg, *reply;

    conn = get_dbus_connection();
    if (!conn)
        return FALSE;

    msg = dbus_message_new_method_call("org.bluez", DEV_PATH,
                                       "org.bluez.Adapter", "SetName");

    if (!dbus_message_append_args(msg, DBUS_TYPE_STRING, &name,
                                  DBUS_TYPE_INVALID)) {
        dbus_message_unref(msg);
        return FALSE;
    }

    dbus_error_init(&err);
    reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);

    dbus_message_unref(msg);

    if (dbus_error_is_set(&err)) {
        error("SetName: %s", err.message);
        dbus_error_free(&err);
        return FALSE;
    }

    dbus_message_unref(reply);

    return TRUE;
}

