/**
  @file btname-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btname-op.h"
#include "dbus.h"
#include "dbus-helper.h"
#include "bttools-dbus.h"
#include "btname-dbus.h"

extern GMainLoop *event_loop;

#define EXIT_TIMEOUT 5000 /* milliseconds */

static GSource *exit_timer = NULL;

static gboolean exit_cb(gpointer user_data) {
    g_main_loop_quit(event_loop);
    return FALSE;
}

static void set_exit_timer(void) {
    if (exit_timer)
        g_source_destroy(exit_timer);
    exit_timer = g_timeout_source_new(EXIT_TIMEOUT);
    g_source_set_callback(exit_timer, exit_cb, NULL, NULL);
    (void) g_source_attach(exit_timer, NULL);
}

static void remove_exit_timer(void) {
    if (exit_timer) {
        g_source_destroy(exit_timer);
        exit_timer = NULL;
    }
}

static void reset_exit_timer(void) {
    remove_exit_timer();
    set_exit_timer();
}

static DBusHandlerResult btname_req_handler(DBusConnection     *connection,
                                            DBusMessage        *message,
                                            void               *user_data) {
    DBusMessage *reply;

    reset_exit_timer();

    if (dbus_message_is_method_call(message,
                                    BTNAME_REQUEST_IF,
                                    BTNAME_REQ_GET)) {
        char name[MAXNAMELEN];
        char *ptr = name;

        debug("Method call: get()");

        if (!get_name(name, sizeof(name)))
            reply = new_dbus_error(message, BTNAME_ERROR_FAILED, NULL); 
        else {
            reply = new_dbus_method_return(message);
            append_dbus_args(reply,
                             DBUS_TYPE_STRING, &ptr,
                             DBUS_TYPE_INVALID);
        }

        if (!send_and_unref(connection, reply))
            error("Sending D-BUS reply failed!");

        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (dbus_message_is_method_call(message,
                                    BTNAME_REQUEST_IF,
                                    BTNAME_REQ_SET)) {
        char *new_name;

        debug("Method call: set()");

        if (!dbus_message_get_args(message, NULL,
                                   DBUS_TYPE_STRING, &new_name,
                                   DBUS_TYPE_INVALID)) {
            send_invalid_args(connection, message);
            return DBUS_HANDLER_RESULT_HANDLED;
        }

        if (!set_name(new_name))
            reply = new_dbus_error(message, BTNAME_ERROR_FAILED, NULL);
        else {
            send_name_changed_signal(connection, new_name);
            reply = new_dbus_method_return(message);
        }

        if (!send_and_unref(connection, reply))
            error("Sending D-BUS reply failed!");

        return DBUS_HANDLER_RESULT_HANDLED;
    }

    error("Got unknown method call: %s", dbus_message_get_member(message));

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable btname_req_vtable = {
    .message_function    = btname_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection) {
    DBusError derr;

    dbus_error_init(&derr);
    dbus_connection_register_object_path(connection,
                                         BTNAME_REQUEST_PATH,
                                         &btname_req_vtable,
                                         &derr);
    if (dbus_error_is_set(&derr)) {
        error("register_object_path(%s) failed: %s",
                BTNAME_REQUEST_PATH, derr.message);
        dbus_error_free(&derr);
    }

    set_exit_timer();
}

void destroy_dbus_handlers(DBusConnection *connection) {
    dbus_connection_unregister_object_path(connection, BTNAME_REQUEST_PATH);
}

gboolean send_name_changed_signal(DBusConnection *connection, const char *new_name) {
    DBusMessage *signal;

    if (!connection)
        return FALSE;

    signal = new_dbus_signal(BTNAME_SIGNAL_PATH,
                             BTNAME_SIGNAL_IF,
                             BTNAME_SIG_CHANGED,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, &new_name,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(connection, signal)) {
        error("Sending \"changed\" D-BUS signal failed!");
        return FALSE;
    }

    return TRUE;
}

