/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2003-2004 Imendio AB
 * Copyright (C) 2006 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include "lm-internals.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>

#include "lm-error.h"

#if (defined(HAVE_GNUTLS) && defined(HAVE_OPENSSL))
#warning "You probably shouldn't have both HAVE_GNUTLS and HAVE_OPENSSL defined..."
#undef HAVE_GNUTLS
#endif

#ifdef HAVE_GNUTLS
#include <gnutls/x509.h>
#endif

#ifdef HAVE_OPENSSL
#include <openssl/ssl.h>
#include <openssl/err.h>
#define LM_SSL_CN_MAX		63
#endif

#ifdef HAVE_CST
#include <cst.h>
#endif

struct _LmSSL {
	LmSSLFunction   func;
	gpointer        func_data;
	GDestroyNotify  data_notify;
	gchar          *expected_fingerprint;
	gchar           fingerprint[20];

	gint            ref_count;
#ifdef HAVE_GNUTLS
	gnutls_session  gnutls_session;
	gnutls_certificate_client_credentials gnutls_xcred;
#endif
#ifdef HAVE_OPENSSL
	SSL_METHOD	*ssl_method;
	SSL_CTX		*ssl_ctx;
	SSL		*ssl;
#endif
#ifdef HAVE_CST
	CST		*cst;
#endif
};

static void           ssl_free                  (LmSSL       *ssl);

static LmSSLResponse  ssl_func_always_continue  (LmSSL       *ssl,
						 LmSSLStatus  status,
						 gpointer     user_data);

#ifdef HAVE_GNUTLS
static gboolean       ssl_verify_certificate    (LmSSL       *ssl,
						 const gchar *server);

static gboolean
ssl_verify_certificate (LmSSL *ssl, const gchar *server)
{
	int           status;

	/* This verification function uses the trusted CAs in the credentials
	 * structure. So you must have installed one or more CA certificates.
	 */
	status = gnutls_certificate_verify_peers (ssl->gnutls_session);

	if (status == GNUTLS_E_NO_CERTIFICATE_FOUND) {
		if (ssl->func (ssl,
			       LM_SSL_STATUS_NO_CERT_FOUND,
			       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
			return FALSE;
		}
	}
	
	if (status & GNUTLS_CERT_INVALID
	    || status & GNUTLS_CERT_REVOKED) {
		if (ssl->func (ssl, LM_SSL_STATUS_UNTRUSTED_CERT,
			       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
			return FALSE;
		}
	}
	
	if (gnutls_certificate_expiration_time_peers (ssl->gnutls_session) < time (0)) {
		if (ssl->func (ssl, LM_SSL_STATUS_CERT_EXPIRED,
			       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
			return FALSE;
		}
	}
	
	if (gnutls_certificate_activation_time_peers (ssl->gnutls_session) > time (0)) {
		if (ssl->func (ssl, LM_SSL_STATUS_CERT_NOT_ACTIVATED,
			       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
			return FALSE;
		}
	}
	
	if (gnutls_certificate_type_get (ssl->gnutls_session) == GNUTLS_CRT_X509) {
		const gnutls_datum* cert_list;
		guint cert_list_size;
		size_t digest_size;
		gnutls_x509_crt cert;
		
		cert_list = gnutls_certificate_get_peers (ssl->gnutls_session, &cert_list_size);
		if (cert_list == NULL) {
			if (ssl->func (ssl, LM_SSL_STATUS_NO_CERT_FOUND,
				       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				return FALSE;
			}
		}

		gnutls_x509_crt_init (&cert);

		if (!gnutls_x509_crt_import (cert, &cert_list[0],
					     GNUTLS_X509_FMT_DER)) {
			if (ssl->func (ssl, LM_SSL_STATUS_NO_CERT_FOUND, 
				       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				return FALSE;
			}
		}
		
		if (!gnutls_x509_crt_check_hostname (cert, server)) {
			if (ssl->func (ssl, LM_SSL_STATUS_CERT_HOSTNAME_MISMATCH,
				       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				return FALSE;
			}
		}

		gnutls_x509_crt_deinit (cert);

		if (gnutls_fingerprint (GNUTLS_DIG_MD5, &cert_list[0],
					     ssl->fingerprint,
					     &digest_size) >= 0) {
			if (ssl->expected_fingerprint &&
			    memcmp (ssl->expected_fingerprint, ssl->fingerprint,
				    digest_size) &&
			    ssl->func (ssl,
				       LM_SSL_STATUS_CERT_FINGERPRINT_MISMATCH,
				       ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				return FALSE;
			}
		} 
		else if (ssl->func (ssl, LM_SSL_STATUS_GENERIC_ERROR,
				    ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
			return FALSE; 
		} 
	}

	return TRUE;
}

void
_lm_ssl_initialize (LmSSL *ssl) 
{
	gnutls_global_init ();
	gnutls_certificate_allocate_credentials (&ssl->gnutls_xcred);
}

gboolean
_lm_ssl_begin (LmSSL *ssl, gint fd, const gchar *server, GError **error)
{
	int ret;
	gboolean auth_ok = TRUE;
	const int cert_type_priority[2] =
	{ GNUTLS_CRT_X509, GNUTLS_CRT_OPENPGP };

	gnutls_init (&ssl->gnutls_session, GNUTLS_CLIENT);
	gnutls_set_default_priority (ssl->gnutls_session);
	gnutls_certificate_type_set_priority (ssl->gnutls_session,
					      cert_type_priority);
	gnutls_credentials_set (ssl->gnutls_session,
				GNUTLS_CRD_CERTIFICATE,
				ssl->gnutls_xcred);

	gnutls_transport_set_ptr (ssl->gnutls_session,
				  (gnutls_transport_ptr) fd);

	ret = gnutls_handshake (ssl->gnutls_session);

	if (ret >= 0) {
		auth_ok = ssl_verify_certificate (ssl, server);
	}

	if (ret < 0 || !auth_ok) {
		char *errmsg;

		gnutls_perror (ret);
	
		if (!auth_ok) {
			errmsg = "*** GNUTLS authentication error";
		} else {
			errmsg = "*** GNUTLS handshake failed";
		}

		g_set_error (error, 
			     LM_ERROR, LM_ERROR_CONNECTION_OPEN,
			     errmsg);

		return FALSE;
	}
	
	return TRUE;
}

GIOStatus
_lm_ssl_read (LmSSL *ssl, gchar *buf, gint len, gsize *bytes_read)
{
	GIOStatus status;
	gint      b_read;

	*bytes_read = 0;
	b_read = gnutls_record_recv (ssl->gnutls_session, buf, len);

	if (b_read == GNUTLS_E_AGAIN) {
		status = G_IO_STATUS_AGAIN;
	}
	else if (b_read > len) {
		status = G_IO_STATUS_EOF;
	}
	else if (b_read < 0) {
		status = G_IO_STATUS_ERROR;
	} else {
		*bytes_read = (guint) b_read;
		status = G_IO_STATUS_NORMAL;
	}

	return status;
}

gint
_lm_ssl_send (LmSSL *ssl, const gchar *str, gint len)
{
	gint bytes_written;

	bytes_written = gnutls_record_send (ssl->gnutls_session, str, len);

	while (bytes_written < 0) {
		if (bytes_written != GNUTLS_E_INTERRUPTED &&
		    bytes_written != GNUTLS_E_AGAIN) {
			return -1;
		}
	
		bytes_written = gnutls_record_send (ssl->gnutls_session, 
						    str, len);
	}

	return bytes_written;
}

gboolean
_lm_ssl_data_pending (LmSSL *ssl)
{
	return gnutls_record_check_pending (ssl->gnutls_session);
}

void 
_lm_ssl_close (LmSSL *ssl)
{
	gnutls_deinit (ssl->gnutls_session);
	gnutls_certificate_free_credentials (ssl->gnutls_xcred);
	gnutls_global_deinit ();
}
#endif  /* HAVE_GNUTLS */


#ifdef HAVE_OPENSSL

static char _ssl_error_code[11];

static void
_lm_ssl_print_state (LmSSL *ssl, const char *func, int val)
{
	unsigned long errid;
	const char *errmsg;

	switch (SSL_get_error(ssl->ssl, val)) {
		case SSL_ERROR_NONE:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_NONE\n",
				func, val);
			break;
		case SSL_ERROR_ZERO_RETURN:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_ZERO_RETURN\n",
				func, val);
			break;
		case SSL_ERROR_WANT_READ:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_WANT_READ\n",
				func, val);
			break;
		case SSL_ERROR_WANT_WRITE:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_WANT_WRITE\n",
				func, val);
			break;
		case SSL_ERROR_WANT_X509_LOOKUP:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_WANT_X509_LOOKUP\n",
				func, val);
			break;
		case SSL_ERROR_SYSCALL:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_SYSCALL\n",
				func, val);
			break;
		case SSL_ERROR_SSL:
			fprintf(stderr,
				"%s(): %i / SSL_ERROR_SSL\n",
				func, val);
			break;
	}
	do {
		errid = ERR_get_error();
		if (errid) {
			errmsg = ERR_error_string(errid, NULL);
			fprintf(stderr, "\t%s\n", errmsg);
		}
	} while (errid != 0);
}

static const char *
_lm_ssl_get_x509_err (long verify_res)
{
	sprintf(_ssl_error_code, "%ld", verify_res);
	return _ssl_error_code;
}

#ifdef HAVE_CST
int
_lm_ssl_cst_cb (X509_STORE_CTX *x509_ctx, void *arg)
{
	int cst_ec;
	int cst_state;
	int cst_error;
	int retval = 1;
	LmSSL *ssl = (LmSSL *) arg;

	cst_ec = CST_is_valid(ssl->cst, x509_ctx->cert);
	cst_error = CST_last_error();
	cst_state = CST_get_state(ssl->cst, x509_ctx->cert);
	if (!cst_ec) {
		if (cst_error == CST_ERROR_CERT_NOTFOUND) {
			if (ssl->func(ssl,
				LM_SSL_STATUS_NO_CERT_FOUND,
				ssl->func_data) !=
				LM_SSL_RESPONSE_CONTINUE) {
				retval = 0;
			}
		}
		switch (cst_state) {
			case CST_STATE_NOTVALID:
			case CST_STATE_REVOKED:
				if (ssl->func(ssl,
					LM_SSL_STATUS_UNTRUSTED_CERT,
					ssl->func_data) !=
					LM_SSL_RESPONSE_CONTINUE) {
					retval = 0;
				}
				break;
			case CST_STATE_EXPIRED:
				if (ssl->func(ssl,
					LM_SSL_STATUS_CERT_EXPIRED,
					ssl->func_data) !=
					LM_SSL_RESPONSE_CONTINUE) {
					retval = 0;
				}
				break;
		}
	}
	return retval;
}
#endif
	
int
_lm_ssl_verify_cb (int preverify_ok, X509_STORE_CTX *x509_ctx)
{
	/* As this callback doesn't get auxiliary pointer parameter we
	 * cannot really use this. However, we can retrieve results later. */
	return 1;
}

void
_lm_ssl_initialize (LmSSL *ssl)
{
	const char *cert_file = NULL;

	SSL_library_init();
	SSL_load_error_strings();
	ssl->ssl_method = TLSv1_client_method();
	if (ssl->ssl_method == NULL) {
		fprintf(stderr, "TLSv1_client_method() == NULL\n");
		abort();
	}
	ssl->ssl_ctx = SSL_CTX_new(ssl->ssl_method);
	if (ssl->ssl_ctx == NULL) {
		fprintf(stderr, "SSL_CTX_new() == NULL\n");
		abort();
	}
	if (access("/etc/ssl/cert.pem", R_OK) == 0)
		cert_file = "/etc/ssl/cert.pem";
	if (!SSL_CTX_load_verify_locations(ssl->ssl_ctx,
		cert_file, "/etc/ssl/certs")) {
		fprintf(stderr, "SSL_CTX_load_verify_locations() failed\n");
	}
#ifdef HAVE_CST
	/* this seems to be broken */
	ssl->cst = CST_open(0, NULL);
	/* use file based approach instead */
	/*ssl->cst = CST_open_file("storage.cst", FALSE, NULL);*/
	/*CST_open_file("/usr/share/certs/certman.cst", FALSE, NULL);*/
	if (ssl->cst == NULL) {
		fprintf(stderr, "CST_open() == NULL\n");
		/*fprintf(stderr, "CST_open_file() == NULL\n");*/
		abort();
	}
	SSL_CTX_set_cert_verify_callback(ssl->ssl_ctx, _lm_ssl_cst_cb, ssl);
#endif
	SSL_CTX_set_verify(ssl->ssl_ctx, SSL_VERIFY_PEER, _lm_ssl_verify_cb);
}

gboolean
_lm_ssl_begin (LmSSL        *ssl,
	       gint          fd,
	       const gchar  *server,
	       GError      **error)
{
	gboolean retval = TRUE;
	int ssl_res;
	long verify_res;
	unsigned int digest_len;
	gchar *cn;
	X509 *srv_crt;
#ifndef HAVE_CST
	X509_NAME *crt_subj;
#else
	char *crt_domain;
#endif

	ssl->ssl = SSL_new(ssl->ssl_ctx);
	if (ssl->ssl == NULL) {
		fprintf(stderr, "SSL_new() == NULL\n");
		g_set_error(error, LM_ERROR, LM_ERROR_CONNECTION_OPEN,
			"SSL_new()");
		return FALSE;
	}
	if (!SSL_set_fd(ssl->ssl, fd)) {
		fprintf(stderr, "SSL_set_fd() failed\n");
		g_set_error(error, LM_ERROR, LM_ERROR_CONNECTION_OPEN,
			"SSL_set_fd()");
		return FALSE;
	}
	SSL_set_connect_state(ssl->ssl);
	ssl_res = SSL_connect(ssl->ssl);
	if (ssl_res <= 0) {
		_lm_ssl_print_state(ssl, "SSL_connect", ssl_res);
		g_set_error(error, LM_ERROR, LM_ERROR_CONNECTION_OPEN,
			"SSL_connect()");
		return FALSE;
	}

	/* certificate verification */

	/*fprintf(stderr, "Cipher: %s/%s/%i\n",
		SSL_get_cipher_version(ssl->ssl),
		SSL_get_cipher_name(ssl->ssl),
		SSL_get_cipher_bits(ssl->ssl, NULL));*/
	verify_res = SSL_get_verify_result(ssl->ssl);
	srv_crt = SSL_get_peer_certificate(ssl->ssl);
	if (ssl->expected_fingerprint != NULL) {
		X509_digest(srv_crt, EVP_md5(), (guchar *) ssl->fingerprint,
			&digest_len);
		if (memcmp(ssl->expected_fingerprint, ssl->fingerprint,
			digest_len) != 0) {
			if (ssl->func(ssl,
				LM_SSL_STATUS_CERT_FINGERPRINT_MISMATCH,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				return FALSE;
			}
		}
	}
	switch (verify_res) {
		case X509_V_OK:
			break;
		case X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT:
			/* special case for self signed certificates? */
		case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT:
		case X509_V_ERR_UNABLE_TO_GET_CRL:
		case X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE:
			if (ssl->func(ssl,
				LM_SSL_STATUS_NO_CERT_FOUND,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
			break;
		case X509_V_ERR_INVALID_CA:
		case X509_V_ERR_CERT_UNTRUSTED:
		case X509_V_ERR_CERT_REVOKED:
			if (ssl->func(ssl,
				LM_SSL_STATUS_UNTRUSTED_CERT,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
			break;
		case X509_V_ERR_CERT_NOT_YET_VALID:
		case X509_V_ERR_CRL_NOT_YET_VALID:
			if (ssl->func(ssl,
				LM_SSL_STATUS_CERT_NOT_ACTIVATED,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
			break;
		case X509_V_ERR_CERT_HAS_EXPIRED:
		case X509_V_ERR_CRL_HAS_EXPIRED:
			if (ssl->func(ssl,
				LM_SSL_STATUS_CERT_EXPIRED,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
			break;
		default:
			if (ssl->func(ssl, LM_SSL_STATUS_GENERIC_ERROR,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
	}
	if (retval == FALSE) {
		g_set_error (error, LM_ERROR, LM_ERROR_CONNECTION_OPEN,
			_lm_ssl_get_x509_err(verify_res), NULL);
	}
#ifndef HAVE_CST
	crt_subj = X509_get_subject_name(srv_crt);
	cn = (gchar *) g_malloc0(LM_SSL_CN_MAX + 1);
	if (cn == NULL) {
		fprintf(stderr, "g_malloc0() out of memory @ %s:%d\n",
			__FILE__, __LINE__);
		abort();
	}
	if (X509_NAME_get_text_by_NID(crt_subj, NID_commonName, cn,
		LM_SSL_CN_MAX) > 0) {
		if (strncmp(server, cn, LM_SSL_CN_MAX) != 0) {
			if (ssl->func(ssl,
				LM_SSL_STATUS_CERT_HOSTNAME_MISMATCH,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
		}
	} else {
		fprintf(stderr, "X509_NAME_get_text_by_NID() failed\n");
	}
	/*fprintf(stderr, "Issuer: %s\nSubject: %s\nFor: %s\n",
		X509_NAME_oneline(X509_get_issuer_name(srv_crt), NULL, 0),
		X509_NAME_oneline(X509_get_subject_name(srv_crt), NULL, 0),
		cn);*/
	g_free(cn);
#else  /* HAVE_CST */
	crt_domain = CST_get_domain_name(srv_crt);
	/*fprintf(stderr, "Issued for CN: %s\n", crt_domain);*/
	if (crt_domain != NULL) {
		if (strcmp(server, crt_domain) != 0) {
			if (ssl->func(ssl,
				LM_SSL_STATUS_CERT_HOSTNAME_MISMATCH,
				ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
				retval = FALSE;
			}
		}
	} else {
		if (ssl->func(ssl,
			LM_SSL_STATUS_CERT_HOSTNAME_MISMATCH,
			ssl->func_data) != LM_SSL_RESPONSE_CONTINUE) {
			retval = FALSE;
		}
	}
#endif  /* HAVE_CST */

	return retval;
}

GIOStatus
_lm_ssl_read (LmSSL *ssl,
	      gchar *buf,
	      gint   len,
	      gsize  *bytes_read)
{
	int ssl_res;
	
	ssl_res = SSL_read(ssl->ssl, buf, len);
	if (ssl_res < 0)
		_lm_ssl_print_state(ssl, "SSL_read", ssl_res);
	*bytes_read = ssl_res;
	if (ssl_res < 0)
		return G_IO_STATUS_ERROR;
	else if (ssl_res == 0)
		return G_IO_STATUS_EOF;

	return G_IO_STATUS_NORMAL;
}

gint
_lm_ssl_send (LmSSL *ssl, const gchar *buf, gint len)
{
	int ssl_res;

	ssl_res = SSL_write(ssl->ssl, buf, len);
	if (ssl_res < 0) {
		_lm_ssl_print_state(ssl, "SSL_write", ssl_res);
		return -1;
	}
	return ssl_res;
}

gboolean
_lm_ssl_data_pending (LmSSL *ssl)
{
	return (SSL_pending (ssl->ssl) > 0);
}

void 
_lm_ssl_close (LmSSL *ssl)
{
	SSL_shutdown(ssl->ssl);
	SSL_free(ssl->ssl);
	ssl->ssl = NULL;
}

void
_lm_ssl_free (LmSSL *ssl)
{
#ifdef HAVE_CST
	if (ssl->cst != NULL)
		CST_free(ssl->cst);
	ssl->cst = NULL;
#endif
	SSL_CTX_free(ssl->ssl_ctx);
	ssl->ssl_ctx = NULL;
	/* there's no way to handle SSL_METHOD? */
}

#endif  /* HAVE_OPENSSL */


static void
ssl_free (LmSSL *ssl)
{
#ifdef HAVE_OPENSSL
	_lm_ssl_free(ssl);
#endif
	g_free (ssl->expected_fingerprint);
	g_free (ssl);
}


static LmSSLResponse  
ssl_func_always_continue (LmSSL       *ssl,
			  LmSSLStatus  status,
			  gpointer     user_data)
{
	return LM_SSL_RESPONSE_CONTINUE;;
}

/**
 * lm_ssl_is_supported:
 *
 * Checks whether Loudmouth supports SSL or not.
 *
 * Return value: #TRUE if this installation of Loudmouth supports SSL, otherwise returns #FALSE.
 **/
gboolean
lm_ssl_is_supported (void)
{
#if (defined(HAVE_GNUTLS) || defined(HAVE_OPENSSL))
	return TRUE;
#else
	return FALSE;
#endif
}

/**
 * lm_ssl_new:
 * @expected_fingerprint: The expected fingerprint. @ssl_function will be called if there is a mismatch. %NULL if you are not interested in this check.
 * @ssl_function: Callback called to inform the user of a problem during setting up the SSL connection and how to proceed. If %NULL is passed the default function that always continues will be used.
 * @user_data: Data sent with the callback.
 * @notify: Function to free @user_dataa when the connection is finished. %NULL if @user_data should not be freed.
 *
 * Creates a new SSL struct, call #lm_connection_set_ssl to use it. 
 *
 * Return value: A new #LmSSL struct.
 **/
LmSSL *
lm_ssl_new (const gchar    *expected_fingerprint,
	    LmSSLFunction   ssl_function,
	    gpointer        user_data,
	    GDestroyNotify  notify)
{
	LmSSL *ssl;

	ssl = g_new0 (LmSSL, 1);
	
	ssl->ref_count      = 1;
	ssl->func           = ssl_function;
	ssl->func_data      = user_data;
	ssl->data_notify    = notify;
	ssl->fingerprint[0] = '\0';

	if (expected_fingerprint) {
		ssl->expected_fingerprint = g_strdup (expected_fingerprint);
	} else {
		ssl->expected_fingerprint = NULL;
	}

	if (!ssl->func) {
		/* If user didn't provide an SSL func the default will be used
		 * this function will always tell the connection to continue.
		 */
		ssl->func = ssl_func_always_continue;
	}

	return ssl;
}

/**
 * lm_ssl_get_fingerprint: 
 * @ssl: an #LmSSL
 *
 * Returns the MD5 fingerprint of the remote server's certificate.
 * 
 * Return value: A 16-byte array representing the fingerprint or %NULL if unknown.
 **/
const gchar *
lm_ssl_get_fingerprint (LmSSL *ssl)
{
	g_return_val_if_fail (ssl != NULL, NULL);
	
	return ssl->fingerprint;
}

/**
 * lm_ssl_ref:
 * @ssl: an #LmSSL
 * 
 * Adds a reference to @ssl.
 * 
 * Return value: the ssl
 **/
LmSSL *
lm_ssl_ref (LmSSL *ssl)
{
	g_return_val_if_fail (ssl != NULL, NULL);

	ssl->ref_count++;

	return ssl;
}

/**
 * lm_ssl_unref
 * @ssl: an #LmSSL
 * 
 * Removes a reference from @ssl. When no more references are present
 * @ssl is freed.
 **/
void 
lm_ssl_unref (LmSSL *ssl)
{
	g_return_if_fail (ssl != NULL);
        
        ssl->ref_count --;
        
        if (ssl->ref_count == 0) {
		if (ssl->data_notify) {
			(* ssl->data_notify) (ssl->func_data);
		}
               
		ssl_free (ssl);
        }
}

/* Define the TLS functions as noops if we compile without support */
#if (!defined(HAVE_GNUTLS) && !defined(HAVE_OPENSSL))

void
_lm_ssl_initialize (LmSSL *ssl)
{
	/* NOOP */
}

gboolean
_lm_ssl_begin (LmSSL        *ssl,
	       gint          fd,
	       const gchar  *server,
	       GError      **error)
{
	return TRUE;
}

GIOStatus
_lm_ssl_read (LmSSL *ssl,
	      gchar *buf,
	      gint   len,
	      gsize  *bytes_read)
{
	/* NOOP */
	*bytes_read = 0;

	return G_IO_STATUS_EOF;
}

gboolean 
_lm_ssl_send (LmSSL *ssl, const gchar *str, gint len)
{
	/* NOOP */
	return TRUE;
}

gboolean
_lm_ssl_data_pending (LmSSL *ssl)
{
	/* NOOP */
	return TRUE;
}

void 
_lm_ssl_close (LmSSL *ssl)
{
	/* NOOP */
}

#endif

