/*
 * rtpstream.c - Source for RTP plugin stream implementation
 *
 * Farsight RTP/AVP/SAVP/AVPF Module
 * Copyright (C) 2005,2006 Collabora Ltd.
 * Copyright (C) 2005,2006 Nokia Corporation
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>
 *   @author Philippe Kalaf <philippe.kalaf@collabora.co.uk>
 * Copyright (C) 2005 INdT 
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "rtp.h"
#include "rtpstream.h"
#include "rtpsession.h"
#include "rtpgstcodecs.h"
#include "farsight-transport.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "helpers/farsight-interfaces.h"
#include <arpa/inet.h>

#ifdef HAVE_SOFIASIP
#include <sofia-sip/su.h>
#include <sofia-sip/su_localinfo.h>
#endif

#ifdef HAVE_GNET
#include <gnet.h>
#endif

#include <gst/gst.h>

#include <netinet/in.h>

#define DEFAULT_AUDIO_PORT 7078
#define DEFAULT_VIDEO_PORT 9078

#define MAX_MTU 100

#define DEFAULT_STUN_IP "64.233.167.126"
#define DEFAULT_STUN_PORT 19302

#define DEFAULT_CONN_TIMEOUT 45

enum
{
  PROP_0,
  PROP_STUN_IP,
  PROP_STUN_PORT,
  PROP_TURN_IP,
  PROP_TURN_PORT,
  PROP_CONN_TIMEOUT,
  PROP_CODEC_PREF
};

FarsightCodecPreference codec_pref[] = {
  { "AMR", 8000 },
  { "GSM", 8000 },
  { "speex", 8000 },
  { "speex", 16000 },
  { "iLBC", 8000 },
  { "PCMA", 8000 },
  { "PCMU", 8000 }
};

static void farsight_rtp_stream_class_init (FarsightRTPStreamClass *klass);
static void farsight_rtp_stream_init (FarsightRTPStream *rtp_stream);
static GObject * farsight_rtp_stream_constructor (GType type,
    guint n_construct_properties,
    GObjectConstructParam *construct_properties);

static void farsight_rtp_stream_dispose (GObject *object);
static void farsight_rtp_stream_finalize (GObject *object);

static void farsight_rtp_stream_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void farsight_rtp_stream_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static gboolean hash_find_func (const gchar *key, 
                                CodecInternal *codec_internal,
                                const FarsightCodec *codec_to_match);
static gboolean farsight_rtp_stream_bus_watch_cb (GstBus *bus, 
                                                  GstMessage *message, 
                                                  gpointer user_data);
static void farsight_rtp_stream_new_payload_type (GstElement *element, 
                                                  gint pt, GstPad *pad, 
                                                  gpointer user_data);
static void farsight_rtp_stream_payload_type_change (GstElement *element, 
                                                     gint pt,
                                                     gpointer user_data);
static void farsight_rtp_stream_create_new_pt_recv_pipeline (FarsightStream *stream, 
                                                             gint id);
static void farsight_rtp_stream_set_active_codec (FarsightStream *stream, 
                                                  gint id);
static gint farsight_rtp_stream_get_active_codec (FarsightStream *stream);
static G_CONST_RETURN GList *farsight_rtp_stream_get_local_codecs (FarsightStream *stream);
static void farsight_rtp_stream_set_remote_codecs (FarsightStream *stream, 
                                                   const GList *codecs);
static GList *farsight_rtp_stream_get_codec_intersection (FarsightStream *stream);
static void farsight_rtp_stream_set_codec_preference_list (FarsightStream *stream,
    FarsightCodecPreference codec_pref[]);
static gboolean farsight_rtp_stream_set_source (FarsightStream *stream, 
                                                GstElement *source);
static gboolean farsight_rtp_stream_set_source_filter (FarsightStream *stream,
    GstCaps *filter);
static GstElement *farsight_rtp_stream_get_source (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_sink (FarsightStream *stream, 
                                              GstElement *sink);
static gboolean farsight_rtp_stream_set_sink_filter (FarsightStream *stream,
    GstCaps *filter);
static GstElement *farsight_rtp_stream_get_sink (FarsightStream *stream);
static GstElement *create_codec_bin (FarsightRTPStream *self, 
                                     gint codec_id, 
                                     gboolean dir);

static gboolean farsight_rtp_stream_build_base_pipeline (FarsightRTPStream *self);
static gboolean farsight_rtp_stream_build_send_pipeline (FarsightRTPStream *selft);
static gboolean farsight_rtp_stream_start (FarsightStream *stream);
static void farsight_rtp_stream_stop (FarsightStream *stream);
static gboolean farsight_rtp_stream_choose_codec (FarsightRTPStream *self, 
                                                  CodecInternal **codec_internal);
static void farsight_rtp_stream_upnp_send_request(FarsightRTPStream *self,
                                          FarsightTransportInfo *derived_trans);
static void farsight_rtp_stream_upnp_close_ports (FarsightRTPStream *self);

static void farsight_rtp_stream_native_candidate_ready (gpointer stream, 
                                                        const GList *candidate);
static gboolean farsight_rtp_stream_candidate_exists (FarsightStream *stream, 
                                                      const GList *candidate_list, 
                                                      const GList *candidate);
static void farsight_rtp_stream_socket_state_changed (gpointer stream, 
                                                      gint state);
static void farsight_rtp_stream_network_error (gpointer stream);
static void farsight_rtp_stream_prepare_transports (FarsightStream *self);
static gboolean farsight_rtp_stream_add_remote_candidate_to_rtpbin (FarsightRTPStream *self, 
                                                                    const gchar *remote_candidate_id);
static gboolean farsight_rtp_stream_set_active_candidate_pair (FarsightStream *stream,
                                                               const gchar *native_candidate_id, 
                                                               const gchar *remote_candidate_id);
static GList *farsight_rtp_stream_get_native_candidate (FarsightStream *stream, 
                                                        const gchar *candidate_id);
static G_CONST_RETURN GList *farsight_rtp_stream_get_native_candidate_list (FarsightStream *stream);
static void farsight_rtp_stream_set_remote_candidate_list (FarsightStream *stream, 
                                                           const GList *remote_candidates);
static void farsight_rtp_stream_add_remote_candidate (FarsightStream *self,
                                                      const GList *remote_candidate);

static gboolean farsight_rtp_stream_connection_timed_out (gpointer data);

FarsightStreamClass *rtp_stream_parent_class = NULL;

GType
farsight_rtp_stream_get_type (void)
{
  static GType type = 0;

  if (type == 0) {
    static const GTypeInfo info = {
      sizeof (FarsightRTPStreamClass),
      NULL,
      NULL,
      (GClassInitFunc) farsight_rtp_stream_class_init,
      NULL,
      NULL,
      sizeof (FarsightRTPStream),
      0,
      (GInstanceInitFunc) farsight_rtp_stream_init
    };

    type = g_type_register_static (FARSIGHT_TYPE_STREAM,
        "FarsightRTPStream", &info, 0);
  }

  return type;
}

static void
farsight_rtp_stream_class_init (FarsightRTPStreamClass *klass)
{
  GObjectClass *gobject_class;
  FarsightStreamClass *farsight_stream_class;

  gobject_class = (GObjectClass *) klass;
  farsight_stream_class = (FarsightStreamClass *) klass;
  rtp_stream_parent_class = g_type_class_peek_parent (klass);

  gobject_class->constructor = farsight_rtp_stream_constructor;

  gobject_class->set_property = farsight_rtp_stream_set_property;
  gobject_class->get_property = farsight_rtp_stream_get_property;

  g_object_class_install_property (gobject_class, PROP_STUN_IP,
      g_param_spec_string ("stun_ip", "STUN server ip", 
        "The IP address of the STUN server to use", 
        NULL, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_STUN_PORT,
      g_param_spec_uint ("stun_port", "STUN server port",
        "The port to the STUN server", 0, G_MAXUINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_STUN_IP,
      g_param_spec_string ("turn_ip", "TURN server ip", 
        "The IP address of the TURN server to use",
        NULL, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_TURN_PORT,
      g_param_spec_uint ("turn_port", "TURN server port",
        "The port to the TURN server", 0, G_MAXUINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CONN_TIMEOUT,
      g_param_spec_uint ("conn_timeout", "Connection timeout",
        "Number of secs before connection timeout", 0, G_MAXUINT, 0,
        G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_CODEC_PREF,
      g_param_spec_pointer ("codec_pref", "Codec preference list", 
          "A pointer to an array of FarsightCodecPreference",
          G_PARAM_READWRITE));

  gobject_class->dispose = farsight_rtp_stream_dispose;
  gobject_class->finalize = farsight_rtp_stream_finalize;

  farsight_stream_class->prepare_transports =
      farsight_rtp_stream_prepare_transports;
  farsight_stream_class->get_native_candidate_list = 
      farsight_rtp_stream_get_native_candidate_list;
  farsight_stream_class->get_native_candidate = 
      farsight_rtp_stream_get_native_candidate;
  farsight_stream_class->set_remote_candidate_list = 
      farsight_rtp_stream_set_remote_candidate_list;
  farsight_stream_class->add_remote_candidate =
      farsight_rtp_stream_add_remote_candidate;
  //farsight_stream_class->remove_remote_candidate =
      //farsight_rtp_stream_remove_remote_candidate;
  farsight_stream_class->set_active_candidate_pair =
      farsight_rtp_stream_set_active_candidate_pair;
  farsight_stream_class->get_local_codecs = 
      farsight_rtp_stream_get_local_codecs;
  farsight_stream_class->set_remote_codecs = 
      farsight_rtp_stream_set_remote_codecs;
  farsight_stream_class->get_codec_intersection =
    farsight_rtp_stream_get_codec_intersection;
  farsight_stream_class->set_codec_preference_list =
    farsight_rtp_stream_set_codec_preference_list;
  farsight_stream_class->set_active_codec = 
      farsight_rtp_stream_set_active_codec;
  farsight_stream_class->get_active_codec = 
      farsight_rtp_stream_get_active_codec;
  farsight_stream_class->set_sink = farsight_rtp_stream_set_sink;
  farsight_stream_class->set_sink_filter = farsight_rtp_stream_set_sink_filter;
  farsight_stream_class->get_sink = farsight_rtp_stream_get_sink;
  farsight_stream_class->set_source = farsight_rtp_stream_set_source;
  farsight_stream_class->set_source_filter = farsight_rtp_stream_set_source_filter;
  farsight_stream_class->get_source = farsight_rtp_stream_get_source;
  farsight_stream_class->start = farsight_rtp_stream_start;
  farsight_stream_class->stop = farsight_rtp_stream_stop;
}

static void
farsight_rtp_stream_init (FarsightRTPStream *rtp_stream)
{
  rtp_stream->remote_codecs = NULL;

  rtp_stream->pipeline = NULL;
  rtp_stream->rtpbin = NULL;
  rtp_stream->send_codec_bin = NULL;

  rtp_stream->src = NULL;
  rtp_stream->src_filter = NULL;

  rtp_stream->rtpdemux = NULL;
  rtp_stream->sink = NULL;
  rtp_stream->sink_filter = NULL;

  rtp_stream->prepared = FALSE;

  rtp_stream->bus_watch = 0;

#ifdef HAVE_JINGLEP2P
  rtp_stream->socket_client = NULL;
#endif

  rtp_stream->stun_ip = g_strdup (DEFAULT_STUN_IP);
  rtp_stream->stun_port = DEFAULT_STUN_PORT;
  rtp_stream->turn_ip = NULL;
  rtp_stream->turn_port = 0;

  rtp_stream->local_candidates = NULL;
  rtp_stream->remote_candidates = NULL;

  /* this tells _start to auto pick a codec */
  rtp_stream->send_codec_id = -1;

  rtp_stream->start_me = FALSE;
  rtp_stream->build_send_pipeline = FALSE;

  rtp_stream->timeout_src = -1;
  rtp_stream->conn_timeout = DEFAULT_CONN_TIMEOUT;

  
#ifdef HAVE_CLINKC
  if (!getenv ("FARSIGHT_DISABLE_UPNP"))
  {
    rtp_stream->use_upnp = TRUE;
    g_message ("Setting uPnP to true");
    if (!upnp_cp_init())
    {
      g_warning ("Error setting up uPnP");
      rtp_stream->use_upnp = FALSE;
    }
  }
#endif
}

static GObject *
farsight_rtp_stream_constructor (GType type,
    guint n_construct_properties,
    GObjectConstructParam *construct_properties)
{
  GObject *obj;

  {
    /* Invoke parent constructor. */
    FarsightRTPStreamClass *klass;
    GObjectClass *parent_class;  
    klass = FARSIGHT_RTP_STREAM_CLASS (g_type_class_peek (FARSIGHT_TYPE_RTP_STREAM));
    parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
    obj = parent_class->constructor (type,
                                     n_construct_properties,
                                     construct_properties);
  }

  FarsightRTPStream *rtp_stream = FARSIGHT_RTP_STREAM (obj);

  guint media_type;

  g_object_get (G_OBJECT (rtp_stream), "media-type", &media_type, NULL);
  switch (media_type) {
    case FARSIGHT_MEDIA_TYPE_AUDIO:
      rtp_stream->base_local_port = 7078;
      break;
    case FARSIGHT_MEDIA_TYPE_VIDEO:
      rtp_stream->base_local_port = 9098;
      break;
    default:
      rtp_stream->base_local_port = 0;
  }

  g_message ("Media type is %d", media_type);

  // get locally available codecs
  const GList *load_codecs_result = load_codecs (media_type,
      &rtp_stream->list_codecs_internal);
  if (!load_codecs_result)
  {
    g_error ("Error running load_codecs()");
    return obj;
  }

  rtp_stream->local_codecs = g_list_copy ((GList *)load_codecs_result);

  if (media_type == FARSIGHT_MEDIA_TYPE_AUDIO)
  {
    sort_codecs (&rtp_stream->local_codecs,
            codec_pref,
            sizeof (codec_pref) / sizeof (FarsightCodecPreference));
  }

  rtp_stream->pt_caps_table =
      create_pt_caps_hashtable (rtp_stream->local_codecs);

  return obj;
}

static void
farsight_rtp_stream_dispose (GObject *object)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);
  
  if (self->disposed) {
    /* If dispose did already run, return. */
    return;
  }

  self->disposed = TRUE;

  farsight_rtp_stream_stop (FARSIGHT_STREAM (self));

  /* chain up to parent */
  G_OBJECT_CLASS (rtp_stream_parent_class)->dispose (object);
}

static void
farsight_rtp_stream_finalize (GObject *object)
{
  FarsightRTPStream *rtpstream = FARSIGHT_RTP_STREAM (object);

  g_return_if_fail (rtpstream != NULL);
  g_return_if_fail (FARSIGHT_IS_RTP_STREAM (rtpstream));

  if (rtpstream->local_candidates)
    farsight_transport_list_destroy (rtpstream->local_candidates);

  if (rtpstream->remote_candidates)
    farsight_transport_list_destroy (rtpstream->remote_candidates);

  if (rtpstream->local_codecs)
    g_list_free (rtpstream->local_codecs);

  if (rtpstream->remote_codecs)
    farsight_codec_list_destroy (rtpstream->remote_codecs);

  if (rtpstream->depayloader_chains_info) {
    g_hash_table_destroy (rtpstream->depayloader_chains_info);
  }

  if (rtpstream->active_native_candidate) {
    g_free (rtpstream->active_native_candidate);
  }

  if (rtpstream->active_remote_candidate) {
    g_free (rtpstream->active_remote_candidate);
  }

  if (rtpstream->stun_ip) {
    g_free (rtpstream->stun_ip);
  }

  if (rtpstream->turn_ip) {
    g_free (rtpstream->turn_ip);
  }

  if (rtpstream->pt_caps_table) {
    g_hash_table_destroy (rtpstream->pt_caps_table);
  }

  if (rtpstream->sink) {
    gst_object_unref (rtpstream->sink);
  }

  if (rtpstream->sink_filter) {
    gst_caps_unref (rtpstream->sink_filter);
  }

  if (rtpstream->src) {
    gst_object_unref (rtpstream->src);
  }

  if (rtpstream->src_filter) {
    gst_caps_unref (rtpstream->src_filter);
  }

  /* TODO free GSTelements */

  G_OBJECT_CLASS (rtp_stream_parent_class)->finalize (object);
}

static void farsight_rtp_stream_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec)
{
  FarsightRTPStream *rtpstream = FARSIGHT_RTP_STREAM (object);

  switch (prop_id) {
    case PROP_STUN_IP:
      rtpstream->stun_ip = g_value_dup_string (value);
      break;
    case PROP_STUN_PORT:
      rtpstream->stun_port = g_value_get_uint (value);
      break;
    case PROP_TURN_IP:
      rtpstream->turn_ip = g_value_dup_string (value);
      break;
    case PROP_TURN_PORT:
      rtpstream->turn_port = g_value_get_uint (value);
      break;
    case PROP_CONN_TIMEOUT:
      rtpstream->conn_timeout = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void farsight_rtp_stream_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec)
{
  FarsightRTPStream *rtpstream = FARSIGHT_RTP_STREAM (object);

  switch (prop_id) {
    case PROP_STUN_IP:
      g_value_set_string (value, rtpstream->stun_ip);
      break;
    case PROP_STUN_PORT:
      g_value_set_uint (value, rtpstream->stun_port);
      break;
    case PROP_TURN_IP:
      g_value_set_string (value, rtpstream->turn_ip);
      break;
    case PROP_TURN_PORT:
      g_value_set_uint (value, rtpstream->turn_port);
      break;
    case PROP_CONN_TIMEOUT:
      g_value_set_uint (value, rtpstream->conn_timeout);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
hash_find_func (const gchar *key, CodecInternal *codec_internal,
                const FarsightCodec *codec_to_match)
{
  /* check if its necessary compare more items */
  if (codec_internal->codec->id != codec_to_match->id)
    return FALSE;

  if (g_ascii_strcasecmp (codec_internal->codec->encoding_name,
        codec_to_match->encoding_name) != 0)
    return FALSE;

  /*
  if (codec_internal->codec->id >= 96 &&
      codec_internal->codec->clock_rate != codec_to_match->clock_rate)
    return FALSE;
    */

  return TRUE;
}

static gboolean
farsight_rtp_stream_bus_watch_cb (GstBus *bus, GstMessage *message, 
                                  gpointer user_data)
{
  FarsightStream *stream = FARSIGHT_STREAM (user_data);

  switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_EOS:
      g_debug ("%s (%d): end of stream on stream pipeline",
          __FUNCTION__, __LINE__);
      farsight_stream_signal_error (stream, 
                                    FARSIGHT_STREAM_ERROR_EOS, NULL);
      farsight_rtp_stream_stop (stream);
      break;
    case GST_MESSAGE_ERROR:
      {
      gchar *debug;
      GError *err;

      gst_message_parse_error (message, &err, &debug);
      g_free (debug);

      g_warning ("%s (%d): error on stream pipeline. "
                 "Error code=%d message=%s",
                 __FUNCTION__, __LINE__, err->code, err->message);
      g_error_free (err);

      farsight_stream_signal_error (stream,
                                    FARSIGHT_STREAM_UNKNOWN_ERROR, NULL);
      farsight_rtp_stream_stop (stream);
      break;
      }
    default:
      break;
  }

  return TRUE;
}

static void
farsight_rtp_stream_new_payload_type (GstElement *element, gint pt, 
                                      GstPad *pad, gpointer user_data)
{
  FarsightRTPStream *stream = FARSIGHT_RTP_STREAM (user_data);
  CodecInternal *codec_internal;
  DepayloaderChainInfo *chain_info;

  codec_internal = g_hash_table_lookup (
      (GHashTable *) stream->list_codecs_internal,
      (gpointer) ((gint) pt));
  if (!codec_internal) {
    g_warning ("%s (%d): g_hash_table_lookup on %d return null", __FUNCTION__,
        __LINE__, pt);
    farsight_rtp_stream_stop (FARSIGHT_STREAM(stream));
    farsight_stream_signal_error (FARSIGHT_STREAM(stream), 
                                  FARSIGHT_STREAM_UNKNOWN_ERROR, NULL);
    return;
  }

  if (!stream->depayloader_chains_info) {
    stream->depayloader_chains_info =
        g_hash_table_new (g_direct_hash, g_direct_equal);
  }
  chain_info = g_new0 (DepayloaderChainInfo, 1);
  chain_info->pad = pad;
  g_hash_table_insert (stream->depayloader_chains_info,
                       (gpointer) pt, (gpointer) chain_info);
}

static void
farsight_rtp_stream_payload_type_change (GstElement *element, gint pt,
                                         gpointer user_data)
{
  FarsightStream *self = FARSIGHT_STREAM (user_data);
  farsight_rtp_stream_create_new_pt_recv_pipeline(self, pt);
}

static void
farsight_rtp_stream_create_new_pt_recv_pipeline (FarsightStream *stream, gint id)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  DepayloaderChainInfo *chain_info = NULL;
  gchar *name = NULL;
  GstElement *codec_bin = NULL;
  CodecInternal *codec_internal = NULL;

  codec_internal = g_hash_table_lookup (
      (GHashTable *) self->list_codecs_internal,
      (gpointer) ((gint) id));
  if (!codec_internal) {
    g_warning ("Codec %d not supported", id);
    goto error;
  }

  g_debug ("%s (%d): active PT change to %d", __FUNCTION__, __LINE__, id);

  chain_info = g_hash_table_lookup (self->depayloader_chains_info,
          (gpointer) ((gint) id));

  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("recv%d", id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->pipeline), name);
  g_free (name);

  if (!codec_bin) {
    g_message ("%s (%d): setting up new pipeline for id=%d",
            __FUNCTION__, __LINE__, id);

    codec_bin = create_codec_bin (self, id, DIR_RECV);
    if (!codec_bin)
    {
      g_warning ("Couldn't create elements, check your gstreamer installation");
      goto error;
    }
    gst_bin_add (GST_BIN (self->pipeline), codec_bin);

    /* we emit now so we can set anything we need on the elements before they
     * are playing */
    farsight_stream_signal_codec_changed (stream, id);

    if (codec_internal->has_sink)
    {
        gst_element_set_state (codec_bin, GST_STATE_PAUSED);
    }

    gchar *padname = gst_pad_get_name (chain_info->pad);
    gst_element_link_pads (self->rtpdemux, padname, GST_ELEMENT(codec_bin),
            "sink");
    if (codec_internal->has_sink)
    {
        gst_element_set_state (codec_bin, GST_STATE_PLAYING);
    }
    g_free (padname);

  } else {
    g_debug ("%s (%d): pipeline for id %d already configured, using it",
        __FUNCTION__, __LINE__, id);
    gst_object_unref (GST_OBJECT (codec_bin));
  }

  if (!codec_internal->has_sink)
  {
    if (self->sink)
    {
      GstPad *sinkpad = gst_element_get_pad (self->sink, "sink");
      if (!gst_pad_is_linked (sinkpad))
      {
        g_debug ("%s (%d): linking audio sink to codec bin",
            __FUNCTION__, __LINE__);

        gst_bin_add (GST_BIN (self->pipeline), self->sink);
      }
      else
      {
        GstBin *old_codec_bin;

        g_debug ("%s (%d): relinking audio sink to new codec bin with pt %d",
            __FUNCTION__, __LINE__, id);

        name = g_strdup_printf ("recv%d", self->recv_codec_id);
        old_codec_bin = GST_BIN (gst_bin_get_by_name (GST_BIN (self->pipeline), name));
        g_free (name);

        gst_element_unlink (GST_ELEMENT (old_codec_bin), self->sink);

        gst_object_unref (GST_OBJECT (old_codec_bin));
      }
      gst_element_set_state (self->sink, GST_STATE_PAUSED);
      gst_element_link_filtered (GST_ELEMENT (codec_bin), self->sink, self->sink_filter);
      gst_element_set_state (self->sink, GST_STATE_PLAYING);

      gst_object_unref (GST_OBJECT (sinkpad));
    }
    else 
    {
      g_warning("received stream while sink unset, stopping");
      goto error;
    }
  }

  self->recv_codec_id = id;

  /* restart self pipeline */
  //gst_xml_write_file (GST_ELEMENT (self->pipeline), stdout);
  g_debug ("%s: setting state to playing", G_STRFUNC);
  return;

error:
  g_warning ("%s: PT change failed", G_STRFUNC);
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream, FARSIGHT_STREAM_UNKNOWN_ERROR, 
      "Error creating new recv pipeline");
}

static void 
farsight_rtp_stream_set_active_codec (FarsightStream *stream, gint id)
{
  /* TODO This dosen't work yet, need to make sure the audiosrc or whatever else is
   * used sends a new_segment after a reconnection, this is done by
   * overwriting the link/unlink functions
   * in the current case without the new segment the timestamp stops
   * incrementing */

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  gchar *name;
  GstElement *codec_bin = NULL;
  GstElement *old_codec_bin = NULL;
  GstElement *new_codec_bin = NULL;

  g_debug ("%s (%d): called to change codec from %d to %d", __FUNCTION__,
          __LINE__, farsight_rtp_stream_get_active_codec(stream), id);

  if (id == farsight_rtp_stream_get_active_codec(stream))
    return;

  /* Let's check if the given codec id is valid */
  if (!g_hash_table_lookup (
              (GHashTable *) self->list_codecs_internal,
              (gpointer) ((gint) id)))
  {
    g_message ("%s (%d): invalid codec id %d", 
              __FUNCTION__, __LINE__, id);
    return;
  }

  /* TODO */
  /* Let's make sure codec is in the remote_codec list */

  if (!self->send_codec_bin)
    goto set_var;

  g_debug ("%s (%d): changing active send PT to %d", __FUNCTION__,
          __LINE__, id);

  /* Let's see if a send codec bin for this codec already exists
   * in shouldn't really exist... */
  name = g_strdup_printf ("send%d", id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->pipeline), name);
  g_free (name);
  if (codec_bin) {
    gst_object_unref (GST_OBJECT (codec_bin));
    g_warning ("Send codec already exists for codec %d, this shouldn't happen",
        id);
    goto error;
  }

  /* Let's unlink the existing codec bin and delete it */
  name = g_strdup_printf ("send%d", self->send_codec_id);
  old_codec_bin = gst_bin_get_by_name (GST_BIN(self->pipeline), name);
  g_debug ("removing send codec bin %p on pipeline %p", codec_bin, self->pipeline);
  g_free (name);
  if (!old_codec_bin)
  {
    g_warning ("Couldn't find current send codec for codec %d!",
        self->send_codec_id);
    goto error;
  }

  gst_element_unlink (old_codec_bin, self->rtpbin);
  if (self->src)
  {
    gst_element_unlink (self->src, old_codec_bin);
  }

  gst_bin_remove (GST_BIN (self->pipeline), old_codec_bin);

  gst_object_unref (GST_OBJECT (old_codec_bin));

  self->send_codec_bin = NULL;

  /* Let's create a new send codec bin for this codec */
  new_codec_bin = create_codec_bin (self, id, DIR_SEND); 
  if (!new_codec_bin)
  {
    g_warning ("Couldn't create elements, check your gstreamer installation");
    goto error;
  }

  g_debug ("adding send codec bin %p on pipeline %p", new_codec_bin, self->pipeline);
  gst_bin_add (GST_BIN (self->pipeline), new_codec_bin);
  /* now we link the src and rtpbin to it */
  gst_element_link (new_codec_bin, self->rtpbin);
  if (self->src)
  {
    gst_element_link_filtered (self->src, new_codec_bin, self->src_filter);
  }

  self->send_codec_bin = new_codec_bin;

set_var:
  self->send_codec_id = id;

  return;

error:
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream, FARSIGHT_STREAM_UNKNOWN_ERROR, 
      "Error while changing send codec");
}

static gint 
farsight_rtp_stream_get_active_codec (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  return self->send_codec_id; 
}

static G_CONST_RETURN GList *
farsight_rtp_stream_get_local_codecs (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  return self->local_codecs;
}

static void
farsight_rtp_stream_set_remote_codecs (FarsightStream *stream, 
                                       const GList *codecs)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  self->remote_codecs = farsight_codec_list_copy (codecs);

  /* we sort the remote codecs according to our priorities */
  /* TODO What is the correct behaviour?
   * it seems wrong to ignore the remote preferences, should they be included in
   * the reply? Should they be only kept in memory for the stream we send while
   * our reply has out preference? 
   * GTalk's behaviour is to save our preference, send us what we prefer, but
   * in the intersection reply, he overwrites our preference with his. The
   * result is that we can get 2 different codecs being sent/recved. This won't
   * happen if we sort as done below. */
  sort_codecs (&self->remote_codecs,
      codec_pref,
      sizeof (codec_pref) / sizeof (FarsightCodecPreference));

  /*
  GList *codec_list = self->remote_codecs;
  do {
    FarsightCodec *codec = codec_list->data;
    g_debug ("remote_codec %s %d", codec->encoding_name, codec->clock_rate);
    codec_list = codec_list->next;
  } while (codec_list);
  */

  /* We can create the send pipeline now */
  farsight_rtp_stream_build_send_pipeline (self);
}

static GList *
farsight_rtp_stream_get_codec_intersection (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  GList *codecs = NULL;
  GList *el = self->remote_codecs;

  for (el = self->remote_codecs; el; el = el->next)
  {
    CodecInternal *ci;

    if ((ci = g_hash_table_find (
                    (GHashTable *) self->list_codecs_internal,
                    (GHRFunc) hash_find_func, el->data))) 
    {
      codecs = g_list_prepend (codecs, farsight_codec_copy (ci->codec));
    }
  }

  codecs = g_list_reverse (codecs);

  return codecs;
}

static void
farsight_rtp_stream_set_codec_preference_list (FarsightStream *stream, 
                                       FarsightCodecPreference codec_pref_list[])
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  sort_codecs (&self->local_codecs,
      codec_pref_list,
      sizeof (codec_pref_list) / sizeof (FarsightCodecPreference));
}

static gboolean
farsight_rtp_stream_set_source (FarsightStream *stream, GstElement *source)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting src", __FUNCTION__, __LINE__);
  if (self->src)
  {
    if (self->send_codec_bin)
    {
      g_warning ("Send pipeline already created, \
          will attempt to replace while pipeline is running");
      /* TODO do I need to pause the pipeline? pause the src? just remove it and
       * replace a new one? */
    }
    gst_object_unref(self->src);
  }
  self->src = source;
  gst_object_ref(source);

  if (self->build_send_pipeline)
  {
    farsight_rtp_stream_build_send_pipeline (self);
  }
  return TRUE;
}

static gboolean
farsight_rtp_stream_set_source_filter (FarsightStream *stream, GstCaps *filter)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting source filter", __FUNCTION__, __LINE__);
  if (self->src_filter)
    gst_caps_unref (self->src_filter);
  self->src_filter = filter;
  gst_caps_ref(filter);
  return TRUE;
}

static GstElement *
farsight_rtp_stream_get_source (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  if (self->src)
    return self->src;

  GstElement *codec_bin = NULL;
  GstIterator *iter;
  gchar *name;

  /* see if we have a src inside the pipeline */
  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("send%d", self->send_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gboolean done = FALSE;
    iter = gst_bin_iterate_elements (GST_BIN(codec_bin));

    while (!done) {
      gpointer data;
      switch (gst_iterator_next (iter, &data)) {
        case GST_ITERATOR_OK:
          {
            GstElement *child;
            gboolean is_src;
            child = GST_ELEMENT_CAST (data);

            GST_OBJECT_LOCK (child);
            if (!GST_OBJECT_FLAG_IS_SET (child, GST_ELEMENT_IS_SINK) &&
                !child->numsinkpads) {
              is_src = TRUE;
            }
            else
            {
              is_src = FALSE;
            }
            GST_OBJECT_UNLOCK (child);

            gst_object_unref (child);

            if (is_src)
            {
              gst_iterator_free (iter);
              return child;
            }
            break;
          }
        case GST_ITERATOR_RESYNC:
          gst_iterator_resync (iter);
          break;
        case GST_ITERATOR_DONE:
          done = TRUE;
          break;
        case GST_ITERATOR_ERROR:
          g_assert_not_reached ();
          break;
      }
    }
    gst_iterator_free (iter);
  }
  return NULL;
}

static gboolean
farsight_rtp_stream_set_sink (FarsightStream *stream, GstElement *sink)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting sink", __FUNCTION__, __LINE__);
  if (self->sink)
    gst_object_unref(self->sink);
  self->sink = sink;
  gst_object_ref(sink);
  return TRUE;
}

static gboolean
farsight_rtp_stream_set_sink_filter (FarsightStream *stream, GstCaps *filter)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting sink filter", __FUNCTION__, __LINE__);
  if (self->sink_filter)
    gst_caps_unref (self->sink_filter);
  self->sink_filter = filter;
  gst_caps_ref(filter);
  return TRUE;
}

static GstElement *
farsight_rtp_stream_get_sink (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  if (self->sink)
    return self->sink;

  GstElement *codec_bin = NULL;
  GstIterator *iter;
  gchar *name;

  /* see if we have a sink inside the pipeline */
  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("recv%d", self->recv_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gboolean done = FALSE;
    iter = gst_bin_iterate_elements (GST_BIN(codec_bin));

    while (!done) {
      gpointer data;

      switch (gst_iterator_next (iter, &data)) {
        case GST_ITERATOR_OK:
          {
            GstElement *child;
            gboolean is_sink;
            child = GST_ELEMENT_CAST (data);

            GST_OBJECT_LOCK (child);
            is_sink = GST_OBJECT_FLAG_IS_SET (child, GST_ELEMENT_IS_SINK);
            GST_OBJECT_UNLOCK (child);

            gst_object_unref (child);

            if (is_sink)
            {
              gst_iterator_free (iter);
              return child;
            }
            break;
          }
        case GST_ITERATOR_RESYNC:
          gst_iterator_resync (iter);
          break;
        case GST_ITERATOR_DONE:
          done = TRUE;
          break;
        case GST_ITERATOR_ERROR:
          g_assert_not_reached ();
          break;
      }
    }
    gst_iterator_free (iter);
  }

  return NULL;
}
/* Creates a codec bin with payloader/encoder or depayloader/decoder based on
 * dir */
static GstElement *
create_codec_bin (FarsightRTPStream *self, gint codec_id, gboolean dir)
{
  GstElement *codec_bin = NULL;
  GstElement *current_element = NULL;
  GstElement *previous_element = NULL;
  GstPad *sinkpad = NULL;
  GstPad *srcpad = NULL;
  GstPad *tmppad;
  GList *walk;
  CodecInternal *codec_internal;
  GList *pipeline_factory = NULL;
  gchar *name = NULL;

  codec_internal = g_hash_table_lookup (
          (GHashTable *) self->list_codecs_internal,
          (gpointer) ((gint) codec_id));
  if (!codec_internal)
    return NULL;

  if (dir == DIR_SEND)
    pipeline_factory = codec_internal->send_pipeline_factory;
  else
    pipeline_factory = codec_internal->receive_pipeline_factory;

  g_debug ("creating %s codec bin for id %d, pipeline_factory %p", (dir==DIR_SEND)?"send":"recv", codec_id, pipeline_factory);
  name = g_strdup_printf ((dir == DIR_SEND)?"send%d":"recv%d",
          codec_id);
  codec_bin = gst_bin_new (name);
  g_free(name);
  for (walk = pipeline_factory; walk; walk = g_list_next (walk)) {
    current_element =
        gst_element_factory_create ((GstElementFactory *) walk->data, NULL);
    if (!current_element)
      return NULL;
    /* first element */
    if (!previous_element)
    {
      GstPad *pad = gst_element_get_pad (current_element, "sink");
      /* make our bin sink pad (only if needed) */
      if (pad)
      {
        sinkpad = gst_ghost_pad_new ("sink", pad);
        gst_element_add_pad (codec_bin, sinkpad);
        gst_object_unref (GST_OBJECT (pad));
      }

      /* the first element is always the depayloader */
      if (dir == DIR_RECV)
      {
        gchar *pad_name = gst_element_get_name (current_element);

        /* queue delay at 0 for now */
        g_object_set (G_OBJECT (current_element), "queue-delay", 0, NULL);

        /* temp hack for 770 for rtpilbcdepay, even though this is the
         * default value i'm setting it again */
        if (g_strrstr (pad_name, "rtpilbcdepay") != NULL)
        {
          g_debug ("setting frame size to 30ms on rtpilbcdepay");
          g_object_set (current_element, "mode", 1, NULL);
        }

        g_free (pad_name);
      }

      if (dir == DIR_SEND)
      {
        gchar *pad_name = gst_element_get_name (current_element);

        /* Another temp hack for the 770 for the dspilbcsrc */
        /* set frame size to 30 */
        if (g_strrstr (pad_name, "dspilbcsrc") != NULL)
        {
          g_debug ("setting frame size to 30ms on dspilbcsrc");
          g_object_set (current_element, "mode", 1, NULL);
        }

        g_free (pad_name);
      }
    }
    gst_bin_add (GST_BIN (codec_bin), current_element);
    if (previous_element)
    {
      gst_element_link (previous_element, current_element);
    }
    previous_element = current_element;
  }

  tmppad = gst_element_get_pad (current_element, "src");
  /* make our bin source pad (only if needed) */
  if (tmppad)
  {
    srcpad = gst_ghost_pad_new ("src", tmppad);
    gst_element_add_pad (codec_bin, srcpad);
    g_object_unref (GST_OBJECT (tmppad));
  }

  /* HACK FOR 770, this should be the dsppcmsink */
  if (dir == DIR_RECV)
  {
    gchar *pad_name = gst_element_get_name(current_element);
    if (g_strrstr (pad_name, "dsppcmsink") != NULL)
    {
      g_debug ("Setting sync to false on %s", pad_name);
      g_object_set (current_element, "sync", FALSE, NULL);
      g_debug ("Turning on AEC");
      g_object_set (current_element, "aec-mode", 1, NULL);
    }
    g_free (pad_name);
  }

  if (dir == DIR_SEND)
  {
    /* sets the payload type to send */
    /* this supposes the last element is the payloader */
    g_object_set (current_element, "pt", codec_internal->codec->id, NULL);
    /* TODO find a better way to set the MTU, RTCP perhaps? */
    g_object_set (current_element, "mtu", MAX_MTU, NULL);
    g_object_set (current_element, "max_ptime", 20 * GST_MSECOND, NULL);
  }

  return codec_bin;
}

/* this build the core of the pipeline, meaning RTPBin and rtpdemux. The
 * send/recv parts are added later */
static gboolean
farsight_rtp_stream_build_base_pipeline (FarsightRTPStream *self)
{
  GstElement *rtpbin = NULL;

  g_return_val_if_fail (self != NULL, FALSE);

  g_debug ("%s (%d): creating core RTP pipeline", __FUNCTION__,
      __LINE__);
  /* 
   * build base pipeline 
   */
  if (self->pipeline == NULL) {
    /* Create pipeline and listen to gst bus */
    self->pipeline = gst_pipeline_new ("pipeline");
    GstBus *pipe_bus = gst_pipeline_get_bus (GST_PIPELINE (self->pipeline));
    self->bus_watch = gst_bus_add_watch (pipe_bus, farsight_rtp_stream_bus_watch_cb, self);
    gst_object_unref (pipe_bus);


    /* create rtpbin element */
    rtpbin = gst_element_factory_make ("rtpbin", NULL);
    if (!rtpbin)
    {
      g_warning ("Couldn't create rtpbin, check your gstreamer install");
      goto error;
    }
    self->rtpbin = rtpbin;
    gst_bin_add (GST_BIN (self->pipeline), rtpbin);

    g_object_set (G_OBJECT (rtpbin),
        "rtp-sockfd", self->local_socket,
        "rtcp-support", FALSE,
        "localport", self->local_port, NULL);

    g_object_set (G_OBJECT(rtpbin), "pt-map", self->pt_caps_table, NULL);

#ifdef HAVE_JINGLEP2P 
    /* create jingle specific icesrc element that get data from
     * libjingle */
    g_object_set (G_OBJECT(rtpbin), "bypass-udp", TRUE, NULL);
    GstElement *icesrc = NULL;

    icesrc = gst_element_factory_make ("icesrc", NULL);
    //icesink = gst_element_factory_make ("fakesink", NULL);

    gst_bin_add (GST_BIN (self->pipeline), icesrc);
    g_debug("added icesrc %p to pipeline %p with sockclient %p", icesrc,
        self->pipeline, self->socket_client);

    g_object_set (G_OBJECT(icesrc), "socketclient", self->socket_client,
        NULL);

    /* TODO HACK ALERT HACK ALERT */
    /* just adding a boggus destination for now */
    /* so that jrtplib releases the packets to someone */
    g_object_set (G_OBJECT (rtpbin), "destinations", "64.34.23.11:5000", NULL);
#endif

    /* link rtpbin to rtpdemux to be able to discover what codec 
     * should be used.
     * We don't actually connect the depayloaders decoders yet */
    self->rtpdemux = gst_element_factory_make ("rtpdemux", NULL);
    if (!self->rtpdemux)
    {
      g_warning ("Couldn't create rtpdemux, check your gstreamer install");
      goto error;
    }
    g_signal_connect (G_OBJECT (self->rtpdemux), "new-payload-type",
        G_CALLBACK (farsight_rtp_stream_new_payload_type), self);
    g_signal_connect (G_OBJECT (self->rtpdemux), "payload-type-change",
        G_CALLBACK (farsight_rtp_stream_payload_type_change), self);

    gst_bin_add (GST_BIN (self->pipeline),
        self->rtpdemux);
    if (!gst_element_link_pads (rtpbin, "src%d", self->rtpdemux,
        "sink"))
    {
      g_warning ("Could not link rtpbin:src to rtpdemux:sink");
      goto error;
    }

#ifdef HAVE_JINGLEP2P
    if (!gst_element_link_pads (icesrc, "src", rtpbin, "rtpsink"))
    {
      g_warning ("Could not link icesrc:src to rtpbin:rtpsink");
      goto error;
    }
#endif
  }

  return TRUE;

error:
  g_warning ("%s (%d): error setting up core RTP pipeline",
      __FUNCTION__, __LINE__);
  if (self->pipeline) {
    gst_object_unref (GST_OBJECT (self->pipeline));
    self->pipeline = NULL;
  }
  farsight_stream_signal_error (FARSIGHT_STREAM(self),
          FARSIGHT_STREAM_UNKNOWN_ERROR, 
          "Error setting up core RTP pipeline");
  return FALSE;
}

static gboolean
farsight_rtp_stream_build_send_pipeline (FarsightRTPStream *self)
{
  CodecInternal *codec_internal = NULL;
  GstElement *codec_bin;

  g_return_val_if_fail (self != NULL, FALSE);
  g_return_val_if_fail (self->rtpbin != NULL, FALSE);
  g_return_val_if_fail (self->pipeline != NULL, FALSE);

  /* Let us automaticaly pick a codec if not set by the user */
  if (self->send_codec_id == -1)
  {
    if (!farsight_rtp_stream_choose_codec (self, &codec_internal))
      return FALSE;

    self->send_codec_id = codec_internal->codec->id;
  }
  else
  {
    /* let's get the CodecInternal for the selected PT */
    codec_internal = g_hash_table_lookup (
        (GHashTable *) self->list_codecs_internal,
        (gpointer) ((gint) self->send_codec_id));
    if (!codec_internal) {
      g_warning ("Codec %d not supported", self->send_codec_id);
      goto error;
    }
  }

  g_message ("%s (%d): creating send pipeline with codec %d", __FUNCTION__,
      __LINE__, self->send_codec_id);

  if (!self->src && !codec_internal->has_src)
  {
    g_message ("No source has been set yet, \
        delaying send pipeline build for later");
    self->build_send_pipeline = TRUE;
    return FALSE;
  }

  /* add active remote candidate as destination if set */
  if (self->active_remote_candidate)
  {
    farsight_rtp_stream_add_remote_candidate_to_rtpbin(self,
        self->active_remote_candidate);
  }

  /*
   * sending part of pipeline
   */

  if (self->src)
  {
    gst_bin_add (GST_BIN (self->pipeline),
        self->src);
  }

  /* build send part based on send_pipeline_factory */
  /* put all these elements in a bin */
  codec_bin = create_codec_bin (self, self->send_codec_id,
      DIR_SEND);
  if (!codec_bin)
  {
    g_warning ("Couldn't create elements for codec %d", self->send_codec_id);
    goto error;
  }

  gst_bin_add (GST_BIN (self->pipeline), codec_bin);

  /* connect src to codec_bin */
  if (self->src)
  {
    g_debug ("linking src %p to codec bin %p",self->src, codec_bin);
    gst_element_link_filtered (self->src, codec_bin, self->src_filter);
  }
  /* connect rtpbin to codec_bin */
  if (!gst_element_link_pads (codec_bin, "src", self->rtpbin, "sink%d"))
  {
    g_warning ("Could not link codec_bin:src to rtpbin:sink");
    goto error;
  }

#ifdef HAVE_JINGLEP2P
  /* create jingle specific icesink element that give data to
   * libjingle */
  GstElement *icesink = NULL;
  icesink = gst_element_factory_make ("icesink", NULL);

  gst_bin_add (GST_BIN (self->pipeline), icesink);

  g_object_set (G_OBJECT(icesink), "socketclient", self->socket_client,
      NULL);

  if (!gst_element_link_pads (self->rtpbin, "rtpsrc", icesink, "sink"))
  {
    g_warning ("Could not link rtpbin:rtpsrc to icesink:sink");
    goto error;
  }
#endif

  self->send_codec_bin = codec_bin;

  self->build_send_pipeline = FALSE;

  if (self->start_me)
  {
    farsight_rtp_stream_start (FARSIGHT_STREAM(self));
  }

return TRUE;

error:
  g_warning ("%s (%d): error setting up send codec pipeline",
      __FUNCTION__, __LINE__);
  return FALSE;
}

static gboolean
farsight_rtp_stream_start (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  g_return_val_if_fail (self != NULL, FALSE);
  if (self->pipeline == NULL || 
      self->rtpbin == NULL || 
      self->send_codec_bin == NULL)
  {
    self->start_me = TRUE;
    return FALSE;
  }

  self->start_me = FALSE;

  gst_element_set_state (self->pipeline, GST_STATE_PLAYING);

  farsight_stream_signal_state_changed (stream, 
      FARSIGHT_STREAM_STATE_PLAYING,
      FARSIGHT_STREAM_DIRECTION_BOTH);

  return TRUE;
}


/**
 * Stop a #FarsightRTPStream instance.
 *
 * @param stream #FarsightRTPStream instance
 */
static void
farsight_rtp_stream_stop (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  g_return_if_fail (stream != NULL);

  if (self->timeout_src != -1) {
      g_source_remove (self->timeout_src);
      self->timeout_src = -1;
  }

  if (self->pipeline) {
    g_debug ("%s (%d): stopping media pipeline", __FUNCTION__, __LINE__);

    GstStateChangeReturn state_return;
    if (gst_element_set_state (self->pipeline, GST_STATE_NULL) == GST_STATE_CHANGE_ASYNC)
    {
      state_return = gst_element_get_state (self->pipeline, NULL, NULL, 5 * GST_SECOND);
      switch(state_return)
      {
        case GST_STATE_CHANGE_FAILURE:
          g_warning("Unable to set pipeline to NULL! This could break the teardown");
          break;
        case GST_STATE_CHANGE_ASYNC:
          g_warning ("State change not finished, after 5 seconds, This could break the teardown");
          break;
        default:
          break;
      }
    }
    else
    {
      g_debug ("Changed pipeline state to NULL succesfully");
    }

    if (self->bus_watch)
      g_source_remove (self->bus_watch);

    gst_object_unref (GST_OBJECT (self->pipeline));
    self->pipeline = NULL;
    self->src = NULL;
    self->sink = NULL;

    if (self->depayloader_chains_info) {
      g_hash_table_destroy (self->depayloader_chains_info);
      self->depayloader_chains_info = NULL;
    }
  }
  else {
    if (self->src) {
      gst_element_set_state (self->src, GST_STATE_NULL);
      gst_object_unref (GST_OBJECT (self->src));
      self->src = NULL;
    }
    if (self->sink) {
      gst_element_set_state (self->sink, GST_STATE_NULL);
      gst_object_unref (GST_OBJECT (self->sink));
      self->sink = NULL;
    }
  }

#ifdef HAVE_JINGLEP2P
  if (self->socket_client) {
    g_debug ("%s (%d): destroying socket client", __FUNCTION__, __LINE__);    
    socketclient_destroy (self->socket_client);
    self->socket_client = NULL;
  }
#endif

#ifdef HAVE_CLINKC
  if (self->use_upnp)
  {
    farsight_rtp_stream_upnp_close_ports (self);
    upnp_cp_shutdown();
  }
#endif

  farsight_stream_signal_state_changed (stream, FARSIGHT_STREAM_STATE_STOPPED, 
                                                FARSIGHT_STREAM_DIRECTION_NONE);
}


gboolean 
farsight_rtp_stream_choose_codec (FarsightRTPStream *self, 
                                  CodecInternal **codec_internal)
{
  GList *codec_list = self->remote_codecs;
  do {
    if ((*codec_internal = g_hash_table_find (
        (GHashTable *) self->list_codecs_internal,
        (GHRFunc) hash_find_func, codec_list->data))) {
      return TRUE;
    }
    codec_list = codec_list->next;
  } while (codec_list);
  return FALSE;
}

static void
farsight_rtp_stream_upnp_send_request(FarsightRTPStream *self,
    FarsightTransportInfo *derived_trans)
{
#ifdef HAVE_CLINKC
  FarsightTransportInfo *trans = NULL;
  const GList *lp;
  GList *candidate_list = self->local_candidates;


  g_debug ("Looking for local ip");

  for (lp = candidate_list; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;
    if (trans->type == FARSIGHT_CANDIDATE_TYPE_LOCAL)
    {
      if (trans->ip)
      {
        /* let's ignore loopback */
        if (g_ascii_strcasecmp(trans->ip, "127.0.0.1") != 0)
        {
          g_debug ("Found local_ip %s", trans->ip);
          /* open ports */
          upnp_open_port_all_igds (trans->ip, derived_trans->port, derived_trans->port,
              (derived_trans->proto == FARSIGHT_NETWORK_PROTOCOL_UDP)?
              IPPROTO_UDP:IPPROTO_TCP);
        }
      }
    }
  }
#endif
}

static void
farsight_rtp_stream_upnp_close_ports (FarsightRTPStream *self)
{
#ifdef HAVE_CLINKC
  FarsightTransportInfo *trans = NULL;
  const GList *lp;
  GList *candidate_list = self->local_candidates;

  for (lp = candidate_list; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;
    if (trans->type == FARSIGHT_CANDIDATE_TYPE_DERIVED)
    {
      g_debug ("Found derived ip %s", trans->ip);
      /* close ports */
      upnp_close_port_all_igds (trans->port, 
          (trans->proto == FARSIGHT_NETWORK_PROTOCOL_UDP)?
            IPPROTO_UDP:IPPROTO_TCP);
    }
  }
#endif
}

static void 
farsight_rtp_stream_native_candidate_ready (gpointer stream, 
                                            const GList *candidate)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  FarsightTransportInfo *trans = NULL;
  GList *candidate_copy = NULL;

  g_debug ("Called farsight_rtp_stream_native_candidate_ready");

  if (farsight_rtp_stream_candidate_exists (stream, self->local_candidates,
              candidate))
  {
    g_message ("Native candidate already in list, not adding");
    return;
  }
  else
  {
    g_debug ("Native candidates found, adding to list");
    candidate_copy = farsight_transport_list_copy(candidate);
    self->local_candidates = g_list_concat(self->local_candidates, 
        candidate_copy);
    trans = (FarsightTransportInfo *) candidate_copy->data;
    farsight_stream_signal_new_native_candidate (stream,
            (gchar *)trans->candidate_id);

    if (self->use_upnp)
    {
      /* TODO this assumes the local IP is found before the derived IP */
      /* if not it won't send any request since it needs the local ip */
      if (trans->type == FARSIGHT_CANDIDATE_TYPE_DERIVED)
      {
        farsight_rtp_stream_upnp_send_request (self, trans);
      }
    }
  }

  // farsight_stream_signal_native_candidates_prepared (stream);
}

static gboolean 
farsight_rtp_stream_candidate_exists (FarsightStream *stream, 
                                      const GList *candidate_list, 
                                      const GList *candidate)
{
  FarsightTransportInfo *trans = NULL;
  FarsightTransportInfo *trans2 = NULL;
  const GList *lp = NULL;
  const GList *lp2 = NULL;
  gint i = 0;

  if (candidate_list == NULL || candidate == NULL)
    return FALSE;

  /* we check ip and port */
  for (lp = candidate; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;

    for (lp2 = candidate_list; lp2; lp2 = g_list_next (lp2)) {
      trans2 = (FarsightTransportInfo *) lp2->data;
      if (farsight_transport_are_equal (trans, trans2))
        i++;
    }
  }

  if (i == g_list_length ((GList *)candidate))
    return TRUE;
  else if (i == 0)
    return FALSE;
  else
  {
    g_error("Candidate only partially exists (some components do), \
              this should not happen!");
    return FALSE;
  }
}

static void 
farsight_rtp_stream_socket_state_changed (gpointer stream, gint state)
{
  g_message ("socket state changed to %d", state);

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  if (state == 1)     /* writable */
    {
      if (self->timeout_src)
      {
        g_source_remove (self->timeout_src);
        self->timeout_src = -1;
      }

      /* FIXME: dirty hack */
      farsight_stream_signal_new_active_candidate_pair
        (FARSIGHT_STREAM (stream), "foo", "bar");
      farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTED, 
          FARSIGHT_STREAM_DIRECTION_BOTH);
    }
  else if (state == 0)
    {
      if (self->timeout_src)
      {
        g_source_remove (self->timeout_src);
        self->timeout_src = -1;
      }
      /* Let us set a timer for a timeout on reestablishing a connection */
      self->timeout_src = g_timeout_add (self->conn_timeout * 1000,
          farsight_rtp_stream_connection_timed_out,
          self);

      /* farsight_rtp_stream_stop (FARSIGHT_STREAM (stream)); */
      /* What we really want here is some sort of _pause/_resume functionality
       * in farsight */
      farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTING, 
          FARSIGHT_STREAM_DIRECTION_BOTH);
    }
}

static void 
farsight_rtp_stream_network_error (gpointer stream)
{
  g_warning ("error from jinglep2p socketmanager.");
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream,
          FARSIGHT_STREAM_UNKNOWN_ERROR, 
          "Network error from jinglep2p socketmanager");
}

static gboolean
farsight_rtp_stream_connection_timed_out (gpointer data)
{
  FarsightRTPStream *self = (FarsightRTPStream *) data;
 
  self->timeout_src = -1;

  /* let's check if we are connected yet, throw an error if not */
  if (farsight_stream_get_state (FARSIGHT_STREAM(self)) ==
      FARSIGHT_STREAM_STATE_CONNECTING)
  {
    g_warning ("Could not establish a connection");
    farsight_stream_signal_error (FARSIGHT_STREAM(self),
        FARSIGHT_STREAM_UNKNOWN_ERROR, "Could not establish a connection");
    farsight_rtp_stream_stop (FARSIGHT_STREAM(self));
  }

  return FALSE;
}

/* This is to signal you to prepare the list of transport condidates,
 * e.g. start off any network probing, like STUN
 * */
static void 
farsight_rtp_stream_prepare_transports (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

#ifdef HAVE_JINGLEP2P
  if (self->prepared)
    return;

  g_message ("Init and running jinglep2p to prepare native candiates, using stun %s : %d and turn %s : %d", 
      self->stun_ip, self->stun_port, self->turn_ip, self->turn_port);
  self->socket_client = socketclient_init(self->stun_ip,
          self->stun_port, self->turn_ip, self->turn_port);

  connect_signal_candidates_ready (self->socket_client,
          farsight_rtp_stream_native_candidate_ready, stream);
  connect_signal_socket_state_change (self->socket_client,
          farsight_rtp_stream_socket_state_changed, stream);
  connect_signal_network_error (self->socket_client,
          farsight_rtp_stream_network_error, stream);

  socketclient_create_socket (self->socket_client, "rtp");

  socketclient_start_processing_candidates (self->socket_client);

  farsight_stream_signal_state_changed (stream,
      FARSIGHT_STREAM_STATE_CONNECTING, 
      FARSIGHT_STREAM_DIRECTION_BOTH);
#else
#ifdef HAVE_SOFIASIP
  FarsightTransportInfo *trans;

  su_sockaddr_t *su;
  su_localinfo_t *interfaces, *li;
  su_getlocalinfo (NULL, &interfaces);
  char buf[256];

  if (self->prepared)
    return;

  /* TODO : replace by farsight net socket api from kai */
  struct sockaddr_in address;
  gint sock;
  guint16 port = self->base_local_port;
  gint bindresult = -1;
  while (bindresult != 0)
  {
    if ((sock = socket (AF_INET,SOCK_DGRAM,0)) > 0)
        g_debug ("The socket was created");
    else
        g_debug ("Error creating socket");
    address.sin_family = AF_INET;
    address.sin_addr.s_addr = INADDR_ANY;
    address.sin_port = htons (port);
    bindresult = bind (sock, (struct sockaddr *) &address, sizeof (address));
    if (bindresult != 0)
    {
      g_debug ("could not bind port %d", port);
      port++;
    }
  }
  g_debug ("bound to port %d", port);
  self->local_socket = sock;
  self->local_port = port;
  /* \ TODO */

  gint i = 1;
  for (li = interfaces; li; li=li->li_next) {
    if (li->li_family == AF_INET) {
      su = li->li_addr;
      trans = g_new0 (FarsightTransportInfo,1);
      trans->candidate_id = g_strdup_printf ("L%d", i);
      trans->component = 1;
      trans->ip = g_strdup (inet_ntop (su->su_family, SU_ADDR(su), 
                                       buf, sizeof(buf)));
      trans->port = self->local_port;
      trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
      trans->proto_subtype = g_strdup ("RTP");
      trans->proto_profile = g_strdup ("AVP");
      trans->preference = 1.0;
      trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;
      self->local_candidates = g_list_prepend (self->local_candidates, trans);
      i++;
      farsight_stream_signal_new_native_candidate (stream, (gchar *) trans->candidate_id);
    }
  }
  self->local_candidates = g_list_reverse (self->local_candidates);
  self->prepared = TRUE;

  farsight_stream_signal_native_candidates_prepared (stream); 
#else
  g_debug ("Getting local interfaces");
  FarsightTransportInfo *trans;
  GList *ips = NULL;
  char buf[256];

  ips = farsight_get_local_ips(FALSE);

  if (self->prepared)
    return;

  /* TODO : replace by farsight net socket api from kai */
  struct sockaddr_in address;
  gint sock;
  guint16 port = self->base_local_port;
  gint bindresult = -1;
  while (bindresult != 0)
  {
    if ((sock = socket (AF_INET,SOCK_DGRAM,0)) > 0)
        g_debug ("The socket was created");
    else
        g_debug ("Error creating socket");
    address.sin_family = AF_INET;
    address.sin_addr.s_addr = INADDR_ANY;
    address.sin_port = htons (port);
    bindresult = bind (sock, (struct sockaddr *) &address, sizeof (address));
    if (bindresult != 0)
    {
      g_debug ("could not bind port %d", port);
      port++;
    }
  }
  g_debug ("bound to port %d", port);
  self->local_socket = sock;
  self->local_port = port;
  /* \ TODO */

  gint i = 1;
  GList *current;
  for (current = ips; current; current = g_list_next(current)) {
    trans = g_new0 (FarsightTransportInfo,1);
    trans->candidate_id = g_strdup_printf ("L%d", i);
    trans->component = 1;
    trans->ip = g_strdup (current->data);
    trans->port = self->local_port;
    trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
    trans->proto_subtype = g_strdup ("RTP");
    trans->proto_profile = g_strdup ("AVP");
    trans->preference = 1.0;
    trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;
    self->local_candidates = g_list_prepend (self->local_candidates, trans);
    i++;
    farsight_stream_signal_new_native_candidate (stream, (gchar *) trans->candidate_id);
  }
  self->local_candidates = g_list_reverse (self->local_candidates);
  self->prepared = TRUE;

  farsight_stream_signal_native_candidates_prepared (stream); 

  /* free list of ips */
  g_list_foreach (ips, (GFunc) g_free, NULL);
  g_list_free (ips);
#endif
#endif

  /* Let us set a timer for a timeout on establishing a connection */
  self->timeout_src = g_timeout_add (self->conn_timeout * 1000,
      farsight_rtp_stream_connection_timed_out,
      self);

  /* We can already create the core pipeline now */
  farsight_rtp_stream_build_base_pipeline (self);
}

static gboolean 
farsight_rtp_stream_add_remote_candidate_to_rtpbin (FarsightRTPStream *self, 
                                                    const gchar *remote_candidate_id)
{
  GList *remote_candidate;
  gchar *addr = NULL;
  FarsightTransportInfo *trans = NULL;
  const GList *lp;

  if (self->rtpbin)
  {
    remote_candidate = farsight_transport_get_list_for_candidate_id
        (self->remote_candidates, remote_candidate_id);
    if (remote_candidate == NULL)
      return FALSE;

    /* Find remote candidate that is of subtype RTP */
    for (lp = remote_candidate; lp; lp = g_list_next (lp)) {
      trans = (FarsightTransportInfo *) lp->data;
      if (g_ascii_strcasecmp(trans->proto_subtype, "RTP") == 0)
      {
        break;
      }
    }
    if (trans == NULL)
      return FALSE;
    /* Set the ip:port of that transport to rtpbin */
    /* TODO there should be a way to set alternative RTCP ip:port that are
     * not simply ip:port+1 */
    addr = g_strdup_printf ("%s:%d", trans->ip, trans->port);
    g_object_set (G_OBJECT (self->rtpbin),
            "destinations", addr, NULL);

    g_free(addr);
    return TRUE;
  }
  return FALSE;
}

static gboolean 
farsight_rtp_stream_set_active_candidate_pair (FarsightStream *stream,
                                               const gchar *native_candidate_id, 
                                               const gchar *remote_candidate_id)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  self->active_native_candidate = g_strdup (native_candidate_id);
  self->active_remote_candidate = g_strdup (remote_candidate_id);

  /* this will not work if the pipeline has not yet been created */
  if (!farsight_rtp_stream_add_remote_candidate_to_rtpbin (self,
              remote_candidate_id))
    return FALSE;

  return TRUE;
}

static GList *
farsight_rtp_stream_get_native_candidate (FarsightStream *stream, 
                                          const gchar *candidate_id)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  GList *candidate = NULL;

  candidate = farsight_transport_get_list_for_candidate_id
          (self->local_candidates, candidate_id);

  return candidate;
}

static G_CONST_RETURN GList *
farsight_rtp_stream_get_native_candidate_list (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  return self->local_candidates;
}

static void 
farsight_rtp_stream_set_remote_candidate_list (FarsightStream *stream, 
                                               const GList *remote_candidates)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  self->remote_candidates = farsight_transport_list_copy(remote_candidates);
}

static void 
farsight_rtp_stream_add_remote_candidate (FarsightStream *stream,
                                          const GList *remote_candidate)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  FarsightTransportInfo *info = (FarsightTransportInfo *)remote_candidate->data;
  g_debug ("%s (%d): adding remote candidate %s %d",
          __FUNCTION__, __LINE__, info->ip, info->port);
  GList *rc_copy = farsight_transport_list_copy (remote_candidate);

  if (self->remote_candidates == NULL)
  {
    self->remote_candidates = rc_copy;
  }
  else
  {
    if (farsight_rtp_stream_candidate_exists (stream,
                self->remote_candidates, rc_copy))
    {
      g_message ("Remote candidate already in list, not adding");
      return;
    }
    else
    {
      self->remote_candidates = g_list_concat(self->remote_candidates, 
                rc_copy);
      g_debug ("%s (%d): Added remote candidate",
                __FUNCTION__, __LINE__);
    }
  }

/* if using jingle, we need to add the candidates to the socket */
#ifdef HAVE_JINGLEP2P
  if (self->socket_client)
    socketclient_add_remote_candidates(self->socket_client,
              rc_copy);
#endif
}
