/*
 * rtpgstcodecs.h - Headers for RTP gstreamer codec setting
 *
 * Farsight RTP/AVP/SAVP/AVPF Module
 * Copyright (C) 2005,2006 Collabora Ltd.
 * Copyright (C) 2005,2006 Nokia Corporation
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>
 *   @author Philippe Kalaf <philippe.kalaf@collabora.co.uk>
 * Copyright (C) 2005 INdT 
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "rtpgstcodecs.h"

#include <string.h>

static gboolean check_for_sink (GList *pipeline);
static gboolean check_for_src (GList *pipeline);

/* the filter function for selecting the elements we can use in
 * autoplugging */
typedef gboolean (FilterFunc) (const gchar *klass);

static gboolean is_encoder (const gchar *klass);
static gboolean is_decoder (const gchar *klass);
static gboolean registry_filter (GstPluginFeature *feature, 
                                 FilterFunc filter);
static gboolean element_is_raw (GstElementFactory *element, 
                                GstPadDirection direction);
static gint compare_ranks_cb (GstPluginFeature *f1, 
                              GstPluginFeature *f2);
static void debug_pipeline (GList *pipeline);
static GList *get_plugins_from_caps (const gchar *klass, 
                                     GstCaps *caps, 
                                     GstPadDirection direction);
static GstElementFactory *find_match (GList *walk1, 
                                      GstElementFactory *element, 
                                      GstPadDirection direction);
static gboolean check_element_compatibility (GstElementFactory *factory1,
                                             GstPadDirection direction1,
                                             GstElementFactory *factory2, 
                                             GstPadDirection direction2);
static gboolean check_caps_compatibility (GstElementFactory *factory,
                                          GstPadDirection direction, 
                                          GstCaps *caps);
static FarsightCodec *create_codec_from_payload (GstElementFactory *payloader,
                                                 GstCaps *rtp_caps);
static gboolean extract_field_data (GQuark field_id, 
                                    const GValue *value, 
                                    gpointer user_data);
static void set_codec_preference_order (GList **list_codecs, 
                                        const FarsightCodec *codec,
                                        guint pos);

static GList *list_codecs[FARSIGHT_MEDIA_TYPE_LAST] = { NULL };
static GHashTable *list_codecs_internal[FARSIGHT_MEDIA_TYPE_LAST] = { NULL };

/**
 * load_codecs:
 * @media_type: a #FarsightMediaType
 *
 * find all plugins that follow the pattern:
 * input (microphone) -> N* -> rtp payloader -> network
 * network  -> rtp depayloader -> N* -> output (soundcard) 
 * media_type defines if we want audio or video codecs
 *
 * Returns : a #GList of codecs
 */
const GList *
load_codecs (FarsightMediaType media_type, 
             const GHashTable **list_codecs_internal_ret)
{
  GstCaps *caps;
  GList *payloaders, *depayloaders, *pdepayloaders, *encoders, *decoders;
  GstElementFactory *payloader, *depayloader;
  FarsightCodec *codec;
  CodecInternal *codec_internal;

  if (list_codecs[media_type]) {
    *list_codecs_internal_ret = list_codecs_internal[media_type]; 
    return list_codecs[media_type];
  }

  /* caps used to find the payloaders and depayloaders */
  if (media_type == FARSIGHT_MEDIA_TYPE_AUDIO)
  {
    caps = gst_caps_new_simple ("application/x-rtp",
            "media", G_TYPE_STRING, "audio", NULL);
  }
  else if (media_type == FARSIGHT_MEDIA_TYPE_VIDEO)
  {
    caps = gst_caps_new_simple ("application/x-rtp",
            "media", G_TYPE_STRING, "video", NULL);
  }
  else
  {
    g_warning ("Invalid media type give to load_codecs");
    return NULL;
  }

  /* find all payloaders. All payloaders should be from klass Codec/Encoder/Network and
   * have as output a data of the mimetype application/x-rtp */
  payloaders = get_plugins_from_caps ("Codec/Payloader/Network",
      caps, GST_PAD_SRC);

  /* no payloader found. giving up */
  if (!payloaders)
  {
    g_warning ("No RTP Payloaders found");
    return NULL;
  }

  /* find all depayloaders. All depayloaders should be from klass Codec/Parser/Network and
   * have as input a data of the mimetype application/x-rtp */
  depayloaders = get_plugins_from_caps ("Codec/Depayr/Network",
      caps, GST_PAD_SINK);

  /* no depayloader found. giving up */
  if (!depayloaders)
  {
    g_list_free (payloaders);
    g_warning ("No RTP Depayloaders found");
    return NULL;
  }

  list_codecs_internal[media_type] = g_hash_table_new_full (g_direct_hash,
      g_direct_equal, NULL, g_free);

  // maybe we should just check if the payloaders/depayloaders are valid (compliant
  // with http://cvs.freedesktop.org/gstreamer/gst-plugins-good/gst/rtp/README)
  // before searching for decoders/encoders

  encoders = gst_default_registry_feature_filter (
      (GstPluginFeatureFilter) registry_filter,
      FALSE, is_encoder);
  decoders = gst_default_registry_feature_filter (
      (GstPluginFeatureFilter) registry_filter,
      FALSE, is_decoder);

  /* sort them according to their ranks */
  payloaders =
      g_list_sort (payloaders, (GCompareFunc) compare_ranks_cb);
  depayloaders =
      g_list_sort (depayloaders, (GCompareFunc) compare_ranks_cb);
  encoders = g_list_sort (encoders, (GCompareFunc) compare_ranks_cb);
  decoders = g_list_sort (decoders, (GCompareFunc) compare_ranks_cb);

  /* first check if there is a depayloader for each payloader, then we 
   * try to find the encoders/muxers and decoders/demuxers that can link to them */
  while (payloaders)
  {
    payloader = payloaders->data;

    pdepayloaders = depayloaders;
    while (pdepayloaders)
    {
      depayloader = pdepayloaders->data;
      /* check if there is a depayloader for each payloader */
      if (check_element_compatibility (payloader, GST_PAD_SRC,
              depayloader, GST_PAD_SINK))
      {
        GList *send_pipeline = NULL;
        GList *receive_pipeline = NULL;
        gboolean has_sink = FALSE;
        gboolean has_src = FALSE;
        GstElementFactory *match;

        // build send pipeline
        match = payloader;
        send_pipeline = g_list_prepend (send_pipeline, match);
        do {
          match = find_match (encoders, match, GST_PAD_SINK);
          if (match)
          {
            send_pipeline = g_list_prepend (send_pipeline, match);
          }
        } while (match && !element_is_raw (match, GST_PAD_SINK));

        if (!match)
        {
          g_list_free (send_pipeline);
          depayloaders = g_list_delete_link (depayloaders, pdepayloaders);
          break;
        }
        debug_pipeline (send_pipeline);
        has_src = check_for_src (send_pipeline);

        // build receive pipeline
        match = depayloader;
        receive_pipeline = g_list_append (receive_pipeline, match);
        do
        {
          match = find_match (decoders, match, GST_PAD_SRC);
          if (match)
          {
            receive_pipeline = g_list_append (receive_pipeline, match);
          }
        } while (match && !element_is_raw (match, GST_PAD_SRC));

        if (!match)
        {
          g_list_free (send_pipeline);
          g_list_free (receive_pipeline);
          depayloaders = g_list_delete_link (depayloaders, pdepayloaders);
          break;
        }
        debug_pipeline (receive_pipeline);
        has_sink = check_for_sink (receive_pipeline);

        codec = create_codec_from_payload (payloader, caps);
        g_debug ("create_codec_from_payload returned %p", codec);
        if (!codec)
        {
          depayloaders = g_list_delete_link (depayloaders, pdepayloaders);
          break;
        }

        codec_internal = g_new0 (CodecInternal, 1);
        codec_internal->codec = codec;
        codec_internal->send_pipeline_factory = send_pipeline;
        codec_internal->receive_pipeline_factory = receive_pipeline;
        codec_internal->has_sink = has_sink;
        codec_internal->has_src = has_src;

        g_hash_table_insert (list_codecs_internal[media_type],
            (gpointer) codec->id, (gpointer) codec_internal);
        g_debug ("adding codec with pt %d, send_pipeline %p, receive_pipeline %p", codec->id, send_pipeline, receive_pipeline);
        list_codecs[media_type] = g_list_append (list_codecs[media_type], codec);

        depayloaders = g_list_delete_link (depayloaders, pdepayloaders);
        break;
      }

      pdepayloaders = g_list_next (pdepayloaders);
    }

    payloaders = g_list_delete_link (payloaders, payloaders);
  }

  g_list_free (payloaders);
  g_list_free (depayloaders);
  gst_caps_unref (caps);

  if (list_codecs_internal[media_type])
  {
    *list_codecs_internal_ret = list_codecs_internal[media_type];
  }
  
  return list_codecs[media_type];
}

void
unload_codecs (FarsightMediaType media_type)
{
  if (list_codecs[media_type]) {
    farsight_codec_list_destroy (list_codecs[media_type]);
    list_codecs[media_type] = NULL;
    g_hash_table_destroy (list_codecs_internal[media_type]);
    list_codecs_internal[media_type] = NULL;
  }
}

void
sort_codecs (GList **list_codecs, 
             FarsightCodecPreference codec_preference[],
             guint size)
{
  guint8 i = 0, pos = 0;
  GList *lc;
  FarsightCodec *codec;
  guint8 count;

  count = size;
  for (; i < count; ++i)
  {
    for (lc = *list_codecs; lc; lc = g_list_next (lc))
    {
      codec = (FarsightCodec *) lc->data;
      /* only match clock_rate if it is different than 0 */
      if ((g_strcasecmp (codec->encoding_name,
              codec_preference[i].encoding_name) == 0) &&
              (codec->clock_rate == 0 || 
               codec->clock_rate == codec_preference[i].clock_rate))
      {
        set_codec_preference_order (list_codecs, codec, pos++);
        break;
      }
    }
  }
}

GHashTable *create_pt_caps_hashtable(GList *codecs)
{
  FarsightCodec *codec = NULL;
  GList *lp = NULL;
  GstCaps *caps = NULL;

  GHashTable *pt_map = g_hash_table_new_full (g_direct_hash,
          g_direct_equal,
          NULL,
          (GDestroyNotify) gst_caps_unref);

  for(lp = codecs; lp; lp = g_list_next(lp))
  {
      codec = (FarsightCodec *)lp->data;
      caps = gst_caps_new_simple ("application/x-rtp",
              "clock-rate", G_TYPE_INT, codec->clock_rate, NULL);
      g_hash_table_insert(pt_map, GINT_TO_POINTER(codec->id), (gpointer) caps);
  }

  return pt_map;
}

static void
set_codec_preference_order (GList **list_codecs, 
                            const FarsightCodec *codec,
                            guint pos)
{
  gint index = 0;

  g_return_if_fail (codec != NULL);
  g_return_if_fail (pos <= g_list_length (*list_codecs));

  GList *lc;
  GList *codec_item = NULL;

  for (lc = *list_codecs; lc; lc = g_list_next (lc), ++index)
  {
    if (lc->data == codec) 
    {
      codec_item = lc;
    }
  }

  if (!codec_item)
  {
    g_print ("%s (%d): codec not supported\n", __FUNCTION__, __LINE__);
    return;
  }

  // do nothing, codec already sorted
  if (index == pos)
  {
    return;
  }

  *list_codecs = g_list_delete_link (*list_codecs, codec_item);
  *list_codecs = g_list_insert_before (*list_codecs, 
          g_list_nth (*list_codecs, pos),
          (gpointer) codec);
}

static gboolean
is_encoder (const gchar *klass)
{
  if (getenv("FS_FAKESTREAM"))
  {
    /* we might have some sources that provide a non raw stream */
    return (g_strrstr (klass, "Encoder") != NULL || g_strrstr (klass, "Muxer") != NULL);
  }
  else
  {
    /* we might have some sources that provide a non raw stream */
    return (g_strrstr (klass, "Encoder") != NULL || g_strrstr (klass, "Muxer") != NULL
            || g_strrstr (klass, "Source") != NULL);
  }
}

static gboolean
is_decoder (const gchar *klass)
{
  if (getenv("FS_FAKESTREAM"))
  {
      /* we might have some sinks that provide decoding */
    return (g_strrstr (klass, "Decoder") != NULL ||
            g_strrstr (klass, "Demuxer") != NULL || g_strrstr (klass, "Parser") != NULL);
  }
  else
  {
    /* we might have some sinks that provide decoding */
    return (g_strrstr (klass, "Decoder") != NULL ||
            g_strrstr (klass, "Demuxer") != NULL || g_strrstr (klass, "Parser") != NULL 
            || g_strrstr (klass, "Sink") != NULL);
  }
}

static gboolean
registry_filter (GstPluginFeature *feature, FilterFunc filter)
{
  // guint rank;
  const gchar *klass;

  /* we only care about element factories */
  if (!GST_IS_ELEMENT_FACTORY (feature))
  {
    return FALSE;
  }

  klass = gst_element_factory_get_klass (GST_ELEMENT_FACTORY (feature));
  if (!filter (klass))
  {
    return FALSE;
  }

  // /* only select elements with autoplugging rank */
  // rank = gst_plugin_feature_get_rank (feature);
  // if (rank < GST_RANK_MARGINAL) {
  //     return FALSE;
  // }

  return TRUE;
}

static gboolean
element_is_raw (GstElementFactory *element, GstPadDirection direction)
{
  const GList *pads;
  GstStaticPadTemplate *padtemplate;
  GstCaps *caps;
  GstStructure *structure;
  const gchar *mimetype;

  /* we don't need this if the element is a sink or a source */
  const gchar *class;
  class = gst_element_factory_get_klass (element);
  if (g_strrstr (class, "Source") != NULL || g_strrstr (class, "Sink") != NULL)
  {
    return TRUE;
  }

  for (pads = element->staticpadtemplates; pads; pads = g_list_next (pads))
  {
    padtemplate = (GstStaticPadTemplate *) pads->data;
    if (padtemplate->direction != direction)
    {
      continue;
    }

    caps = gst_static_caps_get (&padtemplate->static_caps);
    structure = gst_caps_get_structure (caps, 0);
    gst_caps_unref (caps);
    mimetype = gst_structure_get_name (structure);

    if (g_str_has_prefix (mimetype, "audio/x-raw"))
    {
      return TRUE;
    }

    if (g_str_has_prefix (mimetype, "video/x-raw"))
    {
      return TRUE;
    }
  }
  return FALSE;
}

static gint
compare_ranks_cb (GstPluginFeature *f1, GstPluginFeature *f2)
{
  return gst_plugin_feature_get_rank (f2) - gst_plugin_feature_get_rank (f1);
}

static gboolean
check_for_src (GList *pipeline)
{
  GList *walk;
  const gchar *class;

  for (walk = pipeline; walk; walk = g_list_next (walk))
  {
    class = gst_element_factory_get_klass (GST_ELEMENT_FACTORY (walk->data));
    if (g_strrstr (class, "Source") != NULL)
    {
      return TRUE;
    }
  }
  return FALSE;
}

static gboolean
check_for_sink (GList *pipeline)
{
  GList *walk;
  const gchar *class;

  for (walk = pipeline; walk; walk = g_list_next (walk))
  {
      class = gst_element_factory_get_klass (GST_ELEMENT_FACTORY (walk->data));
      if (g_strrstr (class, "Sink") != NULL)
      {
          return TRUE;
      }
  }
  return FALSE;
}

static void
debug_pipeline (GList *pipeline)
{
  GList *walk;

  g_print ("pipeline: ");
  for (walk = pipeline; walk; walk = g_list_next (walk))
  {
    g_print ("%s ",
        gst_plugin_feature_get_name (GST_PLUGIN_FEATURE (walk->data)));
  }
  g_print ("\n");
}

static GList *
get_plugins_from_caps (const gchar *klass,
                       GstCaps *caps, 
                       GstPadDirection direction)
{
  GList *walk;
  GstElementFactory *factory;
  GList *list = NULL;

  walk =
      gst_registry_get_feature_list (gst_registry_get_default (),
      GST_TYPE_ELEMENT_FACTORY);

  while (walk)
  {
    factory = GST_ELEMENT_FACTORY (walk->data);

    if (klass
        && strcasecmp (klass, gst_element_factory_get_klass (factory)) != 0)
    {
      walk = g_list_next (walk);
      continue;
    }

    if (check_caps_compatibility (factory, direction, caps))
    {
      list = g_list_append (list, factory);
    }

    walk = g_list_next (walk);
  }

  gst_plugin_feature_list_free (walk);

  return list;
}

static GstElementFactory *
find_match (GList *walk1, 
            GstElementFactory *element, 
            GstPadDirection direction)
{
  GList *walk = walk1;
  GstElementFactory *factory, *found_factory = NULL;
  const GList *pads, *src_pads;
  GstCaps *caps1, *caps2, *intersect;
  GstStaticPadTemplate *padtemplate;
  gboolean can_link;
  GstPadDirection other_direction;

  if (direction == GST_PAD_SINK)
  {
    other_direction = GST_PAD_SRC;
  }
  else
  {
    other_direction = GST_PAD_SINK;
  }

  while (walk)
  {
    factory = GST_ELEMENT_FACTORY (walk->data);

    for (pads = element->staticpadtemplates; pads; pads = g_list_next (pads))
    {
      padtemplate = (GstStaticPadTemplate *) pads->data;
      if (padtemplate->direction != direction)
      {
        continue;
      }

      caps1 = gst_static_caps_get (&padtemplate->static_caps);
      /* ignore caps ANY */
      if (gst_caps_is_any (caps1))
      {
        gst_caps_unref (caps1);
        continue;
      }

      can_link = FALSE;
      for (src_pads = factory->staticpadtemplates; src_pads;
          src_pads = g_list_next (src_pads))
      {
        GstStaticPadTemplate *src_padtemplate =
            (GstStaticPadTemplate *) (src_pads->data);
        if (src_padtemplate->direction != other_direction)
        {
          continue;
        }

        caps2 = gst_static_caps_get (&src_padtemplate->static_caps);
        /* ignore caps ANY */
        if (gst_caps_is_any (caps2))
        {
          gst_caps_unref (caps2);
          continue;
        }

        intersect = gst_caps_intersect (caps1, caps2);
        if (!gst_caps_is_empty (intersect))
        {
          /* 770 code, since there are Src and Sink encoders/decoders, we want
           * to give them priority on anything else */
          if (g_strrstr (gst_element_factory_get_klass (factory), "Sink") !=
              NULL ||
              g_strrstr (gst_element_factory_get_klass (factory), "Source") !=
              NULL)
          {
            gst_caps_unref (intersect);
            gst_caps_unref (caps1);
            gst_caps_unref (caps2);
            return factory;
          }
          else
          {
            found_factory = factory;
          }
        }
 
        gst_caps_unref (intersect);
        gst_caps_unref (caps2);
      }      
      gst_caps_unref (caps1);
    }

    walk = g_list_next (walk);
  }

  return found_factory;
}

static gboolean
check_element_compatibility (GstElementFactory *factory1,
                             GstPadDirection direction1,
                             GstElementFactory *factory2, 
                             GstPadDirection direction2)
{
  const GList *pads1, *pads2;
  GstStaticPadTemplate *padtemplate1, *padtemplate2;
  GstCaps *caps1, *caps2, *intersect;

  if (!factory1->numpadtemplates || !factory2->numpadtemplates)
  {
    return FALSE;
  }

  pads1 = factory1->staticpadtemplates;
  while (pads1)
  {
    padtemplate1 = (GstStaticPadTemplate *) (pads1->data);
    pads1 = g_list_next (pads1);

    if (padtemplate1->direction != direction1)
    {
      continue;
    }

    caps1 = gst_static_caps_get (&padtemplate1->static_caps);
    if (gst_caps_is_any (caps1))
    {
      gst_caps_unref (caps1);
      return TRUE;
    }

    pads2 = factory2->staticpadtemplates;
    while (pads2)
    {
      padtemplate2 = (GstStaticPadTemplate *) (pads2->data);
      pads2 = g_list_next (pads2);

      if (padtemplate2->direction != direction2)
      {
        continue;
      }

      caps2 = gst_static_caps_get (&padtemplate2->static_caps);
      if (gst_caps_is_any (caps2))
      {
        gst_caps_unref (caps2);
        return TRUE;
      }

      intersect = gst_caps_intersect (caps1, caps2);
      if (!gst_caps_is_empty (intersect))
      {
        gst_caps_unref (intersect);
        gst_caps_unref (caps1);
        gst_caps_unref (caps2);
        return TRUE;
      }
      gst_caps_unref (intersect);
      gst_caps_unref (caps1);
      gst_caps_unref (caps2);
    }
  }

  return FALSE;
}

/* check if the factory is compatible with the caps using 
 * the right direction */
static gboolean
check_caps_compatibility (GstElementFactory *factory,
                          GstPadDirection direction, 
                          GstCaps *caps)
{
  const GList *pads;
  GstStaticPadTemplate *padtemplate;
  GstCaps *padtemplate_caps;

  if (!factory->numpadtemplates)
  {
    return FALSE;
  }

  pads = factory->staticpadtemplates;
  while (pads)
  {
    padtemplate = (GstStaticPadTemplate *) (pads->data);
    pads = g_list_next (pads);

    if (padtemplate->direction != direction)
    {
      continue;
    }

    padtemplate_caps = gst_static_caps_get (&padtemplate->static_caps);
    if (gst_caps_is_any (padtemplate_caps))
    {
      gst_caps_unref (padtemplate_caps);
      continue;
    }

    if (gst_caps_is_subset (padtemplate_caps, caps))
    {
      gst_caps_unref (padtemplate_caps);
      return TRUE;
    }

    /*intersect = gst_caps_intersect (caps, padtemplate_caps);
       if (!gst_caps_is_empty (intersect)) {
       gst_caps_unref (intersect);
       return TRUE;
       }
       gst_caps_unref (intersect); */

    gst_caps_unref (padtemplate_caps);
  }

  return FALSE;
}

static FarsightCodec *
create_codec_from_payload (GstElementFactory *payloader,
                           GstCaps *rtp_caps)
{
  FarsightCodec *codec;
  const GList *pads;
  GstStaticPadTemplate *padtemplate;
  GstCaps *caps, *intersect;
  guint i;

  if (!payloader->numpadtemplates)
  {
    return FALSE;
  }

  pads = payloader->staticpadtemplates;
  while (pads)
  {
    padtemplate = (GstStaticPadTemplate *) (pads->data);

    if (padtemplate->direction != GST_PAD_SRC)
    {
      pads = g_list_next (pads);
      continue;
    }

    caps = gst_static_caps_get (&padtemplate->static_caps);
    if (gst_caps_is_any (caps))
    {
      gst_caps_unref (caps);
      pads = g_list_next (pads);
      continue;
    }

    intersect = gst_caps_intersect (rtp_caps, caps);
    if (!gst_caps_is_empty (intersect))
    {
      gst_caps_unref (intersect);
      gst_caps_unref (caps);
      break;
    }
    gst_caps_unref (intersect);
    gst_caps_unref (caps);

    pads = g_list_next (pads);
  }

  if (!pads)
  {
    return NULL;
  }

  codec = g_new0 (FarsightCodec, 1);
  codec->id = -1;
  codec->clock_rate = -1;
  for (i = 0; i < gst_caps_get_size (caps); i++)
  {
    GstStructure *structure = gst_caps_get_structure (caps, i);

    if (!gst_structure_foreach (structure, extract_field_data,
            (gpointer) codec))
    {
      farsight_codec_destroy (codec);
      return NULL;
    }
  }

  if (codec->id == -1 || codec->clock_rate == -1 || !codec->encoding_name)
  {
    farsight_codec_destroy (codec);
    return NULL;
  }

  /*codec->enabled = TRUE;*/

  return codec;
}

/**
 *  fill FarsightCodec fields based on payloader capabilities 
 *  TODO: optimise using quarks
 */
static gboolean
extract_field_data (GQuark field_id,
                    const GValue *value, 
                    gpointer user_data)
{
  FarsightCodec *codec = (FarsightCodec *) user_data;
  GType type = G_VALUE_TYPE (value);
  const gchar *field_name = g_quark_to_string (field_id);
  const gchar *tmp;
  static guint pt_count = 96;

  if (0 == strcmp (field_name, "media"))
  {
    if (type != G_TYPE_STRING)
    {
      return FALSE;
    }
    tmp = g_value_get_string (value);
    if (strcmp (tmp, "audio") == 0) 
    {
      codec->media_type = FARSIGHT_MEDIA_TYPE_AUDIO;
    }
    else if (strcmp (tmp, "video") == 0) 
    {
      codec->media_type = FARSIGHT_MEDIA_TYPE_VIDEO;
    }

  } 
  else if (0 == strcmp (field_name, "payload"))
  {
    if (type == GST_TYPE_INT_RANGE)
    {
      if (gst_value_get_int_range_min (value) < 96 ||
          gst_value_get_int_range_max (value) > 255 || pt_count == 255)
      {
        return FALSE;
      }

      codec->id = pt_count++;
    } 
    else if (type == G_TYPE_INT)
    {
      codec->id = g_value_get_int (value);
    }
    else
    {
      return FALSE;
    }
  }
  else if (0 == strcmp (field_name, "clock-rate"))
  {
    if (type != G_TYPE_INT)
    {
      return FALSE;
    }
    codec->clock_rate = g_value_get_int (value);
  } 
  else if (0 == strcmp (field_name, "ssrc") ||
      0 == strcmp (field_name, "clock-base") ||
      0 == strcmp (field_name, "seqnum-base"))
  {
    // ignore these fields for now
    ;
  } 
  else if (0 == strcmp (field_name, "encoding-name"))
  {
    if (type != G_TYPE_STRING)
    {
      return FALSE;
    }
    codec->encoding_name = g_strdup (g_value_get_string (value));
  } 
  else if (0 == strcmp (field_name, "encoding-params"))
  {
    if (type != G_TYPE_STRING)
    {
      return FALSE;
    }
    codec->channels = (guint) g_ascii_strtoull (
                                       g_value_get_string (value), NULL, 10);
  } 
  else 
  {
    if (type == G_TYPE_STRING)
    {
      FarsightCodecParameter *optional_param = 
        g_new (FarsightCodecParameter, 1);

      optional_param->name = g_strdup (field_name);
      optional_param->value = g_strdup (g_value_get_string (value));
      codec->optional_params = g_list_append (codec->optional_params,
          optional_param);
    }
  }

  return TRUE;
}
