/*
 * farsight-stream.h - Header for core API streams
 *
 * Farsight Voice+Video library
 * Copyright 2005,2006 Collabora Ltd.
 * Copyright 2005,2006 Nokia Corp.
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>.
 *   @author Philippe Kalaf <philippe.kalaf@collabora.co.uk>
 * Copyright (c) 2005 INdT
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __FARSIGHT_STREAM_H__
#define __FARSIGHT_STREAM_H__

#include <glib.h>
#include <glib-object.h>
#include <gst/gstelement.h>

#include "farsight-codec.h"

G_BEGIN_DECLS

/* TYPE MACROS */
#define FARSIGHT_TYPE_STREAM \
  (farsight_stream_get_type())
#define FARSIGHT_STREAM(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), FARSIGHT_TYPE_STREAM, FarsightStream))
#define FARSIGHT_STREAM_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), FARSIGHT_TYPE_STREAM, FarsightStreamClass))
#define FARSIGHT_IS_STREAM(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), FARSIGHT_TYPE_STREAM))
#define FARSIGHT_IS_STREAM_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), FARSIGHT_TYPE_STREAM))
#define FARSIGHT_STREAM_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS ((obj), FARSIGHT_TYPE_STREAM, FarsightStreamClass))

typedef struct _FarsightStream FarsightStream;
typedef struct _FarsightStreamClass FarsightStreamClass;
typedef struct _FarsightStreamPrivate FarsightStreamPrivate;

/* Type of candidate*/
typedef enum
{
  FARSIGHT_CANDIDATE_TYPE_LOCAL,   /* A local address */
  FARSIGHT_CANDIDATE_TYPE_DERIVED, /* An address derived by some method such as 
                                    * STUN which should reach the local client
                                    */
  FARSIGHT_CANDIDATE_TYPE_RELAY    /* An external stream relay */
} FarsightCandidateType;

typedef enum
{
  FARSIGHT_NETWORK_PROTOCOL_UDP,
  FARSIGHT_NETWORK_PROTOCOL_TCP
} FarsightNetworkProtocol;

typedef enum
{
  FARSIGHT_STREAM_DIRECTION_NONE,
  FARSIGHT_STREAM_DIRECTION_SENDONLY,
  FARSIGHT_STREAM_DIRECTION_RECEIVEONLY,
  FARSIGHT_STREAM_DIRECTION_BOTH,
  FARSIGHT_STREAM_DIRECTION_LAST
} FarsightStreamDirection;

typedef enum
{
  FARSIGHT_STREAM_STATE_STOPPED,
  FARSIGHT_STREAM_STATE_PLAYING,
  FARSIGHT_STREAM_STATE_CONNECTING,
  FARSIGHT_STREAM_STATE_CONNECTED
} FarsightStreamState;

typedef enum
{
  FARSIGHT_STREAM_ERROR_EOS,
  FARSIGHT_STREAM_UNKNOWN_ERROR,
  FARSIGHT_STREAM_LAST_ERROR = FARSIGHT_STREAM_UNKNOWN_ERROR
} FarsightStreamError;

struct _FarsightStreamClass
{
  GObjectClass parent_class;

  /*virtual methods */
  void (* prepare_transports) (FarsightStream *self);
  G_CONST_RETURN GList * (* get_native_candidate_list) (FarsightStream *self);
  GList * (* get_native_candidate) (FarsightStream *self,
                                    const gchar* candidate_id);
  void (* set_remote_candidate_list) (FarsightStream *self, 
                                      const GList *remote_candidates);
  void (* add_remote_candidate) (FarsightStream *self,
                                 const GList *remote_candidate);
  void (* remove_remote_candidate) (FarsightStream *self, 
                                    const gchar *remote_candidate_id);
  gboolean (* set_active_candidate_pair) (FarsightStream *self,
                                          const gchar *native_candidate_id,
                                          const gchar *remote_candidiate_id);

  /* void (*set_local_codec_selection) (FarsightStream *self, const GList *ids); */
  G_CONST_RETURN GList * (* get_local_codecs) (FarsightStream *self);
  void (* set_remote_codecs) (FarsightStream *self, 
                              const GList *codecs);
  GList *(* get_codec_intersection) (FarsightStream *self);
  void (* set_codec_preference_list) (FarsightStream *self,
          FarsightCodecPreference codec_pref[]);
  void (* set_active_codec) (FarsightStream *self, 
                             gint id);
  gint (* get_active_codec) (FarsightStream *self);
  gboolean (* set_sink) (FarsightStream *self, 
                         GstElement *sink);
  gboolean (* set_sink_filter) (FarsightStream *self, 
                         GstCaps *filter);
  GstElement * (* get_sink) (FarsightStream *self);
  gboolean (* set_source) (FarsightStream *self, 
                          GstElement *source);
  gboolean (* set_source_filter) (FarsightStream *self, 
                         GstCaps *filter);
  GstElement * (* get_source) (FarsightStream *self);
  gboolean (* start) (FarsightStream *self);
  void (* stop) (FarsightStream *self);

  gpointer _padding[4];         /* padding for binary-compatible expansion */
};

/**
 * FarsightStream:
 *
 * A object that represents and manages a single real-time audio/video stream.
 *
 */
struct _FarsightStream
{
  GObject parent;

  FarsightStreamPrivate *priv;
};

GType farsight_stream_get_type (void);

FarsightMediaType farsight_stream_get_media_type (FarsightStream *self);

void farsight_stream_prepare_transports (FarsightStream *self);

G_CONST_RETURN GList *farsight_stream_get_native_candidate_list (FarsightStream *self);
GList *farsight_stream_get_native_candidate (FarsightStream *self, 
                                             const gchar*candidate_id);

void farsight_stream_set_remote_candidate_list (FarsightStream *self, 
                                                const GList *remote_candidates);
void farsight_stream_add_remote_candidate (FarsightStream *self,
                                           const GList *remote_candidate);
void farsight_stream_remove_remote_candidate (FarsightStream *self, 
                                              const gchar *remote_candidate_id);

gboolean farsight_stream_set_active_candidate_pair (FarsightStream *self,
                                                    const gchar *native_candidate_id,
                                                    const gchar *remote_candidate_id);

/* void farsight_stream_set_local_codecs (FarsightStream *self,
                                          const GList *ids); */
G_CONST_RETURN GList *farsight_stream_get_local_codecs (FarsightStream *self);

void farsight_stream_set_remote_codecs (FarsightStream *self,
                                        const GList *codecs);
/* G_CONST_RETURN GList *farsight_stream_get_remote_codecs (FarsightStream *self); */

GList *farsight_stream_get_codec_intersection (FarsightStream *self);

void farsight_stream_set_codec_preference_list (FarsightStream *self,
        FarsightCodecPreference codec_pref[]);

void farsight_stream_set_active_codec (FarsightStream *self, 
                                       gint id);
gint farsight_stream_get_active_codec (FarsightStream *self);

gboolean farsight_stream_set_sink (FarsightStream *self, 
                                   GstElement *sink);

GstElement *farsight_stream_get_sink (FarsightStream *self);

gboolean farsight_stream_set_source (FarsightStream *self, 
                                     GstElement *source);

GstElement *farsight_stream_get_source (FarsightStream *self);

FarsightStreamState farsight_stream_get_state (FarsightStream *self);
FarsightStreamDirection farsight_stream_get_direction (FarsightStream *self);

gboolean farsight_stream_start (FarsightStream *self);
void farsight_stream_stop (FarsightStream *self);

/* Signals */
void farsight_stream_signal_error (FarsightStream *self, 
                                   FarsightStreamError err,
                                   const gchar *mesg);
void farsight_stream_signal_native_candidates_prepared (FarsightStream *self);
void farsight_stream_signal_new_native_candidate (FarsightStream *self, 
                                                  const gchar *candidate_id);
void farsight_stream_signal_new_active_candidate_pair (FarsightStream *self,
                                                       const gchar *native_candidate_id, 
                                                       const gchar *remote_candidate_id);
void farsight_stream_signal_codec_changed (FarsightStream *self,
                                           int codec_id);
void farsight_stream_signal_state_changed (FarsightStream *self,
                                           FarsightStreamState state,
                                           FarsightStreamDirection direction);

G_END_DECLS
#endif
