/*
 * farsight-codec.c - Source for core API codecs
 *
 * Farsight Voice+Video library
 * Copyright 2005,2006 Collabora Ltd.
 * Copyright 2005,2006 Nokia Corp.
 *   Contact: Rob Taylor <rob.taylor@collabora.co.uk>.
 * Copyright (c) 2005 INdT
 *   Contac: Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "farsight-codec.h"

/**
 * SECTION:farsight-codec
 * @short_description: A descriptor of a codec
 */

void
farsight_codec_init (FarsightCodec *codec, int id, const char *encoding_name,
                     FarsightMediaType media_type, guint clock_rate)
{
  codec->id =id;
  if (encoding_name)
    codec->encoding_name = g_strdup (encoding_name);
  else
    codec->encoding_name = NULL;
  codec->media_type = media_type;
  codec->clock_rate = clock_rate;
}

void
farsight_codec_destroy (FarsightCodec * codec)
{
  if (codec->encoding_name)
    g_free (codec->encoding_name);
  if (codec->optional_params) {
    GList *lp;
    FarsightCodecParameter *optional_param;

    for (lp = codec->optional_params; lp; lp = g_list_next (lp)) {
      optional_param = (FarsightCodecParameter *) lp->data;
      g_free (optional_param->name);
      g_free (optional_param->value);
      g_free (optional_param);
    }
    g_list_free (codec->optional_params);
  }

  g_free (codec);
}

FarsightCodec *
farsight_codec_copy (FarsightCodec * codec)
{
  FarsightCodec *copy = g_new0 (FarsightCodec, 1);

  copy->id = codec->id;
  copy->media_type = codec->media_type;
  copy->clock_rate = codec->clock_rate;
  copy->channels = codec->channels;

  if (codec->encoding_name)
    copy->encoding_name = g_strdup (codec->encoding_name);
  else
    copy->encoding_name = NULL;

  copy->optional_params = NULL;

  if (codec->optional_params) {
    GList *lp;
    FarsightCodecParameter *param;
    FarsightCodecParameter *param_copy;

    for (lp = codec->optional_params; lp; lp = g_list_next (lp)) {
      param_copy = g_new0(FarsightCodecParameter,1);
      param = (FarsightCodecParameter *) lp->data;
      param_copy->name = g_strdup (param->name);
      param_copy->value = g_strdup (param->value);
      /* prepend then reverse the list for efficiency */
      copy->optional_params = g_list_prepend (copy->optional_params, 
                                              param_copy);
    }
    copy->optional_params = g_list_reverse (copy->optional_params);
  }
  return copy;
}

void
farsight_codec_list_destroy (GList *codec_list)
{
  GList *lp;
  FarsightCodec *codec;

  for (lp = codec_list; lp; lp = g_list_next (lp)) {
    codec = (FarsightCodec *) lp->data;
    farsight_codec_destroy(codec);
    lp->data = NULL;
  }
  g_list_free (codec_list);
}

GList *
farsight_codec_list_copy (const GList *codec_list)
{
  GList *copy = NULL;
  const GList *lp;
  FarsightCodec *codec;

  for (lp = codec_list; lp; lp = g_list_next (lp)) {
    codec = (FarsightCodec *) lp->data;
    /* prepend then reverse the list for efficiency */
    copy = g_list_prepend (copy, farsight_codec_copy (codec));
  }
  copy = g_list_reverse (copy);
  return copy;
}
