/**
 * @file galagod.c Galago daemon
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "filter.h"
#include "service-list.h"
#include "person-list.h"
#include "prefix.h"
#include "utils.h"

#include <dbus/dbus-glib-lowlevel.h>

#define GALAGO_DAEMON_ID "#galago-daemon#"

static GMainLoop *loop;
static GalagoContext *context;

void
galagod_shutdown(void)
{
	g_main_loop_quit(loop);
}

GalagoContext *
galagod_get_context(void)
{
	return context;
}

GMainLoop *
galagod_get_main_loop(void)
{
	return loop;
}

static void
launch_feeds_from_dir(const char *path)
{
	GDir *dir;
	GError *error = NULL;

	if (!g_file_test(path, G_FILE_TEST_EXISTS))
		return;

	if (!g_file_test(path, G_FILE_TEST_IS_DIR))
	{
		g_error("%s should be a directory. Please fix this.", path);
		return;
	}

	if ((dir = g_dir_open(path, 0, &error)) != NULL)
	{
		const char *basename;
		char *filename;

		for (basename = g_dir_read_name(dir);
			 basename != NULL;
			 basename = g_dir_read_name(dir))
		{
			filename = g_build_filename(path, basename, NULL);

			if (g_file_test(path, G_FILE_TEST_IS_EXECUTABLE))
			{
				g_message("Launching feed %s", filename);

				if (!g_spawn_command_line_async(filename, &error))
				{
					g_warning("Unable to execute %s: %s",
							  filename, error->message);
				}
			}
			else
			{
				g_warning("%s is not an executable.", filename);
			}

			g_free(filename);
		}

		g_dir_close(dir);
	}
	else
	{
		g_error("Unable to open directory %s: %s", path, error->message);
	}

	if (error != NULL)
		g_error_free(error);
}

void
galagod_launch_feeds(void)
{
	char *path;
	const char *feed_dir;

	feed_dir = BR_LIBDIR("/galago");

	launch_feeds_from_dir(feed_dir);

	if (strcmp(GALAGO_FEED_DIR, feed_dir))
		launch_feeds_from_dir(GALAGO_FEED_DIR);

	path = g_build_path(g_get_home_dir(), ".local" "lib", "galago", NULL);

	if (strcmp(path, feed_dir))
		launch_feeds_from_dir(path);

	g_free(path);
}

void
galagod_init(void)
{
	DBusConnection *dbus_conn;
	DBusError dbus_error;

	loop = g_main_loop_new(NULL, FALSE);

	if (!galago_init(GALAGO_DAEMON_ID, GALAGO_INIT_CLIENT))
	{
		g_critical("Unable to initialize Galago");
		exit(1);
	}

	dbus_conn = galago_get_dbus_conn();

	dbus_connection_set_exit_on_disconnect(dbus_conn, TRUE);

	dbus_error_init(&dbus_error);

	dbus_bus_request_name(dbus_conn, GALAGO_DBUS_SERVICE, 0, &dbus_error);

	if (dbus_error_is_set(&dbus_error))
	{
		g_critical("Unable to acquire service %s: %s",
				   GALAGO_DBUS_SERVICE, dbus_error.message);

		dbus_error_free(&dbus_error);
		galago_uninit();

		exit(1);
	}

	g_random_set_seed(time(NULL));

	context = galago_context_new();
	galago_context_push(context);
	galago_context_set_obj_path_prefix("/org/freedesktop/Galago/Core");

	galagod_cocos_init();
	galagod_services_init();
	galagod_people_init();
	galagod_core_add_filter();
}

void
galagod_uninit(void)
{
	galagod_people_uninit();
	galagod_services_uninit();
	galagod_cocos_uninit();

	galago_context_pop();
}
