/*
 * This file is part of certman
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Ed Bartosh <Eduard.Bartosh@nokia.com>
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
    @file cst_main.c

    Certificate Management Library

    Storage config and etc. function
*/

#include <string.h>
#include "cst_t.h"
#include <gconf/gconf-client.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <errno.h>

static int CST_global_error = CST_ERROR_OK;

static int initial_scan_db(CST * st);
static void cst_all_list_init(CST * st);
static void cst_all_list_destroy(CST * st);

static t_seqnum cst_next(CST * st, char *cnt_key)
{
    TRACE("*");
    g_assert(st);
    g_assert((cnt_key) && (cnt_key[0] != 0));

    DB * db = st->db;
    t_seqnum result = 0;
    int fd = db->fd(db);
    DBT key; 
    memset(&key, 0, sizeof(key));
    DBT data;
    memset(&data, 0, sizeof(data));

    if (fd > 0)
    {
        key.size = strlen(cnt_key);
        key.data = cnt_key;

        int r = db->get(db, &key, &data, 0);
        t_seqnum t = 0;

        if (0 == r) /* record found */
        {
            g_assert(data.size == sz_seqnum);
            memcpy(&t, data.data, sz_seqnum);
            result = SEQNUM_FROM_LE(t);
        }
        else if (r > 0) /* record not found */
        {
            result = 100;
        }

        if (r >= 0)
        {
            data.size = sz_seqnum;
            data.data = g_malloc0(sz_seqnum);
                                            
            t = SEQNUM_TO_LE(result + 1);
            memcpy(data.data, &t, sz_seqnum);
            r = db->put(db, &key, &data, 0);

            if (0 != r)
            {
                result = 0;
            }

            g_free(data.data);
        }

        db->sync(db, 0);
    }

    TRACE_U(result);
    return result;
}

t_seqnum cst_next_cert_uid(CST * st)
{
    g_assert(st);
    return cst_next(st, "NEXTCERTID");
}

t_seqnum cst_next_key_uid(CST * st)
{
    g_assert(st);
    return cst_next(st, "NEXTKEYID");
}

t_seqnum cst_next_crl_uid(CST * st)
{
    g_assert(st);
    return cst_next(st, "NEXTCRLID");
}

/**
    Scan file for make index structure
*/
int initial_scan(CST * st)
{
    TRACE("*");
    g_assert(st);
    g_assert(st->filename);
    return initial_scan_db(st);
}

/**
    Full scan of db
*/    
int initial_scan_db(CST * st)
{
    TRACE("*");

    int result = CST_ERROR_OK;
    
    DB * db = st->db;
    g_assert(db);
    st->modification_count = cst_mcount_get(st);

    DBT key;
    DBT data;
    int r = db->seq(db, &key, &data, R_FIRST);
    while (0 == r)
    {
        /* begin step */
        if (cst_check_suffix(&key, CST_SUFFIX_CERTINFO)) 
        {
            initial_scan_db_cert(st, &key, &data); 
        }
        else if (cst_check_suffix(&key, CST_SUFFIX_KEYINFO))
        {
            initial_scan_db_key(st, &key, &data);
        }
        else if (cst_check_suffix(&key, CST_SUFFIX_CRLINFO))
        {
            initial_scan_db_crl(st, &key, &data);
        }
        /* end step */
        r = db->seq(db, &key, &data, R_NEXT);
    }
    
    if (r < 0)
    {
        result = cst_bdb_error_translate(errno);
    }

    

    return result;
}

DB * cst_dbopen(const char *filename, const int mode)
{
    return dbopen(filename, mode, 0640, DB_HASH, NULL);
}


static void cst_all_list_init(CST * st)
{
    g_assert(st);
    cst_cert_list_init(st);
    cst_key_list_init(st);
    cst_crl_list_init(st);
}

int cst_bdb_error_translate(const int bdberr)
{
    switch (bdberr)
    {
    case 0:      return CST_ERROR_OK;
    case EINVAL: return CST_ERROR_DBSTRUCTURE_CORRUPT;
    case ENOSPC: return CST_ERROR_NOSPC;
    case EIO:    return CST_ERROR_IO; 
    default:     return CST_ERROR_UNDEF_FILE_ERROR;
    }
}

static int cst_file_error_translate(const int err)
{
    switch (err)
    {
    case 0:      return CST_ERROR_OK;
    case ENOSPC: return CST_ERROR_NOSPC;
    case EIO:    return CST_ERROR_IO; 
    default:     return CST_ERROR_UNDEF_FILE_ERROR;
    }
}

/**
    Initialize internal structure
*/
CST *cst_init(int readonly, const char *filename, char *password)
{
    TRACE("*");

    CST * st = NULL;

    DB * db = cst_dbopen(filename, O_RDWR | O_CREAT);
    CST_error(cst_bdb_error_translate(errno));
    
    if (db)
    {
        st = g_malloc0(sizeof(CST));

        db->close(db);

        st->readonly = readonly;
        st->filename = g_strdup(filename);
        st->password = g_strdup(password);

        g_static_rw_lock_init(&st->rwlock);

        cst_all_list_init(st);
        
        CST_error(CST_ERROR_OK);
    }
    return st;
}

static void cst_all_list_destroy(CST * st)
{
    g_assert(st);
    cst_crl_list_destroy(st);
    cst_cert_list_destroy(st);
    cst_key_list_destroy(st);
}

/**
    Destroy internal structure
*/
int cst_destroy(CST * st)
{
    TRACE("*");
    g_assert(st);

    g_static_rw_lock_free(&st->rwlock);
    
    cst_all_list_destroy(st);

    g_free(st->filename);
    g_free(st->password);

    //g_assert(NULL == st->db);
    //g_assert(st->db->close);

    g_assert(NULL == st->db);
    //st->db->close(st->db);
    
    g_free(st);

    return CST_error(CST_ERROR_OK);
}

CST *CST_open(const int readonly, unsigned char *password)
{
    TRACE("*");
    
    GConfClient * client = gconf_client_get_default();
    gchar * filename = NULL;
    gchar * fn = NULL;
    if (client)
    {
        filename = gconf_client_get_string(client, DEFAULT_GCONF_KEY_FILENAME, NULL);
        fn = filename;
        g_object_unref(G_OBJECT(client));
    }

    if (!fn)
    {
        fn = DEFAULT_FILENAME;
    }
    
    CST *st = CST_open_file(fn, readonly, password);

    g_free(filename);
    
    return st;
}

CST *CST_open_file(const char *filename, const int readonly,
                   unsigned char *password)
{
    CST * st = NULL; 
    if (NAME_IS_NOT_EMPTY(filename))
    {
        st = cst_init(readonly, filename, password);

        if (st) 
        {
            int error = CST_ERROR_LOCK;
            
            CST_LOCK_BEGIN_S(LOCK_SH);
                error = initial_scan(st);
            CST_LOCK_END;
            
            if (CST_ERROR_OK != error)
            {
                CST_free(st);
                st = NULL;
            }

            CST_error(error);
        }
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }

    return st;
}

/**
    Create empty DB1 file
*/    
int CST_create_file(const char *filename, unsigned char *password)
{
    if (NAME_IS_NOT_EMPTY(filename))
    {
        /* Remove file if exist */
        DB * db = cst_dbopen(filename, O_RDWR|O_CREAT|O_TRUNC);
        if (db)
        {
            db->close(db);    
            return CST_ERROR_OK;
        }
        else
        {
            return cst_bdb_error_translate(errno);
        }
    }
    else
    {
        return CST_ERROR_PARAM_INCORRECT;
    }
}

/**
    Save info to file storage
*/
int CST_save(CST * st)
{
    return CST_ERROR_OK;
}

int CST_backup(CST * st, const char *filename, unsigned char *password)
{
    int result = CST_ERROR_PARAM_INCORRECT;
    if (st && NAME_IS_NOT_EMPTY(filename))
    {
        result = CST_ERROR_LOCK;
        
        const size_t BUFFER_SIZE = 1024;
        char buff[BUFFER_SIZE];
        
        CST_LOCK_BEGIN_S(LOCK_SH);
            int dst = open(filename, O_CREAT | O_TRUNC | O_WRONLY, 0640);
            if (dst)
            {
                int src = st->db->fd(st->db);
                g_assert(src > 0);
                lseek(src, 0L, SEEK_SET);
                ssize_t r = 0, w = 0;
                result = CST_ERROR_OK;
                do
                {
                    r = read(src, buff, BUFFER_SIZE);
                    if (r > 0) 
                    {
                        w = write(dst, buff, r);
                    }
                    
                    if ((r < 0) || (w < 0))
                    {
                        result = cst_file_error_translate(errno);
                        break;
                    }
                        
                } while (BUFFER_SIZE == r);
                close(dst);
            }        
            else
            {
                result = cst_file_error_translate(errno);
            }
        CST_LOCK_END;
    }
    return result;
}

void CST_free(CST * st)
{
    if (st)
    {
        cst_destroy(st);
    }
    return;
}

int CST_error(int error_code)
{
    CST_global_error = error_code;
    return CST_global_error;
}

int CST_last_error()
{
    return CST_global_error;
}

int cst_assign(CST * st, const t_seqnum certID, const t_seqnum keyID)
{
    TRACE("*");
    g_assert(st && (certID > 0));

    KEY * k = key_search_id(st, FALSE, keyID);

    if (k)
    {
        return CST_error(cst_set_keylink(st, certID, keyID));
    }
    else
    {
        return CST_error(CST_ERROR_KEY_NOTFOUND);
    }
}

int CST_assign(CST * st, const t_seqnum certID, const t_seqnum keyID, unsigned char *pass)
{
    int result = CST_ERROR_OK;
    GError * error = NULL;
    
    if (st && (certID > 0) && (keyID > 0))
    {
        int res = FALSE;
        result = CST_ERROR_LOCK;
       
        /* Check */
        CST_LOCK_BEGIN(LOCK_SH);
            EVP_PKEY * xkey = key_get_EVP_PKEY(st, FALSE, keyID, pass, &error);
            if (xkey)
            {
                X509 * x = cert_get_X509(st, certID);
                if (x)
                {
                    res = X509_check_private_key(x, xkey);
                    EVP_PKEY_free(xkey);
                    X509_free(x);
                }
                else
                {
                    result = CST_ERROR_CERT_NOTFOUND;
                }
            }
            else
            {
                result = CST_ERROR_KEY_NOTFOUND;
                if (error)
                {
                    result = error->code;
                    g_clear_error(&error);
                }
            }

            /* Assign */
            if (res)
            {
                result = cst_assign(st, certID, keyID);
            }
            else
            {
                result = CST_error(CST_ERROR_ASSIGN_INCORRECT);
            }
        CST_LOCK_END;
        CST_LOCK_SET_ERROR(result);
    }

    return result;
}

gboolean cst_changed(CST * st)
{
    g_assert(st && st->db);
    return cst_mcount_get(st) > st->modification_count;
}

int cst_refresh_if_need(CST * st)
{
    g_assert(st);
    int result = CST_ERROR_OK;

    if (cst_changed(st))
    {
        cst_all_list_destroy(st);

        if (CST_ERROR_OK == result)
        {
            cst_all_list_init(st);
            initial_scan_db(st);
        }
    }
    return result;
}
