/*
 * This file is part of certman
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Ed Bartosh <Eduard.Bartosh@nokia.com>
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
    @file cst_key_idx.c

    Certificate Management Library

    Key functions (indexes)
*/

#include "cst_t.h"
#include <string.h>

/**
    Compare key body
*/
gboolean key_body_equal(const KEY * key_a, const KEY * key_b)
{
    TRACE("*");
    g_assert(key_a && key_b);

    if (key_a->der_buffer_len != key_b->der_buffer_len)
    {
        return FALSE;
    } else
    {
        if (0 == memcmp
            (key_a->der_buffer, key_b->der_buffer, key_b->der_buffer_len))
        {
            return TRUE;
        } else
        {
            return FALSE;
        }
    }
}

KEY *key_search_id(CST * st, const t_bool pub, const t_seqnum id)
{
    g_assert(st);

    KEY * result = g_hash_table_lookup(st->idx_key_uid, SEQNUM_TO_POINTER(id));

    if ((result != NULL) && (pub != result->pub)) return NULL;

    return result;
}

/**
    Check existence key 
*/
gboolean key_is_exist(CST * st, const KEY * k)
{
#if 0    
    g_assert(st && k && k->name);

    if(!k->pub)
    {
        return FALSE;
    }
        
    GSList *list = g_tree_lookup(st->keys, k->name);

    if (list)
    {
        KEY *key;
        GSList *i;
        for (i = list; i != NULL; i = i->next)
        {
            key = (KEY *) i->data;
            if (key_body_equal(key, k))
            {
                return TRUE;
            }
        }
    }
#endif
    return FALSE;
}

gboolean key_equal(const KEY * key_a, const KEY * key_b)
{
    return key_body_equal(key_a, key_b);
}

void key_put_idx_name(CST * st, KEY * k)
{
    GSList *list = (GSList *) g_tree_lookup(st->keys, k->name);
    list = g_slist_append(list, k);
    g_tree_replace(st->keys, k->name, list);
}

void key_remove_idx_name(CST * st, KEY * k)
{
    GSList *list = (GSList *) g_tree_lookup(st->keys, k->name);

    list = g_slist_remove(list, k);

    if (!list)
    {
        g_tree_remove(st->keys, k->name);
    } else
    {
        KEY *a = (KEY *) list->data;
        g_assert(a);
        g_tree_replace(st->keys, a->name, list);
    }
}

void key_put_idx_uid(CST * st, KEY * key)
{
    g_assert(st && key);
    g_assert(key->uid > 0);
    g_hash_table_insert(st->idx_key_uid, SEQNUM_TO_POINTER(key->uid), key); 
}

void key_remove_idx_uid(CST * st, KEY * key)
{
    g_assert(st && key);
    g_assert(key->uid > 0);
    g_hash_table_remove(st->idx_key_uid, GUINT_TO_POINTER(key->uid));
}

/**
    Append key to list and update indexes
    See: key_remove()
*/
gboolean key_put(CST * st, KEY * k)
{
    g_assert(st && k);

    if (!key_is_exist(st, k))
    {
        key_put_idx_name(st, k);
        key_put_idx_uid(st, k);
        return TRUE;
    } else
    {
        return FALSE;
    }
}

/**
    Remove key from list and update indexes
    Memory used by structure not free
    See: key_put()
*/
void key_remove_i(CST * st, KEY * k)
{
    TRACE("*");
    g_assert(st && k);
    key_remove_idx_name(st, k);
    key_remove_idx_uid(st, k);
    key_free(st, k);
}

void key_remove(CST * st, KEY * k)
{
    key_remove_i(st, k);
}
