/*
 * This file is part of certman
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Ed Bartosh <Eduard.Bartosh@nokia.com>
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
    @file cst_helper.c

    Certificate Management Library

    Common helper functions
*/

#include "cst_t.h"
#include <glib.h>
#include <string.h>
#include <openssl/rand.h>
#include <openssl/bio.h>
#include <openssl/evp.h>
#include <errno.h>
#include <unistd.h>

#define ALGORITHM DEFAULT_ALGORITHM
#define MD EVP_sha1()

static int cst_data_put_base(CST * st, const char *suffix, const t_seqnum id, DBT * data);

/* GError */
GQuark cst_error_quark(void)
{
    static GQuark quark = 0;
    if (quark == 0)
        quark = g_quark_from_static_string ("cst");
    return quark;
}

int cst_sync(DB * db)
{
    int result = db->sync(db, 0);
    int last_errno = 0;
    if (-1 == result) last_errno = errno;
    
    result = fsync(db->fd(db));
    if (-1 == result) last_errno = errno;

    return last_errno;
}

int cst_db_delete(CST * st, const char *suffix, const t_seqnum ID)
{
    TRACE("*");
    g_assert(st && suffix);

    int last_errno = 0;
    
    if (ID > 0)
    {
        DBT key;
        make_dbkey(&key, suffix, ID);
        
        DB * db = st->db;
        g_assert(db);
        
        int result = db->del(db, &key, 0);
        if (0 == result)
        {   
            cst_mcount_update(st);
            last_errno = cst_sync(st->db);
        }
        else if (-1 == result)
        {
            last_errno = errno;
        }

        g_free(key.data);
    }
    
    return cst_bdb_error_translate(last_errno);
}

t_seqnum extract_id(DBT * key)
{    
    t_seqnum result;
    g_assert(key->size > sz_seqnum);
    memcpy(&result, key->data + key->size - sz_seqnum, sz_seqnum);
    result = SEQNUM_FROM_LE(result);
    return result;
}

int cst_check_suffix(DBT * key, const char * suffix)
{
    return (0 == memcmp(suffix, key->data, strlen(suffix)));
}

int cst_lock(CST * st, int operation)
{
    g_assert(st);
    g_assert((LOCK_EX == operation) || (LOCK_SH == operation));

    g_assert(NULL == st->db);
    
    int result = CST_ERROR_OK;
   
    if (LOCK_EX == operation)
    {
        g_static_rw_lock_writer_lock(&st->rwlock);

        st->db = cst_dbopen(st->filename, O_RDWR);
        
        if (st->db)
        {
            if (0 != flock(st->db->fd(st->db), LOCK_EX)) 
            {
                result = CST_ERROR_LOCK;
                st->db->close(st->db);
                st->db = NULL;
            }
        }
        else
        {
            result = cst_bdb_error_translate(errno);
        }
        
        if (CST_ERROR_OK != result)
        {
            g_static_rw_lock_writer_unlock(&st->rwlock);
        }
    }    
    else if (LOCK_SH == operation)
    {
        g_static_rw_lock_reader_lock(&st->rwlock);

        st->db = cst_dbopen(st->filename, O_RDWR);
        
        if (st->db)
        {
            if (0 != flock(st->db->fd(st->db), LOCK_SH)) 
            {
                result = CST_ERROR_LOCK;
                st->db->close(st->db);
                st->db = NULL;
            }
        }
        else
        {
            result = cst_bdb_error_translate(errno);
        }
        
        if (CST_ERROR_OK != result)
        {
            g_static_rw_lock_reader_unlock(&st->rwlock);
        }
    }

    return result;
}

int cst_unlock(CST * st, int operation)
{
    g_assert(st);
    g_assert((LOCK_EX == operation) || (LOCK_SH == operation));
    g_assert(NULL != st->db);
    int fd = st->db->fd(st->db);
    g_assert(fd > 0);
    int result = -2;

    if (LOCK_EX == operation)
    {
        result = flock(fd, LOCK_UN);
        st->db->close(st->db);
        st->db = NULL;
        g_static_rw_lock_writer_unlock(&st->rwlock);
    }    
    else if (LOCK_SH == operation)
    {
        result = flock(fd, LOCK_UN);
        st->db->close(st->db);
        st->db = NULL;
        g_static_rw_lock_reader_unlock(&st->rwlock);
    }

    g_assert(NULL == st->db);

    return result;
}

void cst_mcount_update(CST * st)
{
    g_assert(st);
    t_mcount current = st->modification_count + 1;
    current = MCOUNT_TO_LE(current);
    DBT data;
    data.size = sizeof(t_mcount);
    data.data = &current;
    cst_data_put_base(st, CST_SUFFIX_MCOUNT, 1, &data);
}

t_mcount cst_mcount_get(CST * st)
{
    g_assert(st);
    DBT data;
    memset(&data, 0, sizeof(data));
    t_mcount mc = 0;
    if (CST_ERROR_OK == cst_data_get(st, CST_SUFFIX_MCOUNT, 1, &data)
            && data.data 
            && (data.size == sizeof(t_mcount)))
    {
        memcpy(&mc, data.data, sizeof(t_mcount));
        mc = MCOUNT_FROM_LE(mc);
    }
    return mc;
}

int cst_data_put(CST * st, const char *suffix, const t_seqnum id, DBT * data)
{
    cst_mcount_update(st);
    return cst_data_put_base(st, suffix, id, data);
}

static int cst_data_put_base(CST * st, const char *suffix, const t_seqnum id, DBT * data)
{
    TRACE("*");
    g_assert(st && suffix && (id > 0) && data);
    DBT key;
    make_dbkey(&key, suffix, id);
    
    DB * db = st->db;
    g_assert(db);
    
    int last_errno = 0;
    int result = db->put(db, &key, data, 0);
    if (-1 == result) last_errno = errno;
    else if (0 == result) 
    {
        last_errno = cst_sync(st->db);
    }

    g_free(key.data);

    return cst_bdb_error_translate(last_errno);
}

int cst_data_get(CST * st, const char *suffix, const t_seqnum id, DBT * data)
{
    TRACE("*");
    g_assert(st && suffix && (id > 0) && data);
    DBT key;
    make_dbkey(&key, suffix, id);
    
    DB * db = st->db;
    g_assert(db);
    
    int result = db->get(db, &key, data, 0);
    int last_errno = errno;

    g_free(key.data);
    
    if (0 == result)        return CST_ERROR_OK;
    else if (1 == result)   return CST_ERROR_NOT_FOUND;
    else                    return cst_bdb_error_translate(last_errno); 
}

void make_dbkey(DBT *dbkey, const char *suffix, const t_seqnum id)
{
    TRACE("*");
    g_assert(id > 0);
    g_assert(dbkey);

    t_seqnum t = SEQNUM_TO_LE(id);

    dbkey->size = sz_seqnum;

    int suffix_len = (NULL != suffix ? strlen(suffix) : 0);

    dbkey->size += suffix_len;
    dbkey->data = g_malloc0(dbkey->size);

    if (suffix_len > 0)
    {
        memcpy(dbkey->data, suffix, suffix_len);
    }
    
    memcpy(dbkey->data + suffix_len, &t, sz_seqnum);
}

void write_mem(void ** dist, void * src, t_rsize sz)
{
    memcpy(*dist, src, sz);
    *dist += sz;
}    

void write_header_mem(void ** data, t_rtype tp, t_rsize sz)
{
    t_rsize size = RSIZE_TO_LE(sz);
    write_mem(data, &tp, sz_rtype);
    write_mem(data, &size, sz_rsize);
}

void write_header(CRYPT_FILE * cf, t_rtype tp, t_rsize sz)
{
    t_rsize size = RSIZE_TO_LE(sz);
    crypt_writer_put(cf, &tp, sz_rtype);
    crypt_writer_put(cf, &size, sz_rsize);
}

GSList *parse_buffer(unsigned char *buffer, t_rsize len)
{
    g_assert(buffer);
    g_assert(len > 0);

    PARSE_RES *item = NULL;
    GSList *result = NULL;

    int counter = 0;

    while (counter < len)
    {
        item = g_malloc0(sizeof(PARSE_RES));
        result = g_slist_append(result, item);

        memcpy(&item->type, buffer + counter, sz_rtype);
        counter += sz_rtype;

        if (counter >= len)
        {
            parse_res_free(result);
            CST_error(CST_ERROR_STRUCTURE_CORRUPT);
            return NULL;
        }

        memcpy(&item->size, buffer + counter, sz_rsize);
        item->size = RSIZE_FROM_LE(item->size);
        counter += sz_rsize;

        item->data = buffer + counter;
        counter += item->size;

        if (counter > len)
        {
            parse_res_free(result);
            CST_error(CST_ERROR_STRUCTURE_CORRUPT);
            return NULL;
        }
    }

    return result;
}

void parse_res_free(GSList * list)
{
    GSList *i;

    for (i = list; i != NULL; i = i->next)
    {
        g_free(i->data);
    }

    g_slist_free(list);
}

/**
    X509_NAME_hash
*/
guint name_hash(gconstpointer key)
{
    X509_NAME *name = (X509_NAME *) key;
    return (guint) X509_NAME_hash(name);
}

/**
    X509_NAME_cmp
*/
gint helper_name_cmp(gconstpointer akey, gconstpointer bkey)
{
    X509_NAME *a = (X509_NAME *) akey;
    X509_NAME *b = (X509_NAME *) bkey;
    return (gint) X509_NAME_cmp(a, b);
}

gint helper_str_cmp(gconstpointer akey, gconstpointer bkey)
{
    return strcmp((char *) akey, (char *) bkey);
}


gboolean equal_X509_UID(X509 * x, X509_NAME * issuer,
                        ASN1_INTEGER * serial)
{
    g_assert(x && issuer && serial);
    return (0 == X509_NAME_cmp(issuer, X509_get_issuer_name(x)))
        && (0 == ASN1_INTEGER_cmp(serial, X509_get_serialNumber(x)));
}

gboolean equal_CERT_UID(CERT * c, X509_NAME * issuer,
                        ASN1_INTEGER * serial)
{
    g_assert(c && issuer && serial);
    return (0 == X509_NAME_cmp(issuer, c->issuer_name))
        && (0 == ASN1_INTEGER_cmp(serial, c->sn));
}

unsigned char *buffer_to_hex(unsigned char *buffer, int size)
{
    int result_size = size * 2 + 1;
    char *result = g_malloc(result_size);

    int i;
    for (i = 0; i < size; i++)
    {
        sprintf(result + i * 2, "%02X", buffer[i]);
    }

    result[result_size - 1] = 0;

    return result;
}

void cert_free_stack(STACK_OF(X509) * list)
{
    if (list)
    {
        int i;
        for (i = 0; i < sk_X509_num(list); i++)
        {
            X509_free(sk_X509_value(list, i));
        }
        sk_X509_free(list);
    }
}

unsigned char * cst_encrypt_buffer(unsigned char * buffer, int *length, unsigned char *p)
{
    g_assert(length);
    
    unsigned char *def_pass = DEFAULT_PASSWORD;
    unsigned char *password = p;
    if ((!password) || (0 == password[0]))
    {
        password = def_pass;
    }

    if (!buffer || (0 == *length))
    {
        *length = 0;
        return NULL;
    }
    
    BIO * mem = BIO_new(BIO_s_mem());
    BIO * cipher = BIO_new(BIO_f_cipher());

    g_assert(mem && cipher);

    /* Make salt and write it to cf->file */ 
    unsigned char salt[PKCS5_SALT_LEN];
    int res = RAND_pseudo_bytes(salt, sizeof(salt));
    g_assert(res >= 0);
    cst_bio_write(mem, salt, sizeof(salt));
    BIO_flush(mem);
    
    /* Make key based on password and salt*/
    unsigned char key[EVP_MAX_KEY_LENGTH];
    unsigned char iv[EVP_MAX_IV_LENGTH];
    EVP_BytesToKey(ALGORITHM, MD, salt, 
            (unsigned char *)password,
            strlen(password), 1, key, iv);
    
    BIO_set_cipher(cipher, ALGORITHM, key, iv, 1);

    BIO_push(cipher, mem);
    
    /* write data to BIO */
    cst_bio_write(cipher, buffer, *length);
    
    BIO_flush(cipher);

    /* extract result */
    BUF_MEM *bptr;
    BIO_get_mem_ptr(mem, &bptr);
    *length = bptr->length;
    g_assert(*length > 0);
    unsigned char *res_data = g_memdup(bptr->data, *length);
    
    BIO_free_all(cipher);
    
    return res_data;
}

unsigned char * cst_decrypt_buffer(unsigned char * buffer, int *length, unsigned char *p)
{
    g_assert(length);
    
    unsigned char *def_pass = DEFAULT_PASSWORD;
    unsigned char *password = p;
    if ((!password) || (0 == password[0]))
    {
        password = def_pass;
    }
    
    if (!buffer || (0 == *length))
    {
        *length = 0;
        return NULL;
    }
    
    BIO * mem = BIO_new_mem_buf(buffer, *length);
    BIO * cipher = BIO_new(BIO_f_cipher());

    g_assert(mem && cipher);
   
    /* Get salt from buffer */ 
    unsigned char salt[PKCS5_SALT_LEN];
    cst_bio_read(mem, salt, sizeof(salt));

    /* Make key based on password and salt*/
    unsigned char key[EVP_MAX_KEY_LENGTH];
    unsigned char iv[EVP_MAX_IV_LENGTH];
    EVP_BytesToKey(ALGORITHM, MD, salt, 
            (unsigned char *)password,
            strlen(password), 1, key, iv);
    BIO_set_cipher(cipher, ALGORITHM, key, iv, 0);

    BIO_push(cipher, mem);

    /* Read all data */
    const int read_buff_size = *length;
    int result_length = 0;
    int last_read = 0;
    unsigned char *read_buff;
    GSList * read_list = NULL;
   
    do
    {
        read_buff = g_malloc(read_buff_size);
        last_read += cst_bio_read(cipher, read_buff, read_buff_size);
        result_length += last_read;
        if (last_read != 0)
        {
            read_list = g_slist_append(read_list, read_buff);
        }
    } 
    while (last_read == read_buff_size);

    if (0 == last_read)
    {
        last_read = read_buff_size;
        g_free(read_buff);
    }

    /* If not eof then error */
    unsigned char * res_data = NULL; 
    if (BIO_eof(cipher))
    {
        res_data = g_malloc(result_length);        
    }

    int counter = 0;
    GSList * i;
    for (i = read_list; i != NULL; i = i->next)
    {
        if (res_data)
        {
            if (i->next != NULL)
            {
                memcpy(res_data + counter, i->data, read_buff_size);
            }
            else
            {
                memcpy(res_data + counter, i->data, last_read);
            }
        }
        g_free(read_list->data);
        counter += read_buff_size;
    }
    g_slist_free(read_list);
    
    BIO_free_all(cipher);
    
    *length = (NULL == res_data ? 0 : result_length);
    return res_data;
}

void printListUINT(GSList * list, char *name)
{
    GSList * i;
    printf("%s [", name);
    for (i = list; i != NULL; i = i->next)
    {
        printf("%u ", GPOINTER_TO_UINT(i->data));
    }
    printf("]\n");
}
