/*
 * This file is part of certman
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Ed Bartosh <Eduard.Bartosh@nokia.com>
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
    @file cst_crl.c

    Certificate Management Library

    CRL functions (common)
*/

#include "cst_t.h"
#include <memory.h>
#include <string.h>
#include <openssl/conf.h>

static int crl_save(CST * st, const t_seqnum crlID, X509_CRL * crl);
static int crl_load(CST * st, const t_seqnum crlID, X509_CRL ** pcrl);    
static int rescan_crl(CST * st);

static int crl_save(CST * st, const t_seqnum crlID, X509_CRL * crl)
{
    g_assert(st);
    g_assert(crlID > 0);

    DBT body;
    body.data = NULL;
    
    body.size = i2d_X509_CRL(crl, (unsigned char **)&body.data);
    int result = CST_ERROR_BAD_INTERNAL_FORMAT;
    if (body.size > 0)
    {
        result = cst_data_put(st, CST_SUFFIX_CRLINFO, crlID, &body);
        g_free(body.data);
    }
    
    return result;
}

static int crl_load(CST * st, const t_seqnum crlID, X509_CRL ** pcrl)
{
    TRACE("*");
    
    g_assert(st && pcrl);
    
    int result = CST_ERROR_NOT_FOUND;

    if (crlID > 0)
    {
        DBT body;
     
        result = cst_data_get(st, CST_SUFFIX_CRLINFO, crlID, &body);
        if (CST_ERROR_OK == result) 
        {
            /* read man d2i_X509 */
            unsigned char *p = body.data;
            *pcrl = d2i_X509_CRL(NULL, &p, body.size);
            if (NULL != *pcrl) 
            {
                result = CST_ERROR_OK;
            }
            else 
            {
                result = CST_ERROR_STRUCTURE_CORRUPT;
            }
        }
    }
    
    return result;
}

int CST_append_X509_CRL(CST * st, X509_CRL * crl)
{
    if (st && crl)
    {
        return crl_append(st, crl);
    }
    else
    {
        return CST_error(CST_ERROR_PARAM_INCORRECT);
    }
}

X509_CRL *crl_get_X509_CRL(CST * st, const t_seqnum crlID)
{
    g_assert(st);
    X509_CRL * x = NULL;
    if (CST_ERROR_OK == crl_load(st, crlID, &x))
    {
        g_assert(x);
        return x;
    }
    return NULL;
}

/**
    Initialize crl list (constructor)
*/
int cst_crl_list_init(CST * st)
{
    g_assert(st);
    TRACE("*");

    st->simple_crls = NULL;

    return CST_error(CST_ERROR_OK);
}

/**
    Free crl list (destructor)
    Memory used by crl info free also
*/
int cst_crl_list_destroy(CST * st)
{
    g_assert(st);
    g_slist_free(st->simple_crls);
    return CST_error(CST_ERROR_OK);
}

/**
    Append crl to storage  
    and update indexes
*/
int crl_append(CST * st, X509_CRL * crl)
{
    int result = CST_ERROR_PARAM_INCORRECT;

    if (st && crl)
    {
        if (!crl_check_sign(st, crl))
        {
            result = CST_ERROR_CRL_NOT_VALID;
        }
        else
        {
            CST_LOCK_BEGIN(LOCK_EX);
                t_seqnum crlID = cst_next_crl_uid(st);
                
                result = crl_save(st, crlID, crl);
                if (CST_ERROR_OK == result)
                {
                    st->simple_crls = g_slist_append(st->simple_crls, SEQNUM_TO_POINTER(crlID));
                    result = rescan_crl(st);
                    //result = CST_ERROR_OK;
                }
            CST_LOCK_END;
        }
    }
    return CST_error(result);
}

/**
    Import crl from file
*/
static int crl_import_fmt(CST * st, FILE * file, const unsigned char *password, 
        const int fmt)
{
    TRACE("*");
    g_assert(st && file);

    X509_CRL *x = NULL;
    
    switch (fmt)
    {
    case IMPORT_FORMAT_PEM:
        x = PEM_read_X509_CRL(file, NULL, NULL, (void *) password);
        break;
    case IMPORT_FORMAT_DER:
        x = d2i_X509_CRL_fp(file, NULL);
        break;
    }
    
    if (!x)
    {
        return CST_error(CST_ERROR_NOT_FOUND);
    }

    int result = crl_append(st, x);

    X509_CRL_free(x);

    return result;
}

int crl_import(CST * st, FILE * file, const unsigned char *password)
{
    return crl_import_fmt(st, file, password, IMPORT_FORMAT_PEM);
}

int crl_import_DER(CST * st, FILE * file)
{
    return crl_import_fmt(st, file, NULL, IMPORT_FORMAT_DER);
}

gboolean crl_check_sign(CST * st, X509_CRL * xcrl)
{
    g_assert(st && xcrl);
    
    gboolean result = FALSE;
    
    GSList * list = CST_search_by_subj_name(st, X509_CRL_get_issuer(xcrl));

    X509 * x;
    EVP_PKEY * pkey;
    GSList * i;
    t_seqnum certID;
    for (i = list; i != NULL; i = i->next)
    {
        certID = SEQNUM_FROM_POINTER(i->data);
        x = CST_get_cert(st, certID);
        if (!result 
            && (CST_get_folder(st, certID) == CST_FOLDER_CA)    
            && (CST_is_purpose(st, certID, CST_PURPOSE_CA) > 0)
            && CST_is_valid_for(st, x, CST_PURPOSE_CA))
        {
            pkey = X509_get_pubkey(x);
            if (X509_CRL_verify(xcrl, pkey))
            {
                result = TRUE;
                EVP_PKEY_free(pkey);
                X509_free(x);
                break;
            }
            EVP_PKEY_free(pkey);
        }
        X509_free(x);
    }

    g_slist_free(list);
    
    return result;
}

static void crl_put(CST * st, const t_seqnum crlID)
{
    g_assert(st && (crlID > 0));
    st->simple_crls = g_slist_append(st->simple_crls, SEQNUM_TO_POINTER(crlID));
}

static void crl_remove(CST * st, const t_seqnum crlID)
{
    g_assert(st && (crlID > 0));
    st->simple_crls = g_slist_remove(st->simple_crls, SEQNUM_TO_POINTER(crlID));
}

int CST_delete_crl(CST * st, const t_seqnum crlID)
{
    int result = CST_ERROR_PARAM_INCORRECT;

    if (st && (crlID > 0))
    {
        CST_LOCK_BEGIN(LOCK_EX);
            result = cst_db_delete(st, CST_SUFFIX_CRLINFO, crlID);
            crl_remove(st, crlID);
            rescan_crl(st);
        CST_LOCK_END;
        return CST_error(CST_ERROR_OK);
    }

    return CST_error(result);
}

GSList * CST_get_all_crl(CST * st)
{
    GSList * result = NULL;
    if (st)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            result = g_slist_copy(st->simple_crls);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;    
}

void initial_scan_db_crl(CST * st, DBT * key, DBT * data)
{
    g_assert(st && key && data);
    t_seqnum id = extract_id(key); 
    g_assert(id > 0);
    crl_put(st, id);
}

X509_CRL * CST_get_CRL(CST * st, const cst_t_seqnum crlID)
{
    X509_CRL * crl = NULL;
    if (st && (crlID > 0))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            crl = crl_get_X509_CRL(st, crlID);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return crl;
}

typedef struct RESCAN_CRL_DATA_st
{
    CST * st;
    int result;
} RESCAN_CRL_DATA;

static void rescan_crl_unset(gpointer key, gpointer value, gpointer data)
{
    t_seqnum certID = SEQNUM_FROM_POINTER(key);
    g_assert(certID > 0);
    RESCAN_CRL_DATA * d = (RESCAN_CRL_DATA *) data;
    if (CST_ERROR_OK != d->result)
    {
        cst_set_revoked(d->st, certID, FALSE);
    }
    else
    {
        d->result = cst_set_revoked(d->st, certID, FALSE);
    }
}

static int rescan_crl_one(CST * st, X509_CRL * crl)
{
    g_assert(st);
    g_assert(crl);

    int result = CST_ERROR_OK;

    STACK_OF(X509_REVOKED) * items = X509_CRL_get_REVOKED(crl);
    int i;
    X509_REVOKED *revoked = NULL;
    t_seqnum certID = 0;
    for (i = 0; i < sk_X509_REVOKED_num(items); i++)
    {
        revoked = sk_X509_REVOKED_value(items, i);
        certID = cert_search_by_UID(st, X509_CRL_get_issuer(crl), revoked->serialNumber);
        if (certID > 0)
        {
            result = cst_set_revoked(st, certID, TRUE);
        }

        if (CST_ERROR_OK != result)
        {
            break;
        }
    }

    return result;
}

/* Read all CRLs from storage and mark certificates */
static int rescan_crl(CST * st)
{
    g_assert(st);
    int result = CST_ERROR_OK;
    X509_CRL * crl;

    RESCAN_CRL_DATA data;
    data.st = st;
    data.result = result;
    g_hash_table_foreach(st->idx_cert_uid, rescan_crl_unset, &data);
    result = data.result;

    if (CST_ERROR_OK == result)
    {
        GSList * i;
        for (i = st->simple_crls; i != NULL; i = i->next)
        {
            crl = crl_get_X509_CRL(st, (cst_t_seqnum) i->data);
            if (NULL != crl)
            {
                result = rescan_crl_one(st, crl);
                X509_CRL_free(crl);
            }
            else
            {
                result = CST_ERROR_NOT_FOUND;
            }    
            
            if (CST_ERROR_OK != result) break;
        }
    }
    return result;
}
