/*
 * This file is part of certman
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Ed Bartosh <Eduard.Bartosh@nokia.com>
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
    @file cst_cert_idx.c

    Certificate Management Library

    Certificate functions (search)
*/

#include "cst_t.h"
#include <glib.h>

STACK_OF(X509) * slist_to_stack_X509(CST * st, GSList * list)
{
    g_assert(st);
        
    STACK_OF(X509) * result = NULL;

    if (list)
    {
        result = sk_X509_new_null();
        CERT *c;
        GSList *i;
        for (i = list; i != NULL; i = i->next)
        {
            c = (CERT *) i->data;
            sk_X509_push(result, cert_get_X509(st, c->uid));
        }
    }

    return result;
}

static GSList * slist_to_slist(CST * st, GSList * list)
{
    g_assert(st);

    GSList * result = NULL;
    GSList * i;
    CERT * c;
    for (i = list; i != NULL; i = i->next)
    {
        c = (CERT *) i->data;
        result = g_slist_append(result, SEQNUM_TO_POINTER(c->uid));
    }

    return result;
}

GSList * cst_search_by_subj_name(CST * st, X509_NAME * subject_name)
{
    g_assert(st && subject_name);
    return slist_to_slist(st, (GSList *) g_tree_lookup(st->idx_cert_name, subject_name));
}

GSList * CST_search_by_subj_name(CST * st, X509_NAME * subject_name)
{
    GSList * result = NULL;
    if (st && subject_name)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            result = cst_search_by_subj_name(st, subject_name);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

GSList * CST_search_by_email(CST * st, const char *email)
{
    GSList *list = NULL;
    if (st && NAME_IS_NOT_EMPTY(email))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            list = slist_to_slist(st, (GSList *) g_tree_lookup(st->idx_cert_email, email));
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return list;
}

static GSList * cert_search_by_key(CST * st, GHashTable * hash, const char *key)
{
    return g_slist_copy((GSList *) g_hash_table_lookup(hash, key));
}

GSList * CST_search_by_serial(CST * st, const char *serial)
{
    GSList *list = NULL;
    if (st && NAME_IS_NOT_EMPTY(serial))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            list = cert_search_by_key(st, st->idx_cert_serial, serial);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return list;
}

GSList * CST_search_by_fingerprint(CST * st, const char *fingerprint)
{
    GSList *list = NULL;
    if (st && NAME_IS_NOT_EMPTY(fingerprint))
    {
        CST_LOCK_BEGIN(LOCK_SH);
            list = cert_search_by_key(st, st->idx_cert_fingerprint, fingerprint);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return list;
}

GSList * CST_search_by_domain_name(CST * st,
                                   const char *domain_name)
{
    GSList *list = NULL;
    if (st && NAME_IS_NOT_EMPTY(domain_name)) 
    {
        CST_LOCK_BEGIN(LOCK_SH);
            list = slist_to_slist(st, (GSList *) g_tree_lookup(st->idx_cert_dns, domain_name));
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return list;
}

t_seqnum cert_search_by_UID(CST * st, X509_NAME * issuer_name,
                         ASN1_INTEGER * serial_number)
{
    g_assert(st && issuer_name && serial_number);
    t_seqnum res = 0;
        
    CERT *key = cert_new0(st);
    key->issuer_name = X509_NAME_dup(issuer_name);
    key->sn = ASN1_INTEGER_dup(serial_number);

    CERT *result = (CERT *) g_tree_lookup(st->certs, key);

    cert_free(st, key);

    if (result)
    {    
        res = result->uid;
    }

    return res;
}

t_seqnum CST_search_by_UID(CST * st, X509_NAME * issuer_name,
                        ASN1_INTEGER * serial_number)
{
    t_seqnum res = 0;
    if (st && issuer_name && serial_number)
    {
        CST_LOCK_BEGIN(LOCK_SH);    
            res = cert_search_by_UID(st, issuer_name, serial_number);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return res;
}

cst_t_seqnum CST_search_by_X509(CST * st, X509 * xcert)
{
    if (xcert)
    {
        return CST_search_by_UID(st, 
                    X509_get_issuer_name(xcert), 
                    X509_get_serialNumber(xcert));
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return 0;
}

/* Search by folder and purpose */
static gboolean cst_search_by_folder_and_purpose_op(gpointer k, gpointer v, gpointer d)
{
    HELPER_PARAM *param = (HELPER_PARAM *) d;
    CERT *c = (CERT *) v;

    if (cst_get_folder(param->st, c->uid) == param->folder
        && cst_is_purpose(param->st, c->uid, param->purpose))
    {
        param->list = g_slist_append(param->list, c);
    }

    return FALSE;
}

GSList * CST_search_by_folder_and_purpose(CST * st, 
                                          const t_cert_folder folder, 
                                          const t_cert_purpose purpose)
{
    GSList * result = NULL;
    if (st)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            HELPER_PARAM param;
            param.st = st;
            param.list = NULL;
            param.folder = folder;
            param.purpose = purpose;

            g_tree_foreach(st->certs, cst_search_by_folder_and_purpose_op, &param);
            result = slist_to_slist(st, param.list);
            g_slist_free(param.list);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}


/* Search by folder */
static gboolean cst_search_by_folder_op(gpointer k, gpointer v, gpointer d)
{
    HELPER_PARAM *param = (HELPER_PARAM *) d;
    CERT *c = (CERT *) v;

    if (cst_get_folder(param->st, c->uid) == param->folder)
    {
        param->list = g_slist_append(param->list, c);
    }

    return FALSE;
}

GSList * CST_search_by_folder(CST * st, const t_cert_folder folder)
{
    GSList * result = NULL;
    if (st)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            HELPER_PARAM param;
            param.st = st;
            param.list = NULL;
            param.folder = folder;

            g_tree_foreach(st->certs, cst_search_by_folder_op, &param);
            result = slist_to_slist(st, param.list);
            g_slist_free(param.list);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

/* Search by purpose */
static gboolean cst_search_by_purpose_op(gpointer k, gpointer v,
                                         gpointer d)
{
    HELPER_PARAM *param = (HELPER_PARAM *) d;
    CERT *c = (CERT *) v;

    if (cst_is_purpose(param->st, c->uid, param->purpose))
    {
        param->list = g_slist_append(param->list, c);
    }

    return FALSE;
}

GSList * CST_search_by_purpose(CST * st,
                               const t_cert_purpose purpose)
{
    GSList * result = NULL;
    if (st)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            HELPER_PARAM param;
            param.st = st;
            param.list = NULL;
            param.purpose = purpose;

            g_tree_foreach(st->certs, cst_search_by_purpose_op, &param);
            result = slist_to_slist(st, param.list);
            g_slist_free(param.list);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}
