/*
 * This file is part of certman
 *
 * Copyright (C) 2006 Nokia Corporation.
 *
 * Contact: Ed Bartosh <Eduard.Bartosh@nokia.com>
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**
   @file cst_cert.h

   Crypt functions and structures
*/

#ifndef CST_CERT_H_
#define CST_CERT_H_

#include "cst.h"
#include "cst_const.h"
#include <glib.h>
#include <openssl/x509.h>

#ifdef __cplusplus
extern "C" {
#endif


/* Cert description structure */
    typedef struct CERT_st {
        X509_NAME *name;
        t_cert_folder folder;
        t_cert_purpose purpose;
        int state;
        X509_NAME *issuer_name;
        ASN1_INTEGER *sn;
        char *email;
        char *domain_name;
        //unsigned char *der_buffer;
        //int der_buffer_len;
        t_seqnum key_link;
        t_bool def;
        t_seqnum uid;
        GSList * capabilities; // See: CERT_CAPABILITY
        char *serial;
        char *fingerprint;
        t_bool is_revoked;
    } CERT;

    typedef struct CERT_CAPABILITY_st {
        unsigned char * oid;
        int oid_length;
        
        unsigned char * data;
        int data_length;
    } CERT_CAPABILITY;
    
/* Helper structure for export */
    typedef struct CERT_EXPORT_st {
        CST *st;
        FILE *fp;
        t_cert_folder folder;
        int error;
        int fmt;
    } CERT_EXPORT;

/**
    Return TRUE if cert is root (selfsigned)
*/
    int cert_is_root_x(X509 * x);
    int cert_is_root(CERT * c);

    
    char *cert_get_domain_name_x(X509 * x);
    char *cert_get_email_x(X509 * x);    
    
/**
    Save storage (internal)
*/
    int cst_save_t(CST * st, const char *filename, char *password);

/** 
    Compare CST_CERT by Issuer name and Serial number 
*/
    gint cert_cmp_by_iname_sn(gconstpointer a, gconstpointer b);

/**
    Append cert to list and update indexes

    @return TRUE if append and FALSE if exist
*/
    gboolean cert_put(CST * st, CERT * c);

/**
    Unlink cert from key
*/
    void cert_unlink(CST * st, const t_seqnum certID);

    int cst_set_keylink(CST * st, const t_seqnum certID, const t_seqnum keyID);
    t_seqnum cst_get_keylink(CST * st, const t_seqnum certID);    

/**
    Remove cert from list and update indexes
*/
    int cert_remove(CST * st, const t_seqnum id);
    int cert_remove_x(CST * st, X509 * c);
    int cert_remove_uid(CST * st, X509_NAME * issuer_name,
                         ASN1_INTEGER * serial_number);

/**
    Create CST_CERT structure
*/
    CERT *cert_new(CST * st, X509 * x);
    CERT *cert_new0(CST * st);

/**
    Init certificate idx
*/
    int cst_cert_list_init(CST * st);

/**
    Destroy certificate idx
*/
    int cst_cert_list_destroy(CST * st);

/**
    Free CST_CERT structure
*/
    void cert_free(CST * st, CERT * c);

/**
    Read from buffer(file)
*/
    int cert_read(CST * st, unsigned char *buffer, t_rsize len);

/**
    Write to file
*/
    int cert_write(CERT * cert, CRYPT_FILE * cf);
    int cst_cert_all_write(CST * st, CRYPT_FILE * cf);

/**
    Append cert to storage
*/
    int cert_append(CST * st, X509 * cert);
    t_seqnum cert_append_f(CST * st, X509 * x, t_cert_folder f, GError **error);

/**
    Get X509 structure
*/
    X509 *cert_get_X509(CST * st, const t_seqnum cid);

/**
    Internal search
*/
    t_seqnum cert_search_by_UID(CST * st, X509_NAME * issuer_name,
                             ASN1_INTEGER * serial_number);

    //CERT *cert_get_default(CST * st, char *email);
    t_seqnum cert_get_default_id(CST * st, const char *email);


/**
    Search sertificate by id
*/
    CERT * cert_get_by_id(CST * st, const t_seqnum id);

/**
    Deserealize CERT structure
*/ 
    int cert_parser_info(CST * st, DBT * data, CERT ** c);

/**
    Put scanned to indexes
*/ 
    void initial_scan_db_cert(CST * st, DBT * key, DBT * data);

/**
    Check purpose of cert
*/    
    int cst_set_folder(CST * st, const cst_t_seqnum certID, const t_cert_folder f);
    t_cert_folder cst_get_folder(CST * st, const cst_t_seqnum certID);
    int cst_set_purpose(CST * st, const cst_t_seqnum certID, const t_cert_purpose p, const int value);
    int cst_is_purpose(CST * st, const cst_t_seqnum certID, const t_cert_purpose p);
    int cst_set_default(CST * st, const t_seqnum certID);
    t_seqnum cert_get_default_id(CST * st, const char *email);
    int cst_set_revoked(CST * st, const cst_t_seqnum certID, const int value);
    int cst_get_revoked(CST * st, const cst_t_seqnum certID);

    GSList * cst_search_by_subj_name(CST * st, X509_NAME * subject_name);
    
    
#ifdef __cplusplus
}
#endif
#endif                          /* CST_CERT_H_ */
