/*
 * Copyright (c) 2003, 2004 Nokia
 * Author: tsavola@movial.fi
 *
 * This program is licensed under GPL (see COPYING for details)
 */

#ifndef PROTOCOL_H
#define PROTOCOL_H

#include <stdint.h>
#include <pty.h>
#include "common.h"
#include "mount.h"

#define PROTOCOL_VERSION     4

#define INTERNAL_ERROR_CODE  32767

/** Authentication information. */
typedef struct {
	char user[USER_SIZE];
	char pwd[PWD_SIZE];
} auth_info_t;

/** Information on the command execution. */
typedef struct {
	int16_t umount;
	char *target;
	mount_info_t **mounts;
	char **argv;
	char *curdir;
	char **env;
	int16_t is_tty;
	int16_t tty_row;
	int16_t tty_col;
	int16_t tty_xpixel;
	int16_t tty_ypixel;
} cmd_info_t;

extern ssize_t write_ni(int fd, void *buf, size_t len);
extern ssize_t read_ni(int fd, void *buf, size_t len);

extern int write_int16(int fd, int16_t i);
extern int write_int32(int fd, int32_t i);
extern int write_uint64(int fd, uint64_t i);
extern int write_str(int fd, const char *str);
extern int write_strv(int fd, const char **strv);
extern int write_mount(int fd, const mount_info_t *);
extern int write_mountv(int fd, const mount_info_t **);

extern int16_t read_int16(int fd);
extern int32_t read_int32(int fd);
extern int read_uint64(int fd, uint64_t *);
extern char *read_str(int fd);
extern char **read_strv(int fd);
extern mount_info_t *read_mount(int fd);
extern mount_info_t **read_mountv(int fd);

extern int write_buf(int fd, void *buf, size_t len);
extern int read_buf(int fd, void *buf, size_t len);

extern int write_auth(int fd, const char *user, const char *pwd);
extern int read_auth(int fd, auth_info_t *auth);

extern int write_cmd(int fd,
		     int16_t umount,
		     const char *target,
		     const mount_info_t **mounts,
		     const char **argv,
		     const char *curdir,
		     const char **env,
		     int16_t is_tty,
		     const struct winsize *);
extern int read_cmd(int fd, cmd_info_t *cmd);
extern void free_cmd(cmd_info_t *cmd);

typedef enum {
	/** Authentication reply.                      server -> client */
	PTYPE_AUTH_OK  = 1,

	/** Request data from stdin.                   server -> client */
	PTYPE_IN_REQ   = 2,

	/** Some data written into stdin.              client -> server */
	PTYPE_IN_DATA  = 3,

	/** Request data from stdout.                  client -> server */
	PTYPE_OUT_REQ  = 4,

	/** Some data written into stdout.             server -> client */
	PTYPE_OUT_DATA = 5,

	/** Request data from stderr.                  client -> server */
	PTYPE_ERR_REQ  = 6,

	/** Some data written into stderr.             server -> client */
	PTYPE_ERR_DATA = 7,

	/** The return code of the command.            server -> client */
	PTYPE_RC       = 8,

	/** An error message; the client should die.   server -> client */
	PTYPE_ERROR    = 9,

	/** An informative message.                    server -> client */
	PTYPE_MESSAGE  = 10,

	/** Protocol version.                          client <> server */
	PTYPE_VERSION  = 11
} ptype_t;

/** The header of a packet sent by the server. The data follows... */
typedef struct {
	/** One of the PTYPE-values. */
	int16_t type;

	/** The amount of data. */
	int32_t size;
} phead_t;

extern int write_phead(int fd, int16_t type, int32_t size);
extern int write_packet(int fd, int16_t type, void *buf, int32_t size);
extern int read_phead(int fd, phead_t *head);

extern int send_version(int fd);
extern int get_version(int fd);

#endif
