/**
  @file put.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <time.h>

#include <glib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "obc-main.h"
#include "put.h"

static gboolean put_object(ObcContext *ctx, const char *name,
                           gboolean is_dir, gint *err) {
    gboolean ret;

    if (is_dir) {
        DIR *dp;
        struct dirent *ep;

        if (g_str_equal(name, ".") || g_str_equal(name, ".."))
            return TRUE;

        if (chdir(name) < 0) {
            printf("Could not change to local directory %s: %s\n",
                    name, g_strerror(errno));
            if (err)
                *err = GW_OBEX_ERROR_LOCAL_ACCESS;
            return FALSE;
        }

        if (!gw_obex_mkdir(ctx->obex, name, err)) {
            printf("Could not create remote directory %s\n", name);
            chdir("..");
            return FALSE;
        }

        if (!gw_obex_chdir(ctx->obex, name, err)) {
            printf("Could not change to remote directory %s\n", name);
            chdir("..");
            return FALSE;
        }

        dp = opendir("./");
        if (dp == NULL) {
            printf("Could not open local directory %s: %s\n", name,
                    g_strerror(errno));
            chdir("..");
            gw_obex_chdir(ctx->obex, "..", NULL);
            if (err)
                *err = GW_OBEX_ERROR_LOCAL_ACCESS;
            return FALSE;
        }

        while ((ep = readdir(dp))) {
            gboolean dir;

            switch (ep->d_type) {
                case DT_REG:
                    dir = FALSE;
                    break;
                case DT_DIR:
                    dir = TRUE;
                    break;
                default:
                    continue;
            }

            if (!put_object(ctx, ep->d_name, dir, err)) {
                closedir(dp);
                gw_obex_chdir(ctx->obex, "..", NULL);
                chdir("..");
                return FALSE;
            }
        }

        closedir(dp);
        gw_obex_chdir(ctx->obex, "..", NULL);
        chdir("..");

        return TRUE;
    }

    ctx->object = g_strdup(name);
    ctx->xfer_complete = FALSE;
    ctx->start = time(NULL);
    ret = gw_obex_put_file(ctx->obex, name, name, err);
    g_free(ctx->object);
    ctx->object = NULL;

    return ret;
}

gboolean cmd_put(ObcContext *ctx, gint argc, gchar *argv[], gint *err) {
    int c, i;
    gboolean ret, recursive;

    if (argc < 2)
        return FALSE;

    /* Defaults */
    recursive = FALSE;

    while ((c = getopt(argc, argv, "r")) != -1) {
        switch (c) {
            case 'r':
                recursive = TRUE;
                break;
            default:
                printf("Unhandled option character: '%c'\n", c);
                break;
        }
    }

    c = optind;
    optind = 0;

    if (argc == c)
        return FALSE;

    ret = TRUE;
    for (i = c; i < argc && ret; i++) {
        struct stat st;
        gboolean dir;

        if (stat(argv[i], &st) < 0) {
            printf("Could not stat %s: %s\n", argv[i], g_strerror(errno));
            if (err)
                *err = GW_OBEX_ERROR_LOCAL_ACCESS;
            return FALSE;
        }

        if (S_ISDIR(st.st_mode))
            dir = TRUE;
        else if (S_ISREG(st.st_mode))
            dir = FALSE;
        else
            continue;

        if (!put_object(ctx, argv[i], S_ISDIR(st.st_mode), err)) {
            printf("Puting %s failed\n", argv[i]);
            return FALSE;
        }
    }

    return ret;
}

void put_help(ObcContext *ctx, const char *name) {
    printf("%s [-r] <file..>\n"
           "Send local file(s)\n"
           "Parameters:\n"
           "-r\tRecursive\n", name);
}

