/**
  @file get.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <glib.h>

#include <gw-obex.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "obc-main.h"
#include "fl.h"
#include "get.h"

static gboolean has_globs(const char *str) {
    if (strchr(str, '*') || strchr(str, '?'))
        return TRUE;
    else
        return FALSE;
}

static gboolean get_object(ObcContext *ctx, const char *name,
                           gboolean is_dir, gint *err) {
    gboolean ret;

    if (is_dir) {
        GSList *dir, *c;

        /* Ignore parent dir pointers */
        if (g_str_equal(name, ".."))
            return TRUE;

        if (!gw_obex_chdir(ctx->obex, name, err)) {
            printf("Could not chdir to %s\n", name);
            return FALSE;
        }

        if (mkdir(name, 0770) < 0) {
            printf("Could not create local directory %s: %s\n",
                    name, g_strerror(errno));
            gw_obex_chdir(ctx->obex, "..", err);
            if (err)
                 *err = GW_OBEX_ERROR_LOCAL_ACCESS;
            return FALSE;
        }

        chdir(name);

        dir = fl_list_get(ctx, NULL, FALSE, err);
        if (dir == NULL && err && *err >= 0x40) {
            printf("Getting folder-listing failed.\n");
            gw_obex_chdir(ctx->obex, "..", NULL);
            chdir("..");
            return FALSE;
        }

        for (c = dir; c != NULL; c = c->next) {
           FlEntry *e = c->data;
           g_assert(e != NULL);
           if (!get_object(ctx, e->name, e->dir, err)) {
               fl_list_free(dir);
               gw_obex_chdir(ctx->obex, "..", err);
               chdir("..");
               return FALSE;
           }
        }

        fl_list_free(dir);
        chdir("..");

        if (!gw_obex_chdir(ctx->obex, "..", err)) {
            printf("Could not chdir to .. (reported directory may be wrong)\n");
            return FALSE;
        }

        return TRUE;
    }

    ctx->object = g_strdup(name);
    ctx->xfer_complete = FALSE;
    ctx->start = time(NULL);
    ret = gw_obex_get_file(ctx->obex, name, name, err);
    g_free(ctx->object);
    ctx->object = NULL;

    return ret;
}

static gboolean get_file_or_dir(ObcContext *ctx, const char *name,
                                gboolean recursive, gint *err) {
    gboolean is_dir = FALSE;

    if (recursive) {
        FlEntry *e;

        e = fl_list_find_name(ctx->fl_cache, name);
        if (e == NULL) {
            printf("%s not found in folder-listing\n", name);
            if (err)
                *err = GW_OBEX_ERROR_INVALID_PARAMS;
            return FALSE;
        }

        is_dir = e->dir;
    }

    if (!get_object(ctx, name, is_dir, err)) {
        printf("Getting %s failed.\n", name);
        return FALSE;
    }

    return TRUE;
}

gboolean cmd_get(ObcContext *ctx, gint argc, gchar *argv[], gint *err) {
    int c, i;
    gboolean recursive;

    if (argc < 2)
        return FALSE;

    /* Defaults */
    recursive = FALSE;

    while ((c = getopt(argc, argv, "r")) != -1) {
        switch (c) {
            case 'r':
                recursive = TRUE;
                break;
            default:
                printf("Unhandled option character: '%c'\n", c);
                break;
        }
    }

    c = optind;
    optind = 0;

    if (argc == c)
        return FALSE;

    if (recursive && ctx->fl_cache == NULL) {
        ctx->fl_cache = fl_list_get(ctx, NULL, FALSE, err);
        if (ctx->fl_cache == NULL && err && *err >= 0x40) {
            printf("Getting folder-listing failed.\n");
            return FALSE;
        }
    }

    for (i = c; i < argc; i++) {
        GSList *matched;
        
        matched = NULL;

        if (has_globs(argv[i])) {
            GSList *c;

            if (!recursive && ctx->fl_cache == NULL) {
                ctx->fl_cache = fl_list_get(ctx, NULL, FALSE, err);
                if (ctx->fl_cache == NULL && err && *err >= 0x40) {
                    printf("Getting folder-listing failed.\n");
                    return FALSE;
                }
            }

            matched = fl_list_match_glob(ctx->fl_cache, argv[i]);
            for (c = matched; c != NULL; c = c->next) {
                FlEntry *e = c->data;
                g_assert(e != NULL);
                if (e->name == NULL)
                    continue;

                if (!get_file_or_dir(ctx, e->name, recursive, err))
                    return FALSE;
            }
        }

        if (!matched && !get_file_or_dir(ctx, argv[i], recursive, err))
            return FALSE;
    }

    return TRUE;
}

void get_help(ObcContext *ctx, const char *name) {
    printf("%s [-r] <file..>\n"
           "Receive remote file(s)\n"
           "Parameters:\n"
           "-r\tRecursive\n", name);
}

