/* -*- Mode: C; c-basic-offset: 8; indent-tabs-mode: t -*- */

/*
  osso-ic-oss Internet Connectivity library
  Copyright (C) 2005 Nokia Corporation

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 2.1 of the License, or (at
  your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/


/*
  This is an example program how to use the Internet Connectivity Extended
  interface. It connects to an HTTP server, issues a simple GET request and
  prints the output from the server to stdout.

  Compile it with the command:

	gcc osso-iap-connect.c -o osso-iap-connect \
	`pkg-config --cflags --libs osso-ic glib-2.0` \
	-DDBUS_API_SUBJECT_TO_CHANGE

  To run, provide hostname and port as arguments:

	./osso-iap-connect www.maemo.org 80
  
*/


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#include <netdb.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <osso-ic.h>

#define BUFFER_SIZE 3
#define IAP_LENGTH 20
#define GET_REQUEST "GET / HTTP/1.0\n\n"

typedef enum {
	APPSTATE_DISCONNECTED,
	APPSTATE_CONNECTING,
	APPSTATE_CONNECTED,
	APPSTATE_REQUEST_SENT
} APPState;

typedef struct _AppContext 
{
	const char *host, *port;
	char iap_name[IAP_LENGTH];
	APPState appstate;
	GMainLoop *loop;
	GIOChannel *channel;
	guint read_watch, write_watch;
} AppContext;


void iap_callback(struct iap_event_t *event, void *arg);
void connect_server(AppContext *ctxt);
gboolean read_handler(GIOChannel *source, GIOCondition condition,
		      gpointer data);
gboolean write_handler(GIOChannel *source, GIOCondition condition,
		      gpointer data);
void quit(AppContext *ctxt, const char *reason);


gboolean read_handler(GIOChannel *source, GIOCondition condition,
		       gpointer data)
{
	gsize bytes;
	AppContext *ctxt = (AppContext *) data;
	gchar buffer[BUFFER_SIZE + 1];
	GIOStatus status;

	g_assert(data != NULL);
	
	if (condition & G_IO_IN && ctxt->appstate == APPSTATE_REQUEST_SENT) {
		status = g_io_channel_read_chars(source, buffer, BUFFER_SIZE,
						 &bytes, NULL);
		if (status == G_IO_STATUS_NORMAL && bytes > 0) {
			buffer[bytes] = '\0';
			printf("%s", buffer);
		}
		else {
			quit(ctxt, "Couldn't read from the channel.");
			return FALSE;
		}
	}
	if (condition & G_IO_ERR) {
		quit(ctxt, "G_IO_ERR from channel.");
		return FALSE;
	}
	if (condition & G_IO_HUP) {
		quit(ctxt, "G_IO_HUP from channel.");
		return FALSE;
	}

	return TRUE;
}

gboolean write_handler(GIOChannel *source, GIOCondition condition,
		       gpointer data)
{
	gsize bytes;
	AppContext *ctxt = (AppContext *) data;
	
	if (condition & G_IO_OUT && ctxt->appstate == APPSTATE_CONNECTED) {
		g_io_channel_write_chars(source, GET_REQUEST, -1, &bytes, NULL);
		g_io_channel_flush(source, NULL);
		ctxt->appstate = APPSTATE_REQUEST_SENT;
		return FALSE;
	}

	return TRUE;
}



void iap_callback(struct iap_event_t *event, void *arg)
{
	AppContext *ctxt = (AppContext *) arg;
	g_assert(ctxt != NULL);
	
	switch (event->type) {
	case OSSO_IAP_CONNECTED:
		printf("OSSO_IAP_CONNECTED\n");
		if (ctxt->appstate == APPSTATE_CONNECTING) {
			ctxt->appstate = APPSTATE_CONNECTED;
			strncpy(ctxt->iap_name, event->iap_name, IAP_LENGTH);
			connect_server(ctxt);
		}
		break;
	case OSSO_IAP_DISCONNECTED:
		printf("OSSO_IAP_DISCONNECTED\n");
		if (ctxt->appstate == APPSTATE_CONNECTED ||
		    ctxt->appstate == APPSTATE_REQUEST_SENT) {
			if (strncmp(event->iap_name, ctxt->iap_name, IAP_LENGTH) == 0) {
				ctxt->appstate = APPSTATE_DISCONNECTED;
				quit(ctxt, "IAP disconnected.");
			}
		}
		break;
	case OSSO_IAP_ERROR:
		printf("OSSO_IAP_ERROR error_code=0x%x\n",
			-event->u.error_code);
		quit(ctxt, "There was an IAP error.");
		break;
	}
}

void connect_server(AppContext *ctxt)
{
	int fd;
	struct addrinfo hints, *addrinfo;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;

	if (getaddrinfo(ctxt->host, ctxt->port, &hints, &addrinfo) != 0){
		printf("getaddrinfo() failed.");
		exit(1);
	}
	
	fd = socket(addrinfo->ai_family, addrinfo->ai_socktype,
		    addrinfo->ai_protocol);
	if (fd == -1) {
		perror("socket error");
		exit(1);
	}

	ctxt->channel = g_io_channel_unix_new(fd);
	g_io_channel_set_flags(ctxt->channel,
			       g_io_channel_get_flags(ctxt->channel) | G_IO_FLAG_NONBLOCK,
			       NULL);

	ctxt->read_watch = g_io_add_watch(ctxt->channel,
					  G_IO_IN | G_IO_ERR | G_IO_HUP,
					  read_handler, ctxt);
	ctxt->write_watch = g_io_add_watch(ctxt->channel,G_IO_OUT,
					   write_handler, ctxt);
	
	if (connect(fd, addrinfo->ai_addr, addrinfo->ai_addrlen) == -1) {
		switch (errno) {
		case EINPROGRESS:
			break;
		default:
			close(fd);
			freeaddrinfo(addrinfo);
			perror("could not connect");
		}
	}

	freeaddrinfo(addrinfo);
}

void quit(AppContext *ctxt, const char *reason)
{
	printf("Quitting: %s\n", reason);

	if (ctxt->channel != 0) {
		g_io_channel_shutdown(ctxt->channel, FALSE, NULL);
		g_io_channel_unref(ctxt->channel);
	}
	
	g_main_loop_quit(ctxt->loop);
	
}


int main(int argc, char **argv)
{
	DBusConnection *conn;
	DBusError error;
	GMainContext *gctxt;
	AppContext ctxt;
	
	if (argc != 3) {
		printf("Usage: %s hostname port\n", argv[0]);
		exit(1);
	}

	memset(&ctxt, 0, sizeof(AppContext));
	ctxt.host = argv[1];
	ctxt.port = argv[2];
	ctxt.appstate = APPSTATE_DISCONNECTED;
	
	/* Create main loop and initialize D-BUS */
	gctxt = g_main_context_default();
	ctxt.loop = g_main_loop_new(gctxt, FALSE);

	dbus_error_init(&error);
	conn = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
	if (conn == NULL) {
		printf("Error when connecting to the system bus: %s\n",
		       error.message);
		exit(1);
	}

	/* setup D-Bus to use the Glib main loop */
	dbus_connection_setup_with_g_main(conn, gctxt);

	if (osso_iap_cb(iap_callback) != OSSO_OK) {
		printf("osso_iap_cb failed");
		exit(1);
	}
	
	ctxt.appstate = APPSTATE_CONNECTING;
	if (osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, &ctxt)
	    != OSSO_OK) {
		printf("osso_iap_connect failed");
		exit(1);
	}

	/* Run main loop for all eternity until it is quit from
	 * the callback function */
	g_main_loop_run(ctxt.loop);

	/* Clean up */
	dbus_connection_disconnect(conn);
	dbus_connection_unref(conn);

	g_main_loop_unref(ctxt.loop);
	g_main_context_unref(gctxt);
	
	return 0;
}
