/**
  @file obex-priv.c
  
  Private functions for the GW OBEX Library

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA.

*/
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <utime.h>
#include <fcntl.h>
#include <strings.h>
#include <termios.h>
#include <glib.h>
#ifdef DEBUG
#include <time.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>

#include <openobex/obex.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "gw-obex.h"
#include "obex-priv.h"
#include "utils.h"

static gboolean file_is_dir(const char *filename) {
    struct stat st;

    if (stat(filename, &st) < 0)
        return FALSE;

    if (S_ISDIR(st.st_mode))
        return TRUE;

    return FALSE;
}

static void gw_obex_set_error(GwObex *ctx) {
    if (ctx->abort)
        ctx->error = GW_OBEX_ERROR_ABORT;
    else if (ctx->conn_fd < 0 || ctx->link_err)
        ctx->error = GW_OBEX_ERROR_DISCONNECT;
    else
        ctx->error = (gint)ctx->obex_rsp;
}

static void idle_callback(obex_t *handle, obex_object_t *object, int mode,
                    int event, int obex_cmd, int obex_rsp) {
    debug("idle_callback() called\n");
    sleep(1);
}

static void obex_link_error(GwObex *ctx) {
    if (ctx->link_err)
        return;
    ctx->link_err = TRUE;
    OBEX_SetUserCallBack(ctx->handle, idle_callback, NULL);
    ctx->done = TRUE;
    ctx->conid = CONID_INVALID;
    if (ctx->conn_fd >= 0) {
        OBEX_TransportDisconnect(ctx->handle);
        close(ctx->conn_fd);
        ctx->conn_fd = -1;
    }
    if (ctx->gio) {
        g_io_channel_unref(ctx->gio);
        ctx->gio = NULL;
    }
    if (ctx->gio_source) {
        g_source_destroy(ctx->gio_source);
        ctx->gio_source = NULL;
    }
    if (ctx->buf) {
        /* Check that buffer is owned by us */
        if (!(ctx->obex_op == OBEX_CMD_PUT && ctx->stream_fd < 0))
            g_free(ctx->buf);
        ctx->buf = NULL;
        ctx->buf_size = 0;
    }
}

static gboolean gw_obex_do_abort(GwObex *ctx) {
    debug("gw_obex_do_abort()\n");

    if (ctx->conn_fd < 0)
        return FALSE;

    ctx->abort = TRUE;

#ifdef USE_NICE_ABORT
    debug("Performing nice abort\n");
    if (OBEX_CancelRequest(ctx->handle, TRUE) != 0)
        return FALSE;
    ctx->abort_sent = TRUE;
    return TRUE;
#else
    debug("Performing abort through disconnection (without ABORT command)\n");
    ctx->done = TRUE;
    OBEX_CancelRequest(ctx->handle, FALSE);
    obex_link_error(ctx);
    return FALSE;
#endif
}

static gboolean gw_obex_request_sync(GwObex *ctx, obex_object_t *object) {
    /* Set sensible start values */
    ctx->done       = FALSE;
    ctx->abort      = FALSE;
    ctx->abort_sent = FALSE;
    ctx->counter    = 0;

    if (OBEX_Request(ctx->handle, object) < 0) {
        debug("OBEX_Request() failed\n");
        ctx->error = GW_OBEX_ERROR_INTERNAL;
        return FALSE;
    }

    while (TRUE) {
        int ret;

        ret = OBEX_HandleInput(ctx->handle, 10);

        if (ctx->done)
            break;

        if (ctx->cancel_cb && ctx->cancel_cb(ctx->cancel_data)) {
            debug("cancel_cb() returned TRUE, aborting.\n");
            if (ctx->abort_sent)
                continue;
            if (!gw_obex_do_abort(ctx))
                break;
            /* Must continue to receive the abort reply */
            continue;
        }

        if (ret < 0) {
            debug("OBEX_HandleInput() failed\n");
            ctx->error = GW_OBEX_ERROR_INTERNAL;
            return FALSE;
        }
        else if (ret == 0) { /* Timeout */
            debug("OBEX_HandleInput(): timeout\n");
            ctx->error = GW_OBEX_ERROR_TIMEOUT;
            obex_link_error(ctx);
            return FALSE;
        }
        
        debug("gw_obex_request_sync(): looping\n");
    }

    gw_obex_set_error(ctx);

    if (ctx->error == OBEX_RSP_SUCCESS) {
        /* It is possible that a EV_PROGRESS doesn't arrive after all data has
         * been transfered. Call pr_cb here to ensure app gets 100% progress */
        if (ctx->report_progress && ctx->pr_cb)
            ctx->pr_cb(ctx, ctx->obex_op, ctx->counter, ctx->counter, ctx->pr_data);
        return TRUE;
    }
    else
        return FALSE;
}

#ifdef DEBUG
static const char *optostr(uint8_t op) {
    switch (op) {
        case OBEX_CMD_CONNECT:
            return "Connect";
        case OBEX_CMD_DISCONNECT:
            return "Disconnect";
        case OBEX_CMD_PUT:
            return "Put";
        case OBEX_CMD_GET:
            return "Get";
        case OBEX_CMD_COMMAND:
            return "Command";
        case OBEX_CMD_SETPATH:
            return "SetPath";
        case OBEX_CMD_ABORT:
            return "Abort";
        case OBEX_CMD_ACTION:
            return "Action";
        default:
            return "(unknown)";
    }
}
#endif

static void obex_connect_done(GwObex *ctx, obex_object_t *object, int obex_rsp) {
    obex_headerdata_t hv;
    uint8_t hi;
    unsigned int hlen;
    uint8_t *ptr;

    if (OBEX_ObjectGetNonHdrData(object, &ptr)
            != sizeof(obex_connect_hdr_t)) {
        debug("Invalid packet content.\n");
    }
    else {
        obex_connect_hdr_t *nonhdrdata = (obex_connect_hdr_t *)ptr;
        uint16_t mtu = g_ntohs(nonhdrdata->mtu);
        int new_size;
        debug("Version: 0x%02x. Flags: 0x%02x  OBEX packet length: %d\n",
                nonhdrdata->version, nonhdrdata->flags, mtu);
        new_size = mtu - 200;
        /*OBEX_SetTransportMTU(ctx->handle, mtu, mtu);*/
        if (new_size < ctx->chunk_size) {
            debug("Resizing stream chunks to %d\n", new_size);
            ctx->chunk_size = new_size;
        }
    }

    while (OBEX_ObjectGetNextHeader(ctx->handle, object, &hi, &hv, &hlen)) {
        switch (hi) {
#ifdef DEBUG
            case OBEX_HDR_WHO:
                {
                    char *str;
                    str = bytestr(hv.bs, hlen);
                    debug("WHO header (UUID): %s\n", str);
                    g_free(str);
                }
                break;
#endif
            case OBEX_HDR_CONNECTION:
                ctx->conid = hv.bq4;
                debug("got Conection ID: %#x\n", hv.bq4);
                break;
            default:
                debug("Skipped header %02x\n", hi);
                break;
        }
    }
}

#ifdef DEBUG
static void show_headers(obex_t *handle, obex_object_t *object) {
    obex_headerdata_t hv;
    uint8_t hi;
    unsigned int hlen;

    while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
        char *str;

        switch (hi) {
            case OBEX_HDR_WHO:
                debug("OBEX_HDR_WHO\n");
                break;
            case OBEX_HDR_CONNECTION:
                debug("OBEX_HDR_CONNECTION: %#x\n", hv.bq4);
                break;
            case OBEX_HDR_LENGTH:
                debug("OBEX_HDR_LENGTH: %d\n", hv.bq4);
                break;
            case OBEX_HDR_NAME:
                str = g_utf16_to_utf8((gunichar2 *)hv.bs, hlen, NULL, NULL, NULL);
                if (str) {
                    debug("OBEX_HDR_NAME: %s\n", str);
                    g_free(str);
                }
                break;
            case OBEX_HDR_AUTHCHAL:
                str = bytestr(hv.bs, hlen);
                debug("OBEX_HDR_AUTHCHAL: %s\n", str);
                g_free(str);
                break;
            case OBEX_HDR_TIME:
                str = g_strdup_printf("OBEX_HDR_TIME: %%%us\n", hlen);
                debug(str, hv.bs);
                g_free(str);
                break;
            case OBEX_HDR_TYPE:
                debug("OBEX_HDR_TYPE: %s\n", hv.bs);
                break;
            default:
                debug("Skipped header 0x%02x\n", hi);
                break;
        }
    }

    OBEX_ObjectReParseHeaders(handle, object);
}
#endif

static void obex_abort_done(GwObex *ctx, obex_object_t *object,
                            int obex_cmd, int obex_rsp) {
    ctx->done = TRUE;
    ctx->abort_sent = FALSE;

    if (obex_rsp != OBEX_RSP_SUCCESS)
        debug("ABORT of %s command (0x%02x) failed: %s (0x%02x)\n",
                optostr((uint8_t)obex_cmd), (uint8_t)obex_cmd,
                OBEX_ResponseToString(obex_rsp), (uint8_t)obex_rsp);
    else
        debug("ABORT of %s command (0x%02x) succeeded.\n",
                optostr((uint8_t)obex_cmd), (uint8_t)obex_cmd);
}

static void obex_request_done(GwObex *ctx, obex_object_t *object,
                              int obex_cmd, int obex_rsp) {
    ctx->done = TRUE;
    ctx->obex_rsp = obex_rsp;

    if (obex_rsp != OBEX_RSP_SUCCESS) {
        debug("%s command (0x%02x) failed: %s (0x%02x)\n",
                optostr((uint8_t)obex_cmd), (uint8_t)obex_cmd,
                OBEX_ResponseToString(obex_rsp), (uint8_t)obex_rsp);
#ifdef DEBUG
        if (obex_rsp == OBEX_RSP_UNAUTHORIZED) {
            debug("Showing headers..\n");
            show_headers(ctx->handle, object);
        }
#endif
        return;
    }

    debug("%s command (0x%02x) succeeded.\n", optostr((uint8_t)obex_cmd),
            (uint8_t)obex_cmd);

    switch (obex_cmd) {
        case OBEX_CMD_CONNECT:
            obex_connect_done(ctx, object, obex_rsp);
            break;
        default:
            break;
    }
}

static void get_target_size_and_time(obex_t *handle, obex_object_t *object,
                                     gint *size, time_t *time) {
    obex_headerdata_t hv;
    uint8_t hi;
    unsigned int hlen;

    *size = GW_OBEX_UNKNOWN_LENGTH;
    *time = -1;

    while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
        switch (hi) {
            case OBEX_HDR_LENGTH:
                *size = hv.bq4; //(gint) g_ntohl(hv.bq4);
                break;
            case OBEX_HDR_TIME:
                *time = parse_iso8601((char *)hv.bs, hlen);
                break;
            default:
                break;
        }
    }

    OBEX_ObjectReParseHeaders(handle, object);
}

static void obex_readstream(GwObex *ctx, obex_object_t *object) {
    const uint8_t *buf;
    int actual;

    if (ctx->counter == 0)
        get_target_size_and_time(ctx->handle, object,
                                 &ctx->target_size, &ctx->modtime);

#ifdef DEBUG
    if (ctx->counter == 0)
        show_headers(ctx->handle, object);
#endif
    
    actual = OBEX_ObjectReadStream(ctx->handle, object, &buf);
    if (actual > 0) {
        ctx->counter += actual;
        if (ctx->stream_fd >= 0)
            (void) write(ctx->stream_fd, buf, actual);
        else {
            ctx->buf = g_realloc(ctx->buf, ctx->counter);
            memcpy(&ctx->buf[ctx->buf_size], buf, actual);
            ctx->buf_size = ctx->counter;
        }
    }
    else
        debug("Error or no data on OBEX stream\n");
}

static void obex_writestream(GwObex *ctx, obex_object_t *object) {
    obex_headerdata_t hv;
    int actual = -1;

    if (ctx->stream_fd >= 0) {
        actual = read(ctx->stream_fd, ctx->buf, ctx->chunk_size);
        hv.bs = ctx->buf;
#ifdef TEST_ABORT
        if (ctx->counter > 4000)
            actual = -1;
#endif
        if (actual > 0)
            OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_BODY,
                    hv, actual, OBEX_FL_STREAM_DATA);
        else if (actual == 0) /* EOF */
            OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_BODY,
                    hv, 0, OBEX_FL_STREAM_DATAEND);
        else { /* error reading file */
            debug("read(): %s\n", strerror(errno));
            gw_obex_do_abort(ctx);
        }
    }
    else {
        if (ctx->counter < ctx->buf_size) {
            if (ctx->buf_size > ctx->counter + ctx->chunk_size)
                actual = ctx->chunk_size;
            else
                actual = ctx->buf_size - ctx->counter;
            hv.bs = &ctx->buf[ctx->counter]; 
            OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_BODY,
                    hv, actual, OBEX_FL_STREAM_DATA);
        }
        else {
            hv.bs = NULL;
            OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_BODY,
                    hv, 0, OBEX_FL_STREAM_DATAEND);
        }
    }

    if (actual > 0)
        ctx->counter += actual;
}

static void obex_event_handler(obex_t *handle, obex_object_t *object, int mode,
                               int event, int obex_cmd, int obex_rsp) {
    GwObex *ctx = OBEX_GetCustomData(handle);
    switch (event) {
        case OBEX_EV_ABORT:
            debug("OBEX_EV_ABORT\n");
            obex_abort_done(ctx, object, obex_cmd, obex_rsp);
            break;
        case OBEX_EV_PROGRESS:
            debug("OBEX_EV_PROGRESS\n");
            if (ctx->report_progress && ctx->pr_cb)
                ctx->pr_cb(ctx, ctx->obex_op, ctx->counter, ctx->target_size, ctx->pr_data);
            break;
        case OBEX_EV_REQDONE:
            debug("OBEX_EV_REQDONE\n");
            obex_request_done(ctx, object, obex_cmd, obex_rsp);
            break;
        case OBEX_EV_REQ:
            debug("OBEX_EV_REQ: %s (0x%02x)\n",
                    optostr((uint8_t)obex_cmd), (uint8_t)obex_cmd);
            OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED,
                    OBEX_RSP_NOT_IMPLEMENTED);
            break;
        case OBEX_EV_REQHINT:
            debug("OBEX_EV_REQHINT: %s (0x%02x)\n",
                    optostr((uint8_t)obex_cmd), (uint8_t)obex_cmd);
            OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED,
                    OBEX_RSP_NOT_IMPLEMENTED);
            break;
        case OBEX_EV_LINKERR:
            debug("OBEX_EV_LINKERR\n");
            obex_link_error(ctx);
            break;
        case OBEX_EV_STREAMEMPTY:
            debug("OBEX_EV_STREAMEMPTY\n");
            obex_writestream(ctx, object);
            break;
        case OBEX_EV_STREAMAVAIL:
            debug("OBEX_EV_STREAMAVAIL\n");
            obex_readstream(ctx, object);
            break;
        case OBEX_EV_PARSEERR:
            debug("OBEX_EV_PARSEERR\n");
            break;
        default:
            debug("Unknown event %d\n", event);
            break;
    }
}

gboolean fd_raw_mode(int fd) {
    struct termios mode;

    memset(&mode, 0, sizeof (mode));
    if (tcgetattr(fd, &mode) < 0) {
        debug("tcgetattr(%d, &mode): %s", fd, strerror(errno));
        return FALSE;
    }

    mode.c_iflag = 0;
    mode.c_oflag &= ~OPOST;
    mode.c_lflag &= ~(ISIG | ICANON | ECHO
#ifdef XCASE
            | XCASE
#endif
            );
    mode.c_cc[VMIN] = 1;
    mode.c_cc[VTIME] = 0;

    if (tcsetattr(fd, TCSADRAIN, &mode) < 0) {
        debug("tcsetattr(%d, TCSADRAIN, &mode): %s", fd, strerror(errno));
        return FALSE;
    }

    return TRUE;
}

gboolean gw_obex_transport_setup(int fd, obex_t **handle) {
    *handle = OBEX_Init(OBEX_TRANS_FD, obex_event_handler, 0);
    if (*handle == NULL) {
        debug("OBEX_Init() failed\n");
        return FALSE;
    }

    (void) OBEX_SetTransportMTU(*handle, GW_OBEX_RX_MTU, GW_OBEX_TX_MTU);

    if (FdOBEX_TransportSetup(*handle, fd, fd, 0) < 0) {
        debug("FdOBEX_TransportSetup() failed\n");
        OBEX_Cleanup(*handle);
        return FALSE;
    }

    return TRUE;
}

void gw_obex_get_error(GwObex *ctx, gint *error) {
    *error = ctx->error;
    ctx->error = OBEX_RSP_SUCCESS;
}

gboolean gw_obex_cb(GIOChannel *chan, GIOCondition cond, gpointer data) {
    GwObex *ctx = (GwObex *)data;

    debug("gw_obex_cb(): entered\n");

    GW_OBEX_LOCK(ctx);

    if (ctx->conn_fd < 0) {
        debug("obex_cb() called, but connection is closed\n");
        obex_link_error(ctx);
        GW_OBEX_UNLOCK(ctx);
        if (ctx->dc_cb)
            ctx->dc_cb(ctx, ctx->dc_data);
        return FALSE;
    }

    if (cond != G_IO_IN) {
        switch (cond) {
            case G_IO_ERR:
                debug("Error on OBEX FD\n");
                break;
            case G_IO_HUP:
                debug("HUP on OBEX FD\n");
                break;
            case G_IO_NVAL:
                debug("OBEX FD was closed!\n");
                break;
            default:
                debug("Unknown condition (%d) on conn_fd (%d)\n",
                        cond, ctx->conn_fd);
                //g_assert_not_reached();
        }
        obex_link_error(ctx);

        GW_OBEX_UNLOCK(ctx);
        if (ctx->dc_cb)
            ctx->dc_cb(ctx, ctx->dc_data);
        return FALSE;
    }

    OBEX_HandleInput(ctx->handle, 0);

    GW_OBEX_UNLOCK(ctx);

    return TRUE;
}

gboolean gw_obex_disconnect(GwObex *ctx) {
    obex_object_t *object;

    object = OBEX_ObjectNew(ctx->handle, OBEX_CMD_DISCONNECT);

    if (ctx->conid != CONID_INVALID) {
        obex_headerdata_t hv;
        hv.bq4 = ctx->conid;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_CONNECTION, hv, 4, 0);
    }

    return gw_obex_request_sync(ctx, object);
}

gboolean gw_obex_connect(GwObex *ctx, const char *target, size_t target_len) {
    gboolean ret;
    obex_object_t *object;

    ctx->obex_op = OBEX_CMD_CONNECT;

    object = OBEX_ObjectNew(ctx->handle, OBEX_CMD_CONNECT);
    if (target) {
        obex_headerdata_t hv;
        hv.bs = (const unsigned char *)target;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_TARGET, hv, target_len, OBEX_FL_FIT_ONE_PACKET);
    }

    ret = gw_obex_request_sync(ctx, object);
    ctx->obex_op = OBEX_CMD_NONE;
    return ret;
}

GwObex *make_context(obex_t *handle) {
    GwObex *context;

    context = g_new0(GwObex, 1);

    context->handle      = handle;
    context->conn_fd     = OBEX_GetFD(handle);
    context->conid       = CONID_INVALID;
    context->stream_fd   = -1;
    context->chunk_size  = GW_OBEX_TX_MTU - 200;
    context->obex_op     = OBEX_CMD_NONE;
    context->obex_rsp    = OBEX_RSP_SUCCESS;
    context->target_size = GW_OBEX_UNKNOWN_LENGTH;
    context->modtime     = -1;

    return context;
}

gboolean gw_obex_action_op(GwObex *ctx, const gchar *src, const gchar *dst,
                           uint8_t action) {
    obex_object_t *object;
    obex_headerdata_t hv;
    gunichar2 *uname;
    glong uname_len;

    g_assert(src && dst);

    ctx->obex_op = OBEX_CMD_ACTION;

    object = OBEX_ObjectNew(ctx->handle, OBEX_CMD_ACTION);

    if (ctx->conid != CONID_INVALID) {
        hv.bq4 = ctx->conid;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_CONNECTION, hv, 4, 0);
    }

    hv.bq1 = action;
    OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_ACTION_ID, hv, 1, 0);

    uname_len = get_uname(&uname, src);
    if (uname_len < 0) {
        OBEX_ObjectDelete(ctx->handle, object);
        return FALSE;
    }
    hv.bs = (unsigned char *)uname;
    OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_NAME, hv, uname_len, 0);
    g_free(uname);

    uname_len = get_uname(&uname, dst);
    if (uname_len < 0) {
        OBEX_ObjectDelete(ctx->handle, object);
        return FALSE;
    }
    hv.bs = (unsigned char *)uname;
    OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_DESTNAME, hv, uname_len, 0);
    g_free(uname);

    ctx->obex_op = OBEX_CMD_NONE;

    return gw_obex_request_sync(ctx, object);
}

gboolean gw_obex_setpath(GwObex *ctx, const gchar *path, int flags) {
    gboolean ret, is_name;
    obex_headerdata_t hv;
    obex_object_t *object;
    obex_setpath_hdr_t nonhdrdata;
    gunichar2 *uname; 
    glong uname_len;

    ctx->obex_op = OBEX_CMD_SETPATH;

    nonhdrdata.flags = 0x02;
    nonhdrdata.constants = 0;

    if (path == NULL || *path == '\0') {
        /* empty name. change to root directory */
        uname = (gunichar2 *)g_strdup("");
        uname_len = 0;
        is_name = TRUE;
    } else if (strcmp(path, "..") == 0) {
        /* move up one directory */
        nonhdrdata.flags = 0x03;
        is_name = FALSE;
        uname = NULL;
        uname_len = 0;
    } else {
        /* normal directory change */
        uname_len = get_uname(&uname, path);
        is_name = TRUE;
    }

    if (flags & SETPATH_CREATE)
        nonhdrdata.flags &= ~0x02;

    object = OBEX_ObjectNew(ctx->handle, OBEX_CMD_SETPATH);
    OBEX_ObjectSetNonHdrData(object, (uint8_t*)&nonhdrdata, 2);

    if (ctx->conid != CONID_INVALID) {
        hv.bq4 = ctx->conid;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_CONNECTION, hv, 4, 0);
    }

    if (is_name) {
        if (uname_len < 0) {
            OBEX_ObjectDelete(ctx->handle, object);
            return FALSE;
        }
        hv.bs = (unsigned char *)uname;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_NAME, hv, uname_len, 0);
    }

    ret = gw_obex_request_sync(ctx, object);
    if (uname)
        g_free(uname);
    
    ctx->obex_op = OBEX_CMD_NONE;
    
    return ret;
}

gboolean gw_obex_get(GwObex *ctx,
                     const gchar *local, const gchar *remote, const gchar *type,
                     gchar **buf, gint *buf_size) {
    gboolean ret;
    gboolean user_fd = FALSE;
    obex_headerdata_t hv;
    obex_object_t *object;
#ifdef DEBUG
    time_t tm;
#endif

    g_assert(local || buf || ctx->stream_fd > 0);
    g_assert(remote || type);

    ctx->obex_op = OBEX_CMD_GET;

    object = OBEX_ObjectNew(ctx->handle, OBEX_CMD_GET);

    ctx->target_size = GW_OBEX_UNKNOWN_LENGTH;

    if (ctx->conid != CONID_INVALID) {
        hv.bq4 = ctx->conid;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_CONNECTION, hv, 4, 0);
    }

    if (type) {
        hv.bs = (unsigned char *)type;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_TYPE, hv, strlen(type) + 1, 0);
    }

    if (ctx->stream_fd >= 0)
        user_fd = TRUE;

    if (remote) {
        gunichar2 *uname;
        glong uname_len;
        uname_len = get_uname(&uname, remote);
        if (uname_len < 0) {
            OBEX_ObjectDelete(ctx->handle, object);
            ctx->error = GW_OBEX_ERROR_INVALID_PARAMS;
            return FALSE;
        }
        hv.bs = (unsigned char *)uname;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_NAME, hv, uname_len, 0);
        g_free(uname);
    }

    if (local) {
        ctx->stream_fd = open(local, O_WRONLY | O_CREAT, 0600);
        if (ctx->stream_fd < 0) {
            if (errno == ENOENT || errno == ENODEV)
                ctx->error = GW_OBEX_ERROR_INVALID_PARAMS;
            else
                ctx->error = GW_OBEX_ERROR_LOCAL_ACCESS;
            debug("open(%s): %s", local, strerror(errno));
            OBEX_ObjectDelete(ctx->handle, object);
            return FALSE;
        }
    }
    else if (!user_fd) {
        g_assert(buf && buf_size);
        ctx->buf = NULL;
        ctx->buf_size = 0;
        ctx->stream_fd = -1;
    }

    ctx->report_progress = TRUE;

    OBEX_ObjectReadStream(ctx->handle, object, NULL);

#ifdef DEBUG
    tm = time(NULL);
#endif
    ret = gw_obex_request_sync(ctx, object);
    if (ctx->stream_fd >= 0) {
        if (!user_fd)
            close(ctx->stream_fd);
        ctx->stream_fd = -1;
    }
    if (ret == FALSE) {
        if (local)
            unlink(local);
        if (ctx->buf) {
            g_free(ctx->buf);
            ctx->buf = NULL;
            ctx->buf_size = 0;
        }
    }
    else {
        if (local) {
            debug("%s stored in %s\n", remote ? remote : type, local);
            if (ctx->modtime != -1) {
                struct utimbuf ubuf;
                ubuf.actime = time(NULL);
                ubuf.modtime = ctx->modtime;
                if (utime(local, &ubuf) < 0)
                    debug("utime(%s): %s\n", local, g_strerror(errno));
            }
        }
        if (buf) {
            *buf = (gchar *)ctx->buf;
            *buf_size = ctx->buf_size;
            ctx->target_size = ctx->buf_size;
            ctx->buf = NULL;
            ctx->buf_size = 0;
        }
    }

#ifdef DEBUG
    tm = time(NULL) - tm;
    if (tm > 0 && ctx->target_size > 0) {
        double speed = (double)ctx->target_size / tm;
        if (speed < 1000)
            debug("transfer speed: %.2f byte/s\n", speed);
        else
            debug("transfer speed: %.2f kB/s\n", speed / 1024);
    }
    else if (ctx->target_size > 0) {
        debug("%d bytes transfered in less than one second\n", ctx->target_size);
    }
#endif

    ctx->report_progress = FALSE;
    ctx->target_size = GW_OBEX_UNKNOWN_LENGTH;
    ctx->modtime     = -1;

    ctx->obex_op = OBEX_CMD_NONE;

    return ret;
}

gboolean gw_obex_put(GwObex *ctx,
                     const gchar *local, const gchar *remote, const gchar *type,
                     const gchar *buf, gint buf_size) {
    gboolean ret;
    gboolean user_fd = FALSE;
    obex_headerdata_t hv;
    obex_object_t *object;
    gunichar2 *uname = NULL;
    glong uname_len = 0;
#ifdef DEBUG
    time_t tm;
#endif

    g_assert(remote || type);

    ctx->obex_op = OBEX_CMD_PUT;

    if (remote) {
        uname_len = get_uname(&uname, remote);
        if (uname_len < 0) {
            ctx->error = GW_OBEX_ERROR_INVALID_PARAMS;
            return FALSE;
        }
    }

    if (ctx->stream_fd >= 0)
        user_fd = TRUE;

    ctx->report_progress = TRUE;

    if (local) {
        if (file_is_dir(local)) {
            debug("Trying to PUT a directory\n");
            if (uname)
                g_free(uname);
            ctx->error = GW_OBEX_ERROR_INVALID_PARAMS;
            return FALSE;
        }

        ctx->stream_fd = open(local, O_RDONLY);
        if (ctx->stream_fd < 0) {
            if (errno == ENOENT || errno == ENODEV)
                ctx->error = GW_OBEX_ERROR_INVALID_PARAMS;
            else
                ctx->error = GW_OBEX_ERROR_LOCAL_ACCESS;
            debug("open(%s): %s", local, strerror(errno));
            if (uname)
                g_free(uname);
            return FALSE;
        }
        g_assert(ctx->buf == NULL);
        ctx->buf = g_malloc(ctx->chunk_size);
        ctx->buf_size = ctx->chunk_size;
        debug("Sending %s to %s\n", local, remote ? remote : type);
    }
    else if (buf) {
        ctx->buf = (unsigned char *)buf;
        ctx->buf_size = buf_size;
        ctx->stream_fd = -1;
        debug("Sending to %s\n", remote ? remote : type);
    }
    else if (user_fd) {
        ctx->buf = NULL;
        ctx->buf_size = 0;
    }
    else { /* Delete */
        ctx->report_progress = FALSE;
        ctx->stream_fd = -1;
        ctx->buf = NULL;
        ctx->buf_size = 0;
        debug("Deleting %s\n", remote ? remote : type);
    }

    object = OBEX_ObjectNew(ctx->handle, OBEX_CMD_PUT);

    if (ctx->conid != CONID_INVALID) { 
        hv.bq4 = ctx->conid;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_CONNECTION, hv, 4, 0);
    }

    if (uname) {
        hv.bs = (unsigned char *)uname;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_NAME, hv, uname_len, 0);
        g_free(uname);
    }

    if (type) {
        hv.bs = (unsigned char *)type;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_TYPE, hv, strlen(type) + 1, 0);
    }

    if (ctx->stream_fd >= 0 || buf) {
        gint size = -1;
        time_t time = -1;

        if (buf)
            size = buf_size;
        else {
            struct stat stats;
            if (fstat(ctx->stream_fd, &stats) == 0) {
                size = stats.st_size;
                time = stats.st_mtime;
            }
        }

        if (time >= 0) {
            char tstr[17];
            int len;

            len = make_iso8601(time, tstr, sizeof(tstr));

            if (len >= 0) {
                debug("Adding time header: %s\n", tstr);
                hv.bs = (unsigned char *)tstr;
                OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_TIME, hv, len, 0);
            }
        }

        if (size >= 0) {
            ctx->target_size = size;
            debug("Adding size header: %d\n", size);
            hv.bq4 = (uint32_t)size;
            OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_LENGTH, hv, 4, 0);
        }
        else
           ctx->target_size = GW_OBEX_UNKNOWN_LENGTH; 

        hv.bs = NULL;
        OBEX_ObjectAddHeader(ctx->handle, object, OBEX_HDR_BODY, hv, 0, OBEX_FL_STREAM_START);
    }

#ifdef DEBUG
    tm = time(NULL);
#endif
    ret = gw_obex_request_sync(ctx, object);

    if (ctx->stream_fd >= 0) {
        if (!user_fd)
            close(ctx->stream_fd);
        ctx->stream_fd = -1;
        if (ctx->buf) {
            g_free(ctx->buf);
            ctx->buf = NULL;
        }
    }
#ifdef DEBUG
    if (ret == TRUE) {
        tm = time(NULL) - tm;
        if (tm > 0 && ctx->target_size > 0) {
            double speed = (double)ctx->target_size / tm;
            if (speed < 1024)
                debug("transfer speed: %.2f byte/s\n", speed);
            else
                debug("transfer speed: %.2f kB/s\n", speed / 1024);
        }
        else if (ctx->target_size > 0)
            debug("%d bytes transfered in less than one second\n", ctx->target_size);
    }
#endif

    ctx->report_progress = FALSE;
    ctx->buf = NULL;
    ctx->buf_size = 0;
    ctx->target_size = GW_OBEX_UNKNOWN_LENGTH;

    ctx->obex_op = OBEX_CMD_NONE;

    return ret;
}

