/**
  @file sdp-query.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <errno.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#include "log.h"

struct uuid16_map_t {
    const char *svc;
    uint16_t uuid;
} uuid16_map[] = {
    { "SPP", SERIAL_PORT_SVCLASS_ID    },
    { "DUN", DIALUP_NET_SVCLASS_ID     },
    { "FTP", OBEX_FILETRANS_SVCLASS_ID },
    { "OPP", OBEX_OBJPUSH_SVCLASS_ID   },
    { "SAP", SAP_SVCLASS_ID            },
    { NULL,  0 }
};

static const unsigned char
nokia_ftp_uuid[16] = { 0x00, 0x00, 0x50, 0x05, 0x00, 0x00, 0x10, 0x00,
                       0x80, 0x00, 0x00, 0x02, 0xee, 0x00, 0x00, 0x01 };

static const char *svc_from_uuid(uuid_t *uuid) {
    struct uuid16_map_t *u16e;

    switch (uuid->type) {
        case SDP_UUID16:
            for (u16e = uuid16_map; u16e->svc != NULL; u16e++) {
                if (u16e->uuid == uuid->value.uuid16)
                    return u16e->svc;
            }
            return NULL;
        case SDP_UUID32:
            for (u16e = uuid16_map; u16e->svc != NULL; u16e++) {
                if (u16e->uuid == uuid->value.uuid32)
                    return u16e->svc;
            }
            return NULL;
        case SDP_UUID128:
            if (memcmp(&uuid->value.uuid128, nokia_ftp_uuid, 16) == 0)
                return "NFTP";
        default:
           return NULL; 
    }
}

static uuid_t *uuid_from_svc(uuid_t *uuid, const char *svc) {
    struct uuid16_map_t *u16e;

    for (u16e = uuid16_map; u16e->svc != NULL; u16e++) {
        if (g_str_equal(u16e->svc, svc)) {
            sdp_uuid16_create(uuid, u16e->uuid);
            return uuid;
        }
    }

    if (g_str_equal(svc, "NFTP")) {
        sdp_uuid128_create(uuid, nokia_ftp_uuid);
        return uuid;
    }

    return NULL;
}

#if 0
static int disconnect(int sock, uint16_t handle) {
    return hci_disconnect(sock, handle, HCI_OE_USER_ENDED_CONNECTION, 100);
}

static int create_connection(int sock, bdaddr_t *ba, uint16_t clk, uint16_t *handle) {
    uint8_t role;
    int ptype;

    role = 0;
    ptype = HCI_DM1 | HCI_DM3 | HCI_DM5 | HCI_DH1 | HCI_DH3 | HCI_DH5;

    (void) hci_create_connection(sock, ba, ptype, clk, role, handle, 1000);

    return 0;
}
#endif

static void catch_uuid(uuid_t *uuid, char **service) {
    const char *svc;

    if (*service)
        return;

    svc = svc_from_uuid(uuid);
    if (svc == NULL)
        return;

    *service = g_strdup(svc);
}

static void catch_channel(sdp_data_t *p, int *channel) {
    int proto = 0;

    if (*channel >= 0)
        return;

    for (; p; p = p->next) {
        switch (p->dtd) {
            case SDP_UUID16:
            case SDP_UUID32:
            case SDP_UUID128:
                proto = sdp_uuid_to_proto(&p->val.uuid);
                break;
            case SDP_UINT8:
                if (*channel < 0 && proto == RFCOMM_UUID)
                    *channel = (uint8_t)(p->val.uint8);
                break;
            default:
                break;
        }
    }
}

static void access_protos(sdp_list_t *protDescSeq, void *data) {
    sdp_list_foreach(protDescSeq, (sdp_list_func_t)catch_channel, data);
}   

static void add_new_service(sdp_record_t *rec) {
    sdp_list_t *classes = NULL, *protos = NULL;
    char *service = NULL;
    int channel = -1;

    if (sdp_get_service_classes(rec, &classes) == 0) {
        sdp_list_foreach(classes, (sdp_list_func_t)catch_uuid, &service);
        sdp_list_free(classes, free);
    }

    if (service == NULL)
        return;

    /*debug("found %s", service);*/

    if (sdp_get_access_protos(rec, &protos) == 0) {
        sdp_list_foreach(protos, (sdp_list_func_t)access_protos, &channel);
        sdp_list_free(protos, (sdp_free_func_t)sdp_data_free);
    }

    if (channel < 0) {
        g_free(service);
        return;
    }

    printf("r|%s|%d\n", service, channel);
    g_free(service);
    fflush(stdout);
}

static void get_sdp_info(uuid_t *uuid, sdp_session_t *sess) {
    uint32_t range = 0x0000ffff;
    sdp_list_t *attrid, *search, *seq, *next;

    attrid = sdp_list_append(0, &range);
    search = sdp_list_append(0, uuid);
    if (sdp_service_search_attr_req(sess, search, SDP_ATTR_REQ_RANGE, attrid, &seq)) {
        error("Service Discovery failed: %s\n", g_strerror(errno));
        return;
    }
    sdp_list_free(attrid, 0);
    sdp_list_free(search, 0);

    for (; seq; seq = next) {
        sdp_record_t *rec = (sdp_record_t *) seq->data;
        /*struct search_context sub_context;*/
    
        add_new_service(rec);

#if 0
        if (sdp_get_group_id(rec, &sub_context.group) != -1) {
            /* Set the subcontext for browsing the sub tree */
            memcpy(&sub_context, context, sizeof(struct search_context));
            /* Browse the next level down if not done */
            if (sub_context.group.value.uuid16 != context->group.value.uuid16)
                do_search(bdaddr, &sub_context);
        }
#endif
        next = seq->next;
        free(seq);
        sdp_record_free(rec);
    }
}

static void query_error(void) {
    printf("Error\n");
    fflush(stdout);
}

int main (int argc, char *argv[]) {
    int c;
    bdaddr_t ba;
    sdp_session_t *sess;
    uint16_t clock;
    uuid_t *uuid;
    GPtrArray *arr;
   
    clock = 0;

    open_log("sdp-query", TRUE);

    arr = g_ptr_array_new();

    while ((c = getopt(argc, argv, "c:u:s:")) != -1) {
        switch (c) {
            case 's':
                uuid = g_new(uuid_t, 1);
                if (!uuid_from_svc(uuid, optarg)) {
                    g_free(uuid);
                    break;
                }
                g_ptr_array_add(arr, uuid);
                break;
            case 'c':
                clock = (uint16_t)strtol(optarg, NULL, 0);
                break;
            case 'u':
                uuid = g_new(uuid_t, 1);
                sdp_uuid16_create(uuid, (uint16_t)strtol(optarg, NULL, 0));
                g_ptr_array_add(arr, uuid);
                break;
            default:
                error("Unhandled option character: \'%c\'", c);
                break;
        }
    }

    if (optind == argc) {
        query_error();
        die("Not enough arguments");
    }

    if (str2ba(argv[optind], &ba) < 0) {
        query_error();
        die("Invalid BDA");
    }

    sess = sdp_connect(BDADDR_ANY, &ba, SDP_RETRY_IF_BUSY);
    if (sess == NULL) {
        query_error();
        die("SDP connect failed");
    }

    if (arr->len == 0) {
        uuid = g_new(uuid_t, 1);
        sdp_uuid16_create(uuid, PUBLIC_BROWSE_GROUP);
        g_ptr_array_add(arr, uuid);
    }

    g_ptr_array_foreach(arr, (GFunc)get_sdp_info, sess);

    sdp_close(sess); 

    exit(EXIT_SUCCESS);
}

