/**
  @file devlist.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/  
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <ctype.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btsearch-bt.h"
#include "dbus.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "inquiry.h"
#include "devlist.h"

extern GMainLoop *event_loop;

static char *list_receiver = NULL;

static void shutdown_and_exit() {
    inquiry_stop();
    if (!send_dbus_search_complete(get_dbus_connection()))
        error("Unable to send search_complete signal!");
    g_main_loop_quit(event_loop);
}

void stop_search(void) {
    inquiry_stop();
    shutdown_and_exit();
}

/** Start the search process
 * @param receiver D-BUS address of the list receiver, NULL to enable test mode
 * @returns TRUE on success, FALSE on failure
 */
gboolean init_search(const char *receiver) {
    if (receiver)
        list_receiver = g_strdup(receiver);
    else
        list_receiver = NULL;

    return inquiry_start();
}

/* Send "New device found" signal to D-BUS */
gboolean send_dbus_dev_found(DBusConnection *connection, DevInfo *dev) {
    DBusMessage *signal;

    debug("Found: BDA=%s, name=%s, class=0x%02x%02x%02x, rssi=0x%02x, clock=0x%04X",
            dev->bda, dev->name,
            dev->cls[2], dev->cls[1], dev->cls[0],
            dev->rssi, dev->clock);

    /* This happens in test mode */
    if (list_receiver == NULL)
        return TRUE;

    signal = new_dbus_signal(BTSEARCH_SIG_PATH,
                             BTSEARCH_SIG_INTERFACE,
                             BTSEARCH_DEV_FOUND_SIG,
                             list_receiver);

    append_dbus_args(signal,
                     DBUS_TYPE_STRING, dev->bda,
                     DBUS_TYPE_STRING, dev->name,
                     DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, dev->cls, 3,
                     DBUS_TYPE_BYTE, dev->rssi,
                     DBUS_TYPE_UINT32, (uint32_t)(dev->clock),
                     DBUS_TYPE_INVALID);

    return send_and_unref(connection, signal);
}

/* Send search_complete signal to D-BUS */
gboolean send_dbus_search_complete(DBusConnection *connection) {
    DBusMessage *signal;

    /* This happens in test mode */
    if (list_receiver == NULL)
        return TRUE;

    signal = new_dbus_signal(BTSEARCH_SIG_PATH,
                             BTSEARCH_SIG_INTERFACE,
                             BTSEARCH_SEARCH_COMPLETE_SIG,
                             list_receiver);

    return send_and_unref(connection, signal);
}


gboolean search_in_progress(void) {
    if (list_receiver == NULL)
        return FALSE;
    else
        return TRUE;
}

DBusHandlerResult service_exit_filter(DBusConnection *connection,
                                      DBusMessage *message,
                                      void *user_data) {
    char *service;

    if (list_receiver == NULL)
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    if (dbus_message_is_signal(message,
                                DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                                "ServiceDeleted")) {
        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceDeleted signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
    }
    else if (dbus_message_is_signal(message,
                                    DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                                    "ServiceOwnerChanged")) {
        char *old, *new;

        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_STRING, &old,
                    DBUS_TYPE_STRING, &new,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceOwnerChanged signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
        dbus_free(new);
        dbus_free(old);
    }
    else
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;


    if (g_str_equal(service, list_receiver)) {
        debug("List receiver exited");
        stop_search();
        dbus_free(service);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    dbus_free(service);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
                                      
