/**
  @file dbus-helper.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "config.h"

#ifdef USE_MCE
# include <mce/dbus-names.h>
#endif

#include "log.h"
#include "bt-error.h"
#include "dbus-helper.h"

#ifdef USE_MCE
static DBusHandlerResult mode_filter(DBusConnection *connection,
                                     DBusMessage    *message,
                                     void (*mode_cb)(const gchar *mode)) {
    char *mode;

    if (!dbus_message_is_signal(message,
                                MCE_SIGNAL_IF,
                                MCE_DEVICE_MODE_SIG))
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    if (!get_dbus_args(message,
                       DBUS_TYPE_STRING, &mode,
                       DBUS_TYPE_INVALID)) {
        error("Invalid arguments for device_mode_ind signal");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    mode_cb(mode);
    dbus_free(mode);

    return DBUS_HANDLER_RESULT_HANDLED;
}
#endif

char *dbus_strdup(const char *str) {
   char *new;

   if (str == NULL)
       return NULL;

   new = dbus_malloc0(strlen(str) + 1);
   if (new == NULL)
       die("Out of memory during dbus_malloc0!");
   strcpy(new, str);

   return new;
}

DBusMessage *new_dbus_signal(const char *path,
                             const char *interface,
                             const char *name,
                             const char *destination) {
    DBusMessage *signal;

    signal = dbus_message_new_signal(path, interface, name);
    if (signal == NULL)
        die("Out of memory during dbus_message_new_signal()");

    if (destination) {
        if (!dbus_message_set_destination(signal, destination))
            die("Out of memory during dbus_message_set_destination()");
    }

    return signal;
}

DBusMessage *new_dbus_method_call(const char *service,
                                  const char *path,
                                  const char *interface,
                                  const char *method) {
    DBusMessage *message;

    message = dbus_message_new_method_call(service, path, interface, method);
    if (message == NULL)
        die("Out of memory during dbus_message_new_method_call()");

    return message;
}

DBusMessage *new_dbus_method_return(DBusMessage *message) {
    DBusMessage *reply;

    reply = dbus_message_new_method_return(message);
    if (reply == NULL)
        die("Out of memory during dbus_message_new_method_return()");

    return reply;
}

DBusMessage *new_dbus_error(DBusMessage *message, const char *name, const char *msg) {
    DBusMessage *error;

    error = dbus_message_new_error(message, name, msg);
    if (error == NULL)
        die("Out of memory during dbus_message_new_error()");

    return error;
}

DBusMessage *new_dbus_error_gerr(DBusMessage *message, GError *err) {
    return new_dbus_error(message, bt_dbus_error(err), err->message);
}


gboolean send_and_unref(DBusConnection *connection, DBusMessage *message) {
    if (!dbus_connection_send(connection, message, NULL)) {
        dbus_message_unref(message);
        return FALSE;
    }

    dbus_connection_flush(connection);
    dbus_message_unref(message);

    return TRUE;
}

gboolean send_invalid_args(DBusConnection *connection, DBusMessage *message) {
    DBusMessage *reply;

    reply = new_dbus_error(message, DBUS_ERROR_INVALID_ARGS, NULL);

    return send_and_unref(connection, reply);
}

#if 0
void append_dbus_iter_args(DBusMessageIter *iter, int first_arg_type, ...) {
    dbus_bool_t ret;
    va_list ap;

    va_start(ap, first_arg_type);
    ret = dbus_message_iter_append_args_valist(iter, first_arg_type, ap);
    va_end(ap);

    if (ret == FALSE) {
        die("dbus_message_iter_append_args_valist failed");
    }
}
#endif

gboolean get_dbus_iter_args(DBusMessageIter *iter, int first_arg_type, ...) {
    DBusError derr;
    gboolean ret;
    va_list ap;

    dbus_error_init(&derr);

    va_start(ap, first_arg_type);
    ret = dbus_message_iter_get_args_valist(iter, &derr, first_arg_type, ap);
    va_end(ap);

    if (dbus_error_is_set(&derr)) {
        if (dbus_error_has_name(&derr, DBUS_ERROR_NO_MEMORY))
            die("Out of memory during dbus_message_iter_get_args");
        debug("dbus_message_iter_get_args failed: %s", derr.message);
        dbus_error_free(&derr);
    }
            
    return ret;
}

void append_dbus_args(DBusMessage *message, int first_arg_type, ...) {
    dbus_bool_t ret;
    va_list ap;

    va_start(ap, first_arg_type);
    ret = dbus_message_append_args_valist(message, first_arg_type, ap);
    va_end(ap);

    if (ret == FALSE)
        die("dbus_message_append_args failed");
}

gboolean get_dbus_args(DBusMessage *message, int first_arg_type, ...) {
    DBusError derr;
    gboolean ret;
    va_list ap;

    dbus_error_init(&derr);

    va_start(ap, first_arg_type);
    ret = dbus_message_get_args_valist(message, &derr, first_arg_type, ap);
    va_end(ap);

    if (dbus_error_is_set(&derr)) {
        if (dbus_error_has_name(&derr, DBUS_ERROR_NO_MEMORY))
            die("Out of memory during dbus_message_get_args");
        debug("dbus_message_get_args failed: %s\n", derr.message);
        dbus_error_free(&derr);
    }
            
    return ret;
}

gchar *get_device_mode(DBusConnection *connection) {
#ifdef USE_MCE
    DBusError derror;
    char *mode, *ret;
    DBusMessage *message, *reply;

    message = new_dbus_method_call(MCE_SERVICE,
                                   MCE_REQUEST_PATH,
                                   MCE_REQUEST_IF,
                                   MCE_DEVICE_MODE_GET);

    dbus_error_init(&derror);
    reply = dbus_connection_send_with_reply_and_block(connection,
                                                      message,
                                                      -1,
                                                      &derror);
    dbus_message_unref(message);
    if (dbus_error_is_set(&derror)) {
        error("Getting device mode from MCE failed: %s", derror.message);
        dbus_error_free(&derror);
        return NULL;
    }

    if (!get_dbus_args(reply,
                       DBUS_TYPE_STRING, &mode,
                       DBUS_TYPE_INVALID)) {
        error("Invalid arguments for MCE req_device_mode reply");
        dbus_message_unref(reply);
        return NULL;
    }

    dbus_message_unref(reply);
    ret = g_strdup(mode);
    dbus_free(mode);

    return ret;
#else /* USE_MCE */
    return g_strdup("normal");
#endif
}

#ifdef USE_MCE
gboolean add_mode_listener(DBusConnection *connection,
                           void (*mode_cb)(const gchar *mode)) {

    dbus_bus_add_match(connection, "interface=" MCE_SIGNAL_IF, NULL);
    return dbus_connection_add_filter(connection,
                                      (DBusHandleMessageFunction)mode_filter,
                                      mode_cb,
                                      NULL);
}
#endif
