/**
  @file btsearch.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <glib.h>
#include <glib-object.h>

#include "log.h"
#include "btsearch-bt.h"
#include "dbus.h"
#include "btsearch-dbus.h"
#include "devlist.h"
#include "bt-dbus.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

static char *program_name;

/** Startup config options */
static struct {
    gboolean use_syslog; /**< Use syslog (instead of stdout & stderr) */
    gboolean test_mode;  /**< Testing mode (without D-BUS) */
} cfg;

GMainLoop *event_loop = NULL;

/* For getopt */
static struct option const long_options[] = {
    {"help",     no_argument,       0, 'h'},
    {"stdio",    no_argument,       0, 's'},
    {"test",     no_argument,       0, 't'},
    {"version",  no_argument,       0, 'V'},
    {NULL, 0, NULL, 0}
};

/* Print usage information and exit with status */
static void usage(int status) {
    printf("%s - Bluetooth Search Application %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help                 Display this help and exit\n"
     "-s, --stdio                Use stdout & stderr instead of syslog\n"
     "-t, --test                 Test mode (don't use D-BUS), implies -s\n"
     "-V, --version              Output version information and exit\n"
     "\n", program_name);

    exit(status);
}

static void exit_cleanup(void) {
    report("Exiting.");
}

/* Process commandline options. Returns index of first
 * non-option argument */
static int decode_switches(int argc, char *argv[])
{
    int c;

    /* Set defaults */
    cfg.use_syslog = TRUE;
    cfg.test_mode  = FALSE;

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "s"   /* syslog    */
                    "t"   /* test      */
                    ,long_options, NULL)) != EOF) {
        switch (c) {
            case 'V':
                printf("Bluetooth Search Application %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 's':
                cfg.use_syslog = FALSE;
                break;

            case 't':
                cfg.test_mode  = TRUE;
                cfg.use_syslog = FALSE;
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

int main(int argc, char *argv[]) {
    int i;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    open_log("btsearch", cfg.use_syslog);

    atexit(exit_cleanup);

    event_loop = g_main_loop_new(NULL, FALSE);

    g_type_init();

    if (cfg.test_mode) {
        debug("Test mode enabled");
        init_search(NULL);
    }
    else if (!setup_dbus_connection(BTSEARCH_SERVICE, init_dbus_handlers))
        die("D-BUS connection setup failed!");

    report("BT Search %s started.", VERSION);

    /* Enter main loop */
    g_main_loop_run(event_loop);

    if (!cfg.test_mode)
        close_dbus_connection();

    g_main_loop_unref(event_loop);

    exit(EXIT_SUCCESS);
}

