/**
  @file btsdp-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btsdp-bt.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "btsdp-dbus.h"

extern GMainLoop *event_loop;

static gboolean query_in_progress = FALSE;

#define EXIT_TIMEOUT 5000 /* milliseconds */

static GSource *exit_timer = NULL;

static gboolean exit_cb(gpointer user_data) {
    error("btsdp running but no requests received.");
    g_main_loop_quit(event_loop);
    return FALSE;
}

static void set_exit_timer(void) {
    if (exit_timer)
        g_source_destroy(exit_timer);
    exit_timer = g_timeout_source_new(EXIT_TIMEOUT);
    g_source_set_callback(exit_timer, exit_cb, NULL, NULL);
    (void) g_source_attach(exit_timer, NULL);
}

static void remove_exit_timer(void) {
    if (exit_timer) {
        g_source_destroy(exit_timer);
        exit_timer = NULL;
    }
}

static void add_sdp_arg(sdp_info_t *info, DBusMessage *reply) {
    debug("%s, channel %u", info->service, info->channel);

    append_dbus_args(reply,
            DBUS_TYPE_STRING, info->service,
            DBUS_TYPE_BYTE, info->channel,
            DBUS_TYPE_INVALID);
}

static void send_rfcomm_services_reply(DBusMessage    *message,
                                       DBusConnection *connection,
                                       GError         *err, 
                                       GSList         *services) {
    DBusMessage *reply;

    query_in_progress = FALSE;

    if (err) {
        reply = new_dbus_error_gerr(message, err);
        if(!send_and_unref(connection, reply))
            error("Unable to send connect_failed error reply");
        dbus_message_unref(message);
        g_main_loop_quit(event_loop);
        return;
    }

    debug("%d supported services", g_slist_length(services));

    reply = new_dbus_method_return(message);
    g_slist_foreach(services, (GFunc)add_sdp_arg, reply);
    if (!send_and_unref(connection, reply))
        error("Unable to send rfcomm services reply");

    g_slist_foreach(services, (GFunc)free_sdp_info, NULL);
    g_slist_free(services);

    dbus_message_unref(message);

    g_main_loop_quit(event_loop);
}

static DBusHandlerResult get_rfcomm_services_request(DBusMessage    *message,
                                                     DBusConnection *connection) {
    char *bda;
    uint32_t clock;
    GError *err = NULL;
    DBusMessage *reply;
    DBusMessageIter iter;
    GSList *l = NULL;
    sdp_cb_data_t *data;

    remove_exit_timer();

    debug("get_rfcomm_services_request()");

    if (query_in_progress) {
        reply = new_dbus_error(message, BTSDP_ERROR_QUERY_IN_PROGRESS, NULL);
        if(!send_and_unref(connection, reply))
            error("Unable to send query_in_progress error reply");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (!bt_ok()) {
        reply = new_dbus_error(message, BTSDP_ERROR_BT_DISABLED, NULL);
        if (!send_and_unref(connection, reply))
            error("Unable to send bt_disabled error reply");
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    dbus_message_iter_init(message, &iter);
    /* We can't use get_dbus_iter_args because after the call the iter points on the
     * second argument if the message had two arguments, but on the third argument if 
     * the message had more than two arguments */
    if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
        error("Invalid parameters for get_rfcomm_services");
        if (!send_invalid_args(connection, message))
            error("Unable to send invalid_args error");
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    bda = dbus_message_iter_get_string(&iter);
    if (bda == NULL)
        die("Out of memory during dbus_message_iter_get_string()");
    if (!dbus_message_iter_next(&iter)) {
        error("Invalid parameters for get_rfcomm_services");
        if (!send_invalid_args(connection, message))
            error("Unable to send invalid_args error");
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_UINT32) {
        error("Invalid parameters for get_rfcomm_services");
        if (!send_invalid_args(connection, message))
            error("Unable to send invalid_args error");
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }
    clock = dbus_message_iter_get_uint32(&iter);

    while (dbus_message_iter_next(&iter)) {
        SdpServiceId *id;

        if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
            char *svc;

            id = g_new(SdpServiceId, 1);
            id->type = SDP_SERVICE_ID_STRING;
            svc = dbus_message_iter_get_string(&iter);
            if (svc == NULL)
                die("Out of memory during dbus_message_iter_get_string()");
            debug("Adding string to search list: %s", svc);
            id->u.string = g_strdup(svc);

            dbus_free(svc);
        }
        else if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_UINT32) {
            uint32_t val;

            val = dbus_message_iter_get_uint32(&iter);

            debug("Adding UUID to search list: 0x%08x", val);

            /* We only support 16 bit UUID's */
            if (val > 0xFFFF)
                break;

            id = g_new(SdpServiceId, 1);
            id->type = SDP_SERVICE_ID_UUID;
            id->u.uuid = (uint16_t)val;
        }
        else
            break;

        l = g_slist_append(l, id);
    }

    data = g_new0(sdp_cb_data_t, 1);
    data->func       = send_rfcomm_services_reply;
    data->message    = message;
    data->connection = connection;

    dbus_message_ref(message);

    if (!get_services(bda, l, (uint16_t)clock, data, &err)) {
        dbus_message_unref(message);
        g_free(data);
        reply = new_dbus_error_gerr(message, err);
        if(!send_and_unref(connection, reply))
            error("Unable to send connect_failed error reply");
        g_clear_error(&err);
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    query_in_progress = TRUE;

    dbus_free(bda);

    return DBUS_HANDLER_RESULT_HANDLED;
}


/* Handler for D-Bus requests */
static DBusHandlerResult btsdp_req_handler(DBusConnection     *connection,
                                           DBusMessage        *message,
                                           void               *user_data) {
    const char *dest;

    if (dbus_message_get_type(message) != DBUS_MESSAGE_TYPE_METHOD_CALL)
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    dest = dbus_message_get_destination(message);
    if (!g_str_equal(dest, BTSDP_SERVICE)) {
        debug("Received D-Bus message not addressed to me.");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (dbus_message_is_method_call(message,
                                    BTSDP_REQ_INTERFACE,
                                    BTSDP_GET_RFCOMM_SERVICES_REQ))
        return get_rfcomm_services_request(message, connection);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable btsdp_req_vtable = {
    .message_function    = btsdp_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection) {
    dbus_bool_t ret;
    ret = dbus_connection_register_object_path(connection,
                                               BTSDP_REQ_PATH,
                                               &btsdp_req_vtable,
                                               NULL);
    if (ret == FALSE)
        error("register_object_path(req) failed");

    set_exit_timer();
}

void destroy_dbus_handlers(DBusConnection *connection) {
    dbus_connection_unregister_object_path(connection, BTSDP_REQ_PATH);
}
