/**
  @file btsdp-bt.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#include "log.h"
#include "bt-utils.h"
#include "bt-error.h"
#include "dbus.h"
#include "dbus-helper.h"
#include "btsdp-dbus.h"
#include "btsdp-bt.h"

#define SDP_QUERY "sdp-query"

typedef struct {
    gboolean       l2cap_uuid;
    char          *bda;
    uint16_t       clock;
    sdp_cb_data_t *data;
} sdp_query_data_t;

static GIOChannel *sdp_query(const char *bda, GSList *l, uint16_t clock) {
    char *cmd;
    GString *str;
    FILE *p;

    str = g_string_new("");

    g_string_printf(str, "%s -c 0x%04x", SDP_QUERY, clock);

    for (; l != NULL; l = l->next) {
        SdpServiceId *id = l->data;
        switch (id->type) {
            case SDP_SERVICE_ID_STRING:
                g_string_append_printf(str, " -s %s", id->u.string);
                break;
            case SDP_SERVICE_ID_UUID:
                g_string_append_printf(str, " -u 0x%04x", id->u.uuid);
                break;
        }
    }

    g_string_append_printf(str, " %s", bda);

    cmd = g_string_free(str, FALSE);

    debug("running sdp-query: %s", cmd);

    p = popen(cmd, "r");
    g_free(cmd);
    if (p == NULL)
        die("popen(%s): %s", SDP_QUERY, g_strerror(errno));

    return g_io_channel_unix_new(fileno(p));
}

static GIOChannel *sdp_query_single_uuid(const char *bda, uint16_t uuid, uint16_t clock) {
    GSList *l;
    SdpServiceId *id;
    GIOChannel *io;

    id = g_new(SdpServiceId, 1);
    id->type = SDP_SERVICE_ID_UUID;
    id->u.uuid = uuid;

    l = g_slist_append(NULL, id);

    io = sdp_query(bda, l, clock);

    g_slist_free(l);
    g_free(id);

    return io;
}

static void add_rfcomm_service(gchar *line, GSList **services) {
    gchar **tokens;
    gchar *service = NULL;
    gint channel = 0;

    tokens = g_strsplit(line, "|", 2);

    if (tokens[0] != NULL && tokens[1] != NULL) {
        service = g_strdup(tokens[0]);
        channel = atoi(tokens[1]);
    }

    g_strfreev(tokens);

    if (service) {
        sdp_info_t *new;

        new = g_new0(sdp_info_t, 1);
        new->type    = SDP_SERVICE_RFCOMM;
        new->service = service;
        new->channel = (uint8_t)channel;

        *services = g_slist_append(*services, new);
    }
}

static gboolean input_cb(GIOChannel *chan, GIOCondition cond,
                         sdp_query_data_t *d) {
    GIOStatus status;
    GError *err = NULL;
    gchar *line;
    gsize length;

    if (cond == G_IO_STATUS_EOF || cond == G_IO_STATUS_ERROR) {
        g_set_error(&err, BT_SDP_ERROR,
                    BT_SDP_ERROR_CONNECT_FAILED,
                    "sdp-query io error");
        d->data->func(d->data->message, d->data->connection, err, d->data->services);
        goto sdp_finish;
    }

    status = g_io_channel_read_line(chan, &line, &length, NULL, NULL);
    if (status != G_IO_STATUS_NORMAL) {
        if (status == G_IO_STATUS_AGAIN) {
            debug("got G_IO_STATUS_AGAIN");
            return TRUE;
        }
        if (status != G_IO_STATUS_EOF)
            error("Error reading line");
        if (d->data->services == NULL && d->l2cap_uuid) {
            GIOChannel *io;
            debug("Got 0 services. Trying again with L2CAP UUID");
            d->l2cap_uuid = FALSE;
            io = sdp_query_single_uuid(d->bda, L2CAP_UUID, d->clock);
            (void) g_io_add_watch(io, G_IO_IN | G_IO_HUP | G_IO_ERR,
                                  (GIOFunc)input_cb, d);
            g_io_channel_unref(chan);
            return FALSE;
        }

        d->data->func(d->data->message, d->data->connection, err, d->data->services);
        goto sdp_finish;
    }

    if (length == 0) {
        g_set_error(&err, BT_SDP_ERROR,
                    BT_SDP_ERROR_CONNECT_FAILED,
                    "Invalid data from sdp-query");                    
        debug("zero length line!?");
        d->data->func(d->data->message, d->data->connection, err, d->data->services);
        goto sdp_finish;
    }

    if (length < 4) {
        g_set_error(&err, BT_SDP_ERROR,
                    BT_SDP_ERROR_CONNECT_FAILED,
                    "Invalid data from sdp-query");
        debug("too short line from sdp-query");
        g_free(line);
        d->data->func(d->data->message, d->data->connection, err, d->data->services);
        goto sdp_finish;
    }

    /* Remove '\n' from the end */
    line[length - 1] = '\0';

    switch (line[0]) {
        case SDP_SERVICE_RFCOMM:
            add_rfcomm_service(&line[2], &d->data->services);
            break;
        case SDP_SERVICE_ERROR:
            g_free(line);
            g_set_error(&err, BT_SDP_ERROR,
                    BT_SDP_ERROR_CONNECT_FAILED,
                    "SDP error");
            d->data->func(d->data->message, d->data->connection, err, d->data->services);
            goto sdp_finish;
        default:
            debug("Strange line: %s", line);
            break;
    }

    g_free(line);

    return TRUE;

sdp_finish:
    g_clear_error(&err);
    g_free(d->bda);
    g_free(d->data);
    g_free(d);
    g_io_channel_unref(chan);
    return FALSE;
}

void free_sdp_info(sdp_info_t *info, gpointer user_data) {
    if (info->service)
        g_free(info->service);
    g_free(info);
}

gboolean get_services(const char *bda, GSList *l, uint16_t clock,
                      sdp_cb_data_t *data, GError **err) {
    sdp_query_data_t *d;
    GIOChannel *io;

    if (!bda_ok(bda)) {
        g_set_error(err, BT_SDP_ERROR,
                    BT_SDP_ERROR_CONNECT_FAILED,
                    "Invalid BDA: %s", bda);
        return FALSE;
    }

    d = g_new0(sdp_query_data_t, 1);
    d->bda = g_strdup(bda);
    d->clock = clock;
    d->data = data;
    if (l == NULL)
        d->l2cap_uuid = TRUE;

    io = sdp_query(bda, l, clock);
    (void) g_io_add_watch(io, G_IO_IN | G_IO_HUP | G_IO_ERR,
                          (GIOFunc)input_cb, d);

    return TRUE;
}

gboolean bt_ok(void) {
    gboolean ret;
    gchar *mode;

    mode = get_device_mode(get_dbus_connection());
    if (mode == NULL) {
        error("get_device_mode() returned NULL!");
        /* Assume that BT is usable */
        return TRUE;
    }

    if (g_str_equal(mode, "normal"))
        ret = TRUE;
    else
        ret = FALSE;

    g_free(mode);

    return ret;
}
