/**
  @file btcond.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include <glib.h>
#include <glib-object.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <dbus/dbus.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "daemon.h"
#include "btcond-bt.h"
#include "gconf.h"
#include "dbus.h"
#include "dbus-helper.h"
#include "btcond-dbus.h"
#include "btcond-hci.h"
#include "btcond-rfcomm.h"
#include "bt-dbus.h"
#include "bt-gconf.h"
#include "state.h"
#include "log.h"

#define PIDFILE "/var/run/btcond.pid"

static char *program_name;

static int signal_pipe[2];

/** Command line config options */
static struct {
    gboolean daemon;    /**< Enter daemon mode (detach to background) */
    gboolean syslog;    /**< Force logging to syslog */
} cfg;

static GMainLoop *event_loop = NULL;

/* For getopt */
static struct option const long_options[] = {
    {"deamon",   no_argument,       0, 'd'},
    {"syslog",   no_argument,       0, 'l'},
    {"help",     no_argument,       0, 'h'},
    {"version",  no_argument,       0, 'V'},
    {NULL, 0, NULL, 0}
};

/** Check for running hcid
 * @returns TRUE if hcid is running, FALSE if not
 */
static gboolean hcid_is_running(void) {
    char line[32], *ptr;
    FILE *p;

    p = popen("pidof hcid", "r");
    if (p == NULL) {
        error("popen(\"pidof hcid\"): %s", g_strerror(errno));
        return FALSE;
    }

    ptr = fgets(line, sizeof(line), p);
    pclose(p);
    if (ptr == NULL || !isdigit(line[0]))
        return FALSE;

    return TRUE;
}

/** Print usage information
 * @param status exit with status
 */
static void usage(int status) {
    printf("%s - Bluetooth Connection Deamon %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

#ifdef USE_MCE
    printf("+USE_MCE ");
#else
    printf("-USE_MCE ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help                 Display this help and exit\n"
     "-V, --version              Output version information and exit\n"
     "-d, --daemon               Send process to the background\n"
     "\n", program_name);

    exit(status);
}

/** Process commandline options.
 * @param argc Parameter given to main()
 * @param argv Parameter given to main()
 * @returns Index of first non-option argument
 */
static gint decode_switches(int argc, char *argv[])
{
    gint c;

    /* Set defaults */
    cfg.daemon   = FALSE;
    cfg.syslog   = FALSE;

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "d"   /* daemon    */
                    "l"   /* syslog    */
                    ,long_options, NULL)) != EOF) {
        switch (c) {
            case 'V':
                printf("Bluetooth Connection Daemon %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'd':
                cfg.daemon = TRUE;
                cfg.syslog = TRUE;
                break;

            case 'l':
                cfg.syslog = TRUE; 
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static void handle_signal(int sig) {
    report("Signal %d received: %s.", sig, g_strsignal(sig));
    bt_disconnect(NULL, TRUE);
    g_main_loop_quit(event_loop);
}

static void exit_cleanup(void) {
    (void) remove_pid(PIDFILE);
    report("Exiting.");
}

static void signal_handler(int sig) {   
        send(signal_pipe[1], &sig, sizeof(sig), MSG_DONTWAIT);
}   

static gboolean signal_cb(GIOChannel *chan, GIOCondition cond, gpointer data) {
    int fd;

    if (cond != G_IO_IN) {
        error("signal_cb: cond != G_IO_IN");
        handle_signal(SIGTERM);
        g_io_channel_unref(chan);
        return FALSE;
    }

    fd = g_io_channel_unix_get_fd(chan);
    g_assert(fd >= 0);

    for (;;) {
        int sig;

        if (recv(fd, &sig, sizeof(sig), MSG_DONTWAIT) != sizeof(sig))
            break;

        handle_signal(sig);
    }

    return TRUE;
}

static void bind_unix_signals(void) {
    GIOChannel *gio;

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, signal_pipe) < 0)
        die("socketpair: %s", g_strerror(errno));

    gio = g_io_channel_unix_new(signal_pipe[0]);
    g_io_add_watch(gio, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP,
                   signal_cb, NULL);

    if (signal(SIGINT, signal_handler) == SIG_ERR)
        die("signal(SIGINT) failed");
    if (signal(SIGTERM, signal_handler) == SIG_ERR)
        die("signal(SIGTERM) failed");
}

int main(int argc, char *argv[]) {
    GError *err = NULL;
    gchar *mode;
    gint i, old_pid;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    open_log("btcond", cfg.syslog);

    old_pid = check_pid(PIDFILE);
    if (old_pid)
        die("Unable to run: another instance running (PID %d)", old_pid);

    if (!hcid_is_running())
        die("Unable to run: hcid does not seem to be running");

    if (cfg.daemon)
        daemonize();

    write_pid(PIDFILE);
    atexit(exit_cleanup);

    bind_unix_signals();

    event_loop = g_main_loop_new(NULL, FALSE);

    g_type_init();

    if (!setup_dbus_connection(BTCOND_SERVICE, init_dbus_handlers))
        die("D-BUS connection setup failed!");

    if (!rfcomm_init(&err))
        die("rfcomm_init failed: %s", err->message);

    /* Listen for HCI events */
    if (!monitor_connections(&err))
        die("Opening raw HCI socket failed: %s", err->message);

#ifdef USE_MCE
    update_cover_state();
#endif

    mode = get_device_mode(get_dbus_connection());
    if (mode != NULL) {
        mode_change(mode);
        g_free(mode);
        mode = NULL;
    }
    else {
        error("Unable to determine actual device mode. Assuming normal mode.");
        mode_change("normal");
    }

    report("Bluetooth Connection Daemon %s started.", VERSION);

    /* Enter main loop */
    g_main_loop_run(event_loop);

    hci_bt_down();

    close_dbus_connection();
    g_main_loop_unref(event_loop);

    exit(EXIT_SUCCESS);
}

