/**
  @file btcond-signals.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btcond-hci.h"
#include "state.h"
#include "dbus-helper.h"
#include "bt-dbus.h"
#include "dbus.h"
#include "btcond-signals.h"

/* Send connection_status signal to D-BUS */
void send_dbus_connection_status(con_evt_t *event) {
    DBusMessage *signal;
    const char *status;
    uint8_t reason;

    if (event->type == EVT_TYPE_CC) {
        status = "connected";
        reason = 0x00;
    }
    else {
        status = "disconnected";
        reason = event->dc_reason;
    }

    debug("sending connection_status(\"%s\", \"%s\", 0x%02X) signal",
            event->bda, status, reason);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_CONNECTION_STATUS_SIG,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, event->bda,
                     DBUS_TYPE_STRING, status,
                     DBUS_TYPE_BYTE, reason,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

/* Send connection_status signal to D-BUS */
void send_dbus_rfcomm_status(const char *bda, const char *svc, const char *status) {
    DBusMessage *signal;

    debug("sending rfcomm_status(\"%s\", \"%s\", \"%s\") signal",
            bda, svc, status);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_RFCOMM_STATUS_SIG,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, bda,
                     DBUS_TYPE_STRING, svc,
                     DBUS_TYPE_STRING, status,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

void send_dbus_pin_req(con_evt_t *event) {
    DBusMessage *signal;

    debug("sending pin_req(\"%s\") signal", event->bda);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_PIN_REQ_SIG,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, event->bda,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

void send_dbus_cc_failed(con_evt_t *event) {
    DBusMessage *signal;

    debug("sending connect_failed(\"%s\", 0x%02x) signal",
            event->bda, event->status);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_CONNECT_FAILED_SIG,
                             NULL);

    append_dbus_args(signal,
                     DBUS_TYPE_STRING, event->bda,
                     DBUS_TYPE_BYTE, event->status,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

void send_dbus_auth_failed(con_evt_t *event) {
    DBusMessage *signal;

    debug("sending auth_failed(\"%s\", 0x%02x) signal",
            event->bda, event->status);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_AUTH_FAILED_SIG,
                             NULL);

    append_dbus_args(signal,
                     DBUS_TYPE_STRING, event->bda,
                     DBUS_TYPE_BYTE, event->status,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

void send_dbus_link_key_ok(con_evt_t *event) {
    DBusMessage *signal;

    debug("sending link_key_ok(\"%s\") signal", event->bda);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_LINK_KEY_OK_SIG,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, event->bda,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

void send_dbus_hci_dev_up(const char *dev) {
    DBusMessage *signal;

    debug("sending hci_dev_up(\"%s\") signal", dev);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_HCI_DEV_UP_SIG,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, dev,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}

void send_dbus_hci_dev_down(const char *dev) {
    DBusMessage *signal;

    debug("sending hci_dev_down(\"%s\") signal", dev);

    signal = new_dbus_signal(BTCOND_SIG_PATH,
                             BTCOND_SIG_INTERFACE,
                             BTCOND_HCI_DEV_DOWN_SIG,
                             NULL);
    append_dbus_args(signal,
                     DBUS_TYPE_STRING, dev,
                     DBUS_TYPE_INVALID);

    if (!send_and_unref(get_dbus_connection(), signal))
        error("sending message failed!");
}
