/**
  @file btcond-rfcomm.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include <dbus/dbus.h>

#include <glib.h>

#include <bttools-dbus.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "bt-error.h"
#include "dbus-helper.h"
#include "dbus.h"
#include "btcond-bt.h"
#include "btcond-rfcomm.h"

#define BT_DEV_ID 0

#define PAIR_TIMEOUT 30000

/* Between intermediate waits (milliseconds) */
#define OPEN_WAIT 300
#define OPEN_MAX_TRIES 10

typedef struct {
    GIOChannel         *sock;
    int                 try;
    int                 id;
    char                devname[16];
    int                 dd;
    uint8_t             min_klen;
    uint8_t             role;
    uint16_t            handle;
    gboolean            auth;
    gboolean            encrypt;
    gboolean            canceled;
    struct sockaddr_rc  laddr;
    struct sockaddr_rc  raddr;
    rfcomm_connect_cb   func;
    gpointer            data;
} rfcomm_cb_info;

#ifdef USE_NOKIA_HCI
# define OCF_VENDOR_CMD 0x0000
# define EVT_VENDOR_CMD 0xFF

typedef struct {
    uint8_t  channel;
    uint8_t  _unused;
    uint8_t  eventcode;
    uint16_t handle;
    uint8_t  klen;
} __attribute__ ((packed)) evt_enc_key_len_read;
# define EVT_ENC_KEY_LEN_SIZE 6

typedef struct {
    uint8_t  channel;
    uint8_t  opcode;
    uint16_t handle;
} __attribute__ ((packed)) read_enc_key_len;
# define READ_ENC_KEY_LEN_CP_SIZE 4
#endif /* USE_NOKIA_HCI */

static GSList *pending_connects = NULL;

static int rfcomm_ctl = -1;

static void rfcomm_create_dev(rfcomm_cb_info *info);
static void rfcomm_connect_finalize(int fd, rfcomm_cb_info *info, GError *err);


static rfcomm_cb_info *find_pending_connect(const char *bda, uint8_t ch) {
    bdaddr_t ba;
    GSList *l;

    str2ba(bda, &ba);

    for (l = pending_connects; l != NULL; l = l->next) {
        rfcomm_cb_info *info = l->data;
        if (bacmp(&ba, &info->raddr.rc_bdaddr) == 0 &&
                ch == info->raddr.rc_channel)
            return info;
    }

    return NULL;
}


static gboolean set_socket_params(int sock, uint32_t opts, GError **err) {
    long arg;

    if (opts) {
        uint32_t old, len;

        len = sizeof(old);

        if (getsockopt(sock, SOL_RFCOMM, RFCOMM_LM, &old, &len) < 0) {
            g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "getsockopt: %s", g_strerror(errno));
            return FALSE;
        }

        opts |= old;

        if (setsockopt(sock, SOL_RFCOMM, RFCOMM_LM, &opts, sizeof(opts)) < 0) {
            g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "setsockopt: %s", g_strerror(errno));
            return FALSE;
        }
    }

    arg = fcntl(sock, F_GETFL);
    if (arg < 0) {
        g_set_error(err, BT_COND_ERROR,
                BT_COND_ERROR_CONNECT_FAILED,
                "fcntl(F_GETFL): %s", g_strerror(errno));
        return FALSE;
    }

    /* Return if already nonblocking */
    if (arg & O_NONBLOCK)
        return TRUE;

    arg |= O_NONBLOCK;
    if (fcntl(sock, F_SETFL, arg) < 0) {
        g_set_error(err, BT_COND_ERROR,
                BT_COND_ERROR_CONNECT_FAILED,
                "fcntl(F_SETFL, O_NONBLOCK): %s", g_strerror(errno));
        return FALSE;
    }

    return TRUE;
}

#ifdef USE_NOKIA_HCI
static gboolean get_key_length(int dd, uint16_t handle,
                               uint8_t *klen, GError **err) {
    evt_enc_key_len_read rp;
    read_enc_key_len cp;
    struct hci_request rq;

    memset(&cp, 0, sizeof(cp));
    cp.channel = 0xF0;
    cp.opcode  = 0x07;
    cp.handle  = handle;

    memset(&rq, 0, sizeof(rq));
    rq.ogf    = OGF_VENDOR_CMD;
    rq.ocf    = OCF_VENDOR_CMD;
    rq.event  = EVT_VENDOR_CMD;
    rq.cparam = &cp;
    rq.clen   = READ_ENC_KEY_LEN_CP_SIZE;
    rq.rparam = &rp;
    rq.rlen   = EVT_ENC_KEY_LEN_SIZE;

    if (hci_send_req(dd, &rq, 1000) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "Could not send ENCRYPTION_KEY_LENGTH_READ_CMD: %s",
                    g_strerror(errno));
        return FALSE;
    }

    if (rp.klen == 0x00) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "ENCRYPTION_KEY_LENGTH_READ_CMD: Invalid connection");
        return FALSE;
    }

    *klen = rp.klen;

    return TRUE;
}
#endif

static void btpair_cb(DBusPendingCall *pending, rfcomm_cb_info *info) {
    GError *err = NULL;
    DBusMessage *reply;
    DBusError derr;

    debug("btpair_cb");

    if (info->canceled) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CANCELLED,
                    "Connection was canceled");
        rfcomm_connect_finalize(-1, info, err);
        dbus_pending_call_unref(pending);
        return;
    }

    reply = dbus_pending_call_get_reply(pending);

    dbus_error_init(&derr);

    if (reply == NULL) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "No reply from btpair");
        rfcomm_connect_finalize(-1, info, err);
        dbus_pending_call_unref(pending);
        return;
    }
    else if (dbus_set_error_from_message(&derr, reply)) {
        gint e;

        debug("btpair failed: %s (%s)", derr.message, derr.name);

        if (dbus_error_has_name(&derr, BTPAIR_ERROR_AUTH_FAILED))
            e = BT_COND_ERROR_AUTH_FAILED;
        else
            e = BT_COND_ERROR_CONNECT_FAILED;

        g_set_error(&err, BT_COND_ERROR, e,
                    "%s: %s", derr.name, derr.message);
        dbus_error_free(&derr);
        rfcomm_connect_finalize(-1, info, err);
        dbus_pending_call_unref(pending);
        return;
    }

    if (info->min_klen) {
#ifdef USE_NOKIA_HCI
        uint8_t klen;
        if (!get_key_length(info->dd, htobs(info->handle), &klen, NULL))
            error("Could not get encryption key length");
        else {
            debug("Got encryption key length %d for connection handle %d", klen,
                    htobs(info->handle));
            if (klen < info->min_klen) {
                g_set_error(&err, BT_COND_ERROR,
                            BT_COND_ERROR_SMALL_CRYPT_KEY,
                            "Encryption key is less than required minimum: %d < %d",
                            klen, info->min_klen);
                rfcomm_connect_finalize(-1, info, err);
                dbus_pending_call_unref(pending);
                return;
            }
        }
#else
        (void)(0);
#endif
    }

    rfcomm_create_dev(info);
    dbus_pending_call_unref(pending);
}

static gboolean setup_conn(rfcomm_cb_info *info, GError **err) {
    struct hci_conn_info_req *cr;

    info->dd = hci_open_dev(BT_DEV_ID);
    if (info->dd < 0) {
        error("hci_open_dev: %s", g_strerror(errno));
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "hci_open_dev(%d): %s", BT_DEV_ID, g_strerror(errno));
        return FALSE;
    }

    cr = g_malloc0((sizeof(*cr) + sizeof(struct hci_conn_info)));
    bacpy(&cr->bdaddr, &info->raddr.rc_bdaddr);
    cr->type = ACL_LINK;
    if (ioctl(info->dd, HCIGETCONNINFO, (unsigned long)cr) < 0) {
        error("HCIGETCONNINFO: %s", g_strerror(errno));
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "ioctl(HCIGETCONNINFO): %s", g_strerror(errno));
        g_free(cr);
        return FALSE;
    }

    info->handle = cr->conn_info->handle;
    g_free(cr);

    if (info->role != ROLE_ANY) {
        if (hci_switch_role(info->dd, &info->raddr.rc_bdaddr, info->role, 10000) < 0) {
            error("hci_switch_role(%s): %s",
                    info->role == ROLE_SLAVE ? "slave" : "master",
                    g_strerror(errno));
            /* Ignore role switch error (not fatal) */
        }
    }

    if (info->auth) {
        DBusConnection *conn;
        DBusPendingCall *pending;
        DBusMessage *msg;

        msg = new_dbus_method_call(BTPAIR_SERVICE,
                                   BTPAIR_PATH,
                                   BTPAIR_INTERFACE,
                                   BTPAIR_REQ_PAIR_HANDLE);
        append_dbus_args(msg,
                         DBUS_TYPE_UINT32, (uint32_t)(info->handle),
                         DBUS_TYPE_BOOLEAN, info->encrypt,
                         DBUS_TYPE_INVALID);
        dbus_message_set_auto_activation(msg, TRUE);

        conn = get_dbus_connection();

        if (!dbus_connection_send_with_reply(conn, msg, &pending,
                                             PAIR_TIMEOUT))
            error("Sending method call to btpair failed");

        dbus_connection_flush(conn);
        dbus_message_unref(msg);

        if (!dbus_pending_call_set_notify(pending,
                                     (DBusPendingCallNotifyFunction)btpair_cb,
                                     info, NULL))
            die("Out of memory during dbus_pending_call_set_notify()");

        return TRUE;
    }

    rfcomm_create_dev(info);

    return TRUE;
}

/* Connection setup stuff which doesn't block can go into this function */
static void rfcomm_connect_finalize(int fd, rfcomm_cb_info *info, GError *err) {
    pending_connects = g_slist_remove(pending_connects, info);

    if (fd < 0) {
        if (info->id >= 0) {
            rfcomm_release(info->id, NULL);
            info->id = -1;
        }
        info->func(-1, NULL, err, info->data);
    }
    else {
        GIOChannel *io;

        io = g_io_channel_unix_new(fd);
        g_io_channel_set_close_on_unref(io, TRUE);

        info->func(info->id, io, err, info->data);
    }

    if (info->dd >= 0) {
        close(info->dd);
        info->dd = -1;
    }

    if (info->sock) {
        g_io_channel_unref(info->sock);
        info->sock = NULL;
    }
    
    g_free(info);

    if (err != NULL)
        g_error_free(err);
}

static gboolean rfcomm_open_cb(rfcomm_cb_info *info) {
    GError *err = NULL;
    int fd;

    if (info->canceled) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CANCELLED,
                    "Connection was canceled");
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    fd = open(info->devname, O_RDONLY | O_NOCTTY);
    if (fd < 0) {
        info->try--;
        if (info->try > 0) {
            debug("Waiting %dms for udev to catch up", OPEN_WAIT);
            return TRUE;
        }
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "unable to open %s: %s",
                    info->devname, g_strerror(errno));
    }

    rfcomm_connect_finalize(fd, info, err);

    return FALSE;
}

static void rfcomm_create_dev(rfcomm_cb_info *info) {
    struct rfcomm_dev_req req;
    GError *err = NULL;
    int sk, fd;

    sk = g_io_channel_unix_get_fd(info->sock);

    memset(&req, 0, sizeof(req));

    req.dev_id = -1;
    req.flags = (1 << RFCOMM_REUSE_DLC) | (1 << RFCOMM_RELEASE_ONHUP);

    bacpy(&req.src, &info->laddr.rc_bdaddr);
    bacpy(&req.dst, &info->raddr.rc_bdaddr);
    req.channel = info->raddr.rc_channel;

    info->id = ioctl(sk, RFCOMMCREATEDEV, &req);
    if (info->id < 0) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "ioctl(RFCOMMCREATEDEV): %s", g_strerror(errno));
        rfcomm_connect_finalize(-1, info, err);
        return;
    }

    bt_devname_from_id(info->id, info->devname, sizeof(info->devname));

    fd = open(info->devname, O_RDONLY | O_NOCTTY);
    if (fd < 0 && OPEN_MAX_TRIES > 1) {
        info->try = OPEN_MAX_TRIES - 1;
        g_timeout_add(OPEN_WAIT, (GSourceFunc)rfcomm_open_cb, info);
    }
    else
        rfcomm_connect_finalize(fd, info, NULL);
}

static gboolean sock_io_cb(GIOChannel *chan, GIOCondition cond, rfcomm_cb_info *info) {
    GError *err = NULL;
    int alen, sk, ret;
    socklen_t len;

    if (cond != G_IO_OUT) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "sock_io_cb: cond != G_IO_OUT");
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    if (info->canceled) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CANCELLED,
                    "Connection was canceled");
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    sk = g_io_channel_unix_get_fd(chan);

    len = sizeof(ret);
    if (getsockopt(sk, SOL_SOCKET, SO_ERROR, &ret, &len) < 0) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "getsockopt(..., SO_ERROR, ...): %s", g_strerror(errno));
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    if (ret != 0) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "connect(): %s", g_strerror(ret));
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    debug("sock_io_cb: connected");

    alen = sizeof(info->laddr);
    if (getsockname(sk, (struct sockaddr *)&info->laddr, &alen) < 0) {
        g_set_error(&err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "getsockname: %s", g_strerror(errno));
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    if (!setup_conn(info, &err)) {
        rfcomm_connect_finalize(-1, info, err);
        return FALSE;
    }

    return FALSE;
}

gboolean rfcomm_release(int dev_num, GError **err) {
    struct rfcomm_dev_req req;

    g_assert(rfcomm_ctl >= 0);

    memset(&req, 0, sizeof(req));
    req.dev_id = dev_num;
    req.flags = (1 << RFCOMM_HANGUP_NOW);

    if (ioctl(rfcomm_ctl, RFCOMMRELEASEDEV, &req) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "RFCOMMRELEASEDEV failed: %s (%d)",
                    g_strerror(errno), errno);
        return FALSE;
    }

    return TRUE;
}

gboolean rfcomm_bind(const char *bda, uint8_t ch, int *node, GError **err) {
    struct rfcomm_dev_req req;

    g_assert(rfcomm_ctl >= 0);

    memset(&req, 0, sizeof(req));
    req.dev_id = -1;
    req.flags = 0;
    bacpy(&req.src, BDADDR_ANY);

    str2ba(bda, &req.dst);
    req.channel = ch;

    *node = ioctl(rfcomm_ctl, RFCOMMCREATEDEV, &req);
    if (*node < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "RFCOMMCREATEDEV failed: %s",
                    g_strerror(errno));
        return FALSE;
    }

    return TRUE;
}

gboolean rfcomm_connect(const char *bda, uint8_t ch, rfcomm_params *params,
                        rfcomm_connect_cb cb, gpointer user_data, GError **err) {
    struct sockaddr_rc laddr, raddr;
    rfcomm_cb_info *info;
    int sk;
    uint32_t sk_params;

    if (find_pending_connect(bda, ch)) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "Connection creation to channel %u on %s already in progress",
                    ch, bda);
        return FALSE;
    }

    memset(&laddr, 0, sizeof(laddr));
    memset(&raddr, 0, sizeof(raddr));

    laddr.rc_family = AF_BLUETOOTH;
    bacpy(&laddr.rc_bdaddr, BDADDR_ANY);
    laddr.rc_channel = 0;

    raddr.rc_family = AF_BLUETOOTH;
    str2ba(bda, &raddr.rc_bdaddr);
    raddr.rc_channel = ch;

    sk = socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
    if (sk < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "Can't create RFCOMM socket: %s", g_strerror(errno));
        return FALSE;
    }

    if (bind(sk, (struct sockaddr *)&laddr, sizeof(laddr)) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_CONNECT_FAILED,
                    "Can't bind RFCOMM socket: %s", g_strerror(errno));
        close(sk);
        return FALSE;
    }

    sk_params = 0;

#if 0
    if (params && params->auth) {
        /* Some phones (e.g. ) don't work well with these socket options */
        sk_params |= RFCOMM_LM_AUTH | RFCOMM_LM_ENCRYPT;
    }
#endif

    if (!set_socket_params(sk, sk_params, err)) {
        close(sk);
        return FALSE;
    }

    info = g_new0(rfcomm_cb_info, 1);
    info->id = -1;
    info->dd = -1;
    if (params) {
        info->min_klen = params->min_klen;
        info->role     = params->role;
        info->auth     = params->auth;
        info->encrypt  = params->encrypt;
    }
    else
        info->role = ROLE_ANY;
    info->func = cb;
    info->data = user_data;
    memcpy(&info->laddr, &laddr, sizeof(laddr));
    memcpy(&info->raddr, &raddr, sizeof(raddr));

    info->sock = g_io_channel_unix_new(sk);
    g_io_channel_set_close_on_unref(info->sock, TRUE);

    if (connect(sk, (struct sockaddr *)&raddr, sizeof(raddr)) < 0) {
        /* BlueZ returns EAGAIN eventhough it should return EINPROGRESS */
        if (!(errno == EAGAIN || errno == EINPROGRESS)) {
            debug("connect() failed: %s", g_strerror(errno));
            g_set_error(err, BT_COND_ERROR,
                        BT_COND_ERROR_CONNECT_FAILED,
                        "connect(): %s (%d)", g_strerror(errno), errno);
            g_io_channel_unref(info->sock);
            g_free(info);
            return FALSE;
        }

        debug("Connect in progress");
        g_io_add_watch(info->sock, G_IO_OUT, (GIOFunc)sock_io_cb, info);
        pending_connects = g_slist_append(pending_connects, info);
    }
    else {
        debug("Connect succeeded with first try");
        (void) sock_io_cb(info->sock, G_IO_OUT, info);
    }

    return TRUE;
}

gboolean rfcomm_cancel_connect(const char *bda, uint8_t ch, GError **err) {
    rfcomm_cb_info *info;

    info = find_pending_connect(bda, ch);
    if (info == NULL) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "No connect in progress to channel %u on %s", ch, bda);
        return FALSE;
    }

    info->canceled = TRUE;

    return TRUE;
}

gboolean rfcomm_disconnect(GIOChannel *sock, int dev_num, GError **err) {
    g_io_channel_shutdown(sock, FALSE, NULL);
    g_io_channel_unref(sock);
    return rfcomm_release(dev_num, err);
}

gboolean rfcomm_init(GError **err) {
    if (rfcomm_ctl < 0) {
        rfcomm_ctl = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_RFCOMM);
        if (rfcomm_ctl < 0) {
            g_set_error(err, BT_COND_ERROR,
                        BT_COND_ERROR_FAILED,
                        "Can't open RFCOMM control socket: %s",
                        g_strerror(errno));
            return FALSE;
        }
    }

    return TRUE;
}

