/**
  @file btcond-hci.c

  Functions for monitoring BT HCI events, and sending
  BT_ON/BT_OFF HCI commands.

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/  
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include <glib.h>

#include "log.h"
#include "btcond-bt.h"
#include "state.h"
#include "bt-error.h"
#include "btcond-signals.h"
#include "btcond-hci.h"

/* hci0 */
#define DEV_NUMBER 0

static int hci_sock = -1;

static int event_socket(int *events, GError **err) {
    struct sockaddr_hci addr;
    struct hci_filter flt;
    int sock, i;

    /* Create HCI socket */
    sock = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (sock < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't create HCI socket: %s (%d)",
                    g_strerror(errno), errno);
        return -1;
    }

    /* Setup filter */
    hci_filter_clear(&flt);
    hci_filter_set_ptype(HCI_EVENT_PKT, &flt);
    for (i = 0; events[i] != 0; i++)
        hci_filter_set_event(events[i], &flt);
    if (setsockopt(sock, SOL_HCI, HCI_FILTER, &flt, sizeof(flt)) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't set HCI filter: %s (%d)",
                    g_strerror(errno), errno);
        return -1;
    }

    /* Bind socket to the HCI device */
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = DEV_NUMBER;
    if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't attach to device hci%d: %s (%d)",
                    DEV_NUMBER, g_strerror(errno), errno);
        return -1;
    }

    return sock;
}

static con_evt_t *new_con_evt(uint8_t type) {
    con_evt_t *evt = g_new0(con_evt_t, 1);
    evt->type = type;
    return evt;
}

static void connection_complete(char *data, con_evt_t **event) {
    evt_conn_complete *evt = (evt_conn_complete *)data;
    char addr[18];

    ba2str(&evt->bdaddr, addr);

    *event = new_con_evt(EVT_TYPE_CC);
    (*event)->handle   = evt->handle;
    (*event)->status   = evt->status;
    (*event)->bda      = g_strdup(addr);

    if (evt->status != 0x00) {
        debug("Connection failed. status: 0x%02x", evt->status);
        if (connection_status(addr) != CONN_STATUS_CONNECTED) {
            debug("Marking device capability as old");
            expire_device_capability(addr);
        }
    }
}

static void pin_request(char *data, con_evt_t **event) {
    evt_pin_code_req *evt = (evt_pin_code_req *)data;
    char addr[18];

    ba2str(&evt->bdaddr, addr);

    *event = new_con_evt(EVT_TYPE_PIN_REQ);
    (*event)->bda = g_strdup(addr);
}

static void link_key_notify(char *data, con_evt_t **event) {
    evt_link_key_notify *evt = (evt_link_key_notify *)data;
    char addr[18];

    ba2str(&evt->bdaddr, addr);

    *event = new_con_evt(EVT_TYPE_LINK_KEY);
    (*event)->bda = g_strdup(addr);
}

static void auth_complete(char *data, con_evt_t **event) {
    evt_auth_complete *evt = (evt_auth_complete *)data;

    debug("auth_complete(): handle=%u, status=0x%02X",
            evt->handle, evt->status);

    if (evt->status != 0x00) {
        const gchar *bda;

        bda = get_handle_bda(evt->handle);
        if (bda == NULL) {
            error("auth_complete: Unable to get BDA for handle %u",
                  g_ntohs(evt->handle));
            return;
        }

        *event = new_con_evt(EVT_TYPE_AUTH_FAILED);
        (*event)->status = evt->status;
        (*event)->bda = g_strdup(bda);
    }
}

static void disconnection_complete(char *data, con_evt_t **event) {
    evt_disconn_complete *evt = (evt_disconn_complete *)data;

    if (evt->status != 0x00) {
        debug("disconnection complete error. status: 0x%02x", evt->status);
        return;
    }

    if (!handle_is_connected(evt->handle)) {
        debug("disconnection of unknown connection handle (%u)", evt->handle);
        return;
    }

    *event = new_con_evt(EVT_TYPE_DC);
    (*event)->handle = evt->handle;
    (*event)->bda = g_strdup(get_handle_bda(evt->handle));

    (*event)->dc_reason = evt->reason;

    if (evt->reason == HCI_CONNECTION_TERMINATED)
        debug("BT connection to %s was closed by local event", (*event)->bda);
    else {
        switch (evt->reason) {
            case HCI_OE_USER_ENDED_CONNECTION:
                debug("Connection closed by %s", (*event)->bda);
                break;
            case HCI_CONNECTION_TIMEOUT:
                debug("Supervision timeout to %s. Device out of rage?",
                        (*event)->bda);
                break;
            default:
                debug("Disconnection Complete from %s, reason: 0x%02x",
                        (*event)->bda, evt->reason);
                break;
        }
        debug("Marking device capability as old");
        expire_device_capability((*event)->bda);
    }
}

/* Bluez does not provide this definition */
#define EVT_HW_ERROR 0x10

static con_evt_t *process_data(char *data, int data_len) {
    uint8_t type = data[0];
    con_evt_t *event = NULL;

    if (type == HCI_EVENT_PKT) {
        hci_event_hdr *hdr = (hci_event_hdr *) &data[1];
        char *body = &data[HCI_EVENT_HDR_SIZE + 1];

        switch (hdr->evt) {
            case EVT_CONN_COMPLETE:
                connection_complete(body, &event);
                break;
            case EVT_DISCONN_COMPLETE:
                disconnection_complete(body, &event);
                break;
            case EVT_PIN_CODE_REQ:
                pin_request(body, &event);
                break;
            case EVT_LINK_KEY_NOTIFY:
                link_key_notify(body, &event);
                break;
            case EVT_AUTH_COMPLETE:
                auth_complete(body, &event);
                break;
            case EVT_HW_ERROR:
                bt_hw_error(*body);
                break;
            default:
                error("Invalid event type (0x%02x)", hdr->evt);
                break;
        }
    }
    else
        error("Invalid packet type (0x%02x)", type);

    return event;
}

static void free_con_evt(con_evt_t *evt) {
    if (evt->bda)
        g_free(evt->bda);
    g_free(evt);
}

static gboolean monitor_cb(GIOChannel *chan, GIOCondition cond, gpointer data) {
    con_evt_t *event;
    char buf[HCI_MAX_EVENT_SIZE];
    int buf_len;
    int sock;

    if (cond != G_IO_IN) {
        error("HUP or error on HCI socket. Unable to receive more HCI events.");
        g_io_channel_unref(chan);
        return FALSE;
    }
       
    sock = g_io_channel_unix_get_fd(chan);
    g_assert(sock >= 0);

    buf_len = read(sock, buf, sizeof(buf));
    if (buf_len < 0) {
        error("reading HCI socket failed: %s", g_strerror(errno));
        g_io_channel_unref(chan);
        return FALSE;
    }

    event = process_data(buf, buf_len);
    if (event != NULL) {
        switch (event->type) {
            case EVT_TYPE_CC:
                if (event->status == 0)
                    update_state(event);
                else
                    send_dbus_cc_failed(event);
                break;
            case EVT_TYPE_DC:
                update_state(event);
                break;
            case EVT_TYPE_PIN_REQ:
                send_dbus_pin_req(event);
                break;
            case EVT_TYPE_LINK_KEY:
                send_dbus_link_key_ok(event);
                break;
            case EVT_TYPE_AUTH_FAILED:
                send_dbus_auth_failed(event);
                break;
            default:
                debug("Invalid event type (0x%02x)", event->type);
                break;
        }

        free_con_evt(event);
    }

    return TRUE;
}

gboolean monitor_connections(GError **err) {
    GIOChannel *gio;
    int events[] = { EVT_CONN_COMPLETE,
                     EVT_DISCONN_COMPLETE,
                     EVT_PIN_CODE_REQ,
                     EVT_LINK_KEY_NOTIFY,
                     EVT_AUTH_COMPLETE,
                     EVT_HW_ERROR,
                     0 };

    hci_sock = event_socket(events, err);
    if (hci_sock < 0)
        return FALSE;

    gio = g_io_channel_unix_new(hci_sock);
    g_io_channel_set_close_on_unref(gio, TRUE);
    g_io_add_watch(gio, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                   monitor_cb, NULL);
    return TRUE;
}

gboolean hci_bt_up(void) {
    char dev[6];
    GError *err = NULL;

    if (ioctl(hci_sock, HCIDEVUP, DEV_NUMBER) < 0) {
        if (errno == EALREADY)
            return TRUE;

        error("Can't init device hci%d: %s (%d)", DEV_NUMBER,
              g_strerror(errno), errno);
        return FALSE;
    }

    snprintf(dev, sizeof(dev), "hci%d", DEV_NUMBER);
    send_dbus_hci_dev_up(dev);

    g_spawn_command_line_async("/usr/bin/btname -r", &err);
    if (err != NULL) {
        error("Running btname failed: %s", err->message);
        g_error_free(err);
    }

    debug("Running btname succeeded");

    return TRUE;
}

gboolean hci_bt_down(void) {
    char dev[6];

    if (ioctl(hci_sock, HCIDEVDOWN, DEV_NUMBER) < 0) {
        error("Can't down device hci%d: %s (%d)", DEV_NUMBER,
              g_strerror(errno), errno);
        return FALSE;
    }

    snprintf(dev, sizeof(dev), "hci%d", DEV_NUMBER);
    send_dbus_hci_dev_down(dev);

    return TRUE;
}
