/**
  @file btcond-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "config.h"

#include "log.h"
#include "btcond-bt.h"
#include "btcond-rfcomm.h"
#include "state.h"
#include "dbus.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "bt-utils.h"
#include "bt-error.h"
#include "btcond-dbus.h"

#define UDEV_WAIT       300
#define UDEV_MAX_TRIES  10

typedef struct {
    int          try;
    char        *dev;
    DBusMessage *message;
    DBusMessage *reply;
} UDevData;

struct service_filter_data {
    char *service;
    char *filter_string;
    gboolean (*func)(void);
};

/* For checking correct message parameters */
static const char rfcomm_bind_old_sgn[]       = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };
                                              
static const char rfcomm_bind_sgn[]           = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };
                                              
static const char rfcomm_connect_old_sgn[]    = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_BOOLEAN,
                                                  DBUS_TYPE_INVALID };
                                              
static const char rfcomm_connect_sgn[]        = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_BOOLEAN,
                                                  DBUS_TYPE_INVALID };

static const char rfcomm_connect_ext_sgn[]    = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_BOOLEAN,
                                                  DBUS_TYPE_BOOLEAN,
                                                  DBUS_TYPE_BOOLEAN,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char rfcomm_cancel_connect_sgn[] = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char rfcomm_disconnect_old_sgn[] = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char rfcomm_disconnect_sgn[]     = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char disconnect_sgn[]            = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char rfcomm_status_sgn[]         = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char rfcomm_status_old_sgn[]     = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char update_dev_info_sgn[]       = { DBUS_TYPE_STRING,
                                                  DBUS_TYPE_INVALID };

static const char update_dev_info_old_sgn[]   = { DBUS_TYPE_INVALID };

#ifdef USE_MCE
# define COVER_SWITCH_PATH "/org/kernel/devices/platform/gpio-switch/prot_shell/cover_switch"
# define KEVENT_DBUS_IF    "org.kernel.kevent"

static DBusHandlerResult cover_filter(DBusConnection *connection,
                                      DBusMessage    *message,
                                      void          (*cover_cb)(void)) {

    if (dbus_message_is_signal(message, KEVENT_DBUS_IF, "change"))
        cover_cb();
    else
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    return DBUS_HANDLER_RESULT_HANDLED;
}
#endif



static void string_list_to_array(GSList *l, gchar ***array, gint *length) {
    gint i;

    *length = g_slist_length(l);
    if (*length == 0) {
        *array = NULL;
        return;
    }

    *array  = g_new(gchar *, *length);

    for (i = 0; i < (*length); i++)
        (*array)[i] = g_slist_nth_data(l, i);
}

static DBusHandlerResult service_exit_filter(DBusConnection *connection,
                                             DBusMessage *message,
                                             void *user_data) {
    char *service;

    if (dbus_message_is_signal(message,
                                DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                                "ServiceDeleted")) {
        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceDeleted signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
    }
    else if (dbus_message_is_signal(message,
                                    DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                                    "ServiceOwnerChanged")) {
        char *old, *new;

        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_STRING, &old,
                    DBUS_TYPE_STRING, &new,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceOwnerChanged signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
        dbus_free(old);
        /* We are not interested of service creations */
        if (*new != '\0') {
            dbus_free(service);
            dbus_free(new);
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
        dbus_free(new);
    }
    else
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    bt_check_owner_disconnect(service);

    dbus_free(service);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusMessage *rfcomm_bind_reply(DBusMessage *msg) {
    DBusMessage *reply;
    GError *err = NULL;
    gboolean t;
    gchar *bda, *svc_name;
    int dev_id;

    debug("rfcomm_bind_reply()");

    if (dbus_message_has_signature(msg, rfcomm_bind_old_sgn)) {
        t = get_dbus_args(msg,
                DBUS_TYPE_STRING, &svc_name,
                DBUS_TYPE_INVALID);
        g_assert(t == TRUE);
        bda = NULL;
    }
    else {
        t = get_dbus_args(msg,
                DBUS_TYPE_STRING, &bda,
                DBUS_TYPE_STRING, &svc_name,
                DBUS_TYPE_INVALID);
        g_assert(t == TRUE);
    }

    dev_id = bind_dev(bda, svc_name, &err);

    dbus_free(svc_name);
    if (bda)
        dbus_free(bda);

    if (err != NULL) {
        reply = new_dbus_error_gerr(msg, err);
        g_clear_error(&err);
    }
    else {
        char dev[32];
        bt_devname_from_id(dev_id, dev, sizeof(dev));
        debug("rfcomm_bind_reply: replying with \"%s\"", dev);
        reply = new_dbus_method_return(msg);
        append_dbus_args(reply,
                DBUS_TYPE_STRING, dev,
                DBUS_TYPE_INVALID);
    }

    return reply;
}

static DBusMessage *dev_cap_reply(DBusMessage *msg) {
    gchar *bda;
    DBusMessageIter iter;
    DBusMessage *reply;
    gchar **array;
    gint length;
    GError *err = NULL;
    GSList *list;

    debug("dev_cap_reply()");

    dbus_message_iter_init(msg, &iter);
    if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
        bda = dbus_message_iter_get_string(&iter);
        if (bda == NULL) {
            die("OOM during dbus_message_iter_get_string");
        }
    }
    else
        bda = NULL;

    list = get_service_name_list(bda, &err);
    if (err != NULL) {
        reply = new_dbus_error_gerr(msg, err);
        g_error_free(err);
        return reply;
    }

    string_list_to_array(list, &array, &length);

    reply = new_dbus_method_return(msg);

    append_dbus_args(reply,
            DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, array, length,
            DBUS_TYPE_INVALID);

    g_free(array);

    if (bda)
        dbus_free(bda);

    return reply;
}

static void rfcomm_connect_complete(int dev_id, GError *err, DBusMessage *msg) {
    DBusMessage *reply;

    if (dev_id >= 0) {
        char devname[32];

        bt_devname_from_id(dev_id, devname, sizeof(devname));
        reply = new_dbus_method_return(msg);
        append_dbus_args(reply,
                DBUS_TYPE_STRING, devname,
                DBUS_TYPE_INVALID);
    }
    else
        reply = new_dbus_error_gerr(msg, err);

    if (!send_and_unref(get_dbus_connection(), reply))
        error("sending D-BUS reply failed");
}

static DBusMessage *rfcomm_connect_reply(DBusMessage *msg) {
    DBusMessage *reply = NULL;
    GError *err = NULL;
    dbus_bool_t auto_dc, auth = TRUE, encr = FALSE;
    char *bda, *svc, *role = NULL;
    const char *owner;

    if (dbus_message_has_signature(msg, rfcomm_connect_old_sgn)) {
        bda = NULL;
        (void) get_dbus_args(msg,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_BOOLEAN, &auto_dc,
                             DBUS_TYPE_INVALID);
    }
    else if (dbus_message_has_signature(msg, rfcomm_connect_sgn)) {
        (void) get_dbus_args(msg,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_BOOLEAN, &auto_dc,
                             DBUS_TYPE_INVALID);
    }
    else if (dbus_message_has_signature(msg, rfcomm_connect_ext_sgn)) {
        (void) get_dbus_args(msg,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_BOOLEAN, &auto_dc,
                             DBUS_TYPE_BOOLEAN, &auth,
                             DBUS_TYPE_BOOLEAN, &encr,
                             DBUS_TYPE_STRING, &role,
                             DBUS_TYPE_INVALID);
    }
    else
        g_assert_not_reached();

    if (auto_dc)
        owner = dbus_message_get_sender(msg);
    else
        owner = NULL;

    dbus_message_ref(msg);
    if (!connect_service(bda, svc, owner, auth, encr, role,
                (btcon_cb_t)rfcomm_connect_complete,
                msg, (GFreeFunc)dbus_message_unref, &err)) {
        dbus_message_unref(msg);
        error("connect(%s, %s) failed: %s", bda ? bda : "\"\"", svc, err->message);
        reply = new_dbus_error_gerr(msg, err);
        g_clear_error(&err);
    }

    dbus_free(svc);
    if (bda != NULL)
        dbus_free(bda);
    if (role != NULL)
        dbus_free(role);

    return reply;
}

static void rfcomm_connect_sdp_cb(GError *err, DBusMessage *msg) {
    DBusMessage *reply;

    if (err)
        reply = new_dbus_error_gerr(msg, err);
    else
        reply = rfcomm_connect_reply(msg);

    if (reply != NULL) {
        if (!send_and_unref(get_dbus_connection(), reply))
            error("sending D-BUS message failed!");
    }
}

static void dev_cap_reply_cb(GError *err, DBusMessage *msg) {
    DBusMessage *reply;

    if (err)
        reply = new_dbus_error_gerr(msg, err);
    else
        reply = dev_cap_reply(msg);

    if (!send_and_unref(get_dbus_connection(), reply))
        error("sending D-BUS message failed!");
}

static gboolean udev_wait(UDevData *data) {
    struct stat finfo;
    
    if (stat(data->dev, &finfo) < 0) {
        gchar *msg;

        data->try--;
        if (data->try > 0) {
            debug("Waiting %dms for udev to catch up", UDEV_WAIT);
            return TRUE;
        }

        msg = g_strdup_printf("Giving up trying to stat %s: %s",
                data->dev, g_strerror(errno));
        error("%s", msg);

        dbus_message_unref(data->reply);
        data->reply = new_dbus_error(data->message, BTCOND_ERROR_INTERNAL, msg);

        g_free(msg);
    }


    if (!send_and_unref(get_dbus_connection(), data->reply))
        error("sending D-BUS message failed!");

    dbus_message_unref(data->message);
    g_free(data->dev);
    g_free(data);

    return FALSE;
}

static gboolean should_wait_for_udev(DBusMessage *reply, DBusMessage *message) {
    /* reply could be a D-BUS error, so we need to check this */
    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_METHOD_RETURN) {
        char *dev;
        struct stat finfo;

        get_dbus_args(reply,
                DBUS_TYPE_STRING, &dev,
                DBUS_TYPE_INVALID);

        if (stat(dev, &finfo) < 0) {
            UDevData *data;

            debug("Waiting for udev to catch up");

            data = g_new0(UDevData, 1);
            data->dev     = g_strdup(dev);
            data->try     = UDEV_MAX_TRIES;
            data->reply   = reply;
            data->message = message;

            dbus_message_ref(message);

            g_timeout_add(UDEV_WAIT, (GSourceFunc)udev_wait, data);

            dbus_free(dev);
            return TRUE;
        }

        dbus_free(dev);
    }

    return FALSE;
}

static void update_dev_info_cb(GError *err, DBusMessage *msg) {
    DBusMessage *reply;

    if (err)
        reply = new_dbus_error_gerr(msg, err);
    else
        reply = new_dbus_method_return(msg);

    if (!send_and_unref(get_dbus_connection(), reply))
        error("sending D-BUS message failed!");
}

static void rfcomm_bind_reply_cb(GError *err, DBusMessage *msg) {
    DBusMessage *reply;

    if (err)
        reply = new_dbus_error_gerr(msg, err);
    else
        reply = rfcomm_bind_reply(msg);

    if (should_wait_for_udev(reply, msg))
        return;

    if (!send_and_unref(get_dbus_connection(), reply))
        error("sending D-BUS message failed!");
}

static DBusHandlerResult rfcomm_status_request(DBusMessage    *message,
                                               DBusConnection *connection) {
    DBusMessage *reply;
    GError *err = NULL;
    char *svc, *bda;
    const char *status;

    if (dbus_message_has_signature(message, rfcomm_status_old_sgn)) {
        bda = NULL;
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else if (dbus_message_has_signature(message, rfcomm_status_sgn)) {
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed");
        return DBUS_HANDLER_RESULT_HANDLED;
    }


    status = bt_service_status(bda, svc, &err);
    if (status == NULL) {
        reply = new_dbus_error_gerr(message, err);
        g_clear_error(&err);
    }
    else {
        reply = new_dbus_method_return(message);
        append_dbus_args(reply,
                DBUS_TYPE_STRING, status,
                DBUS_TYPE_INVALID);
    }

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    if (bda != NULL)
        dbus_free(bda);
    dbus_free(svc);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult connection_status_request(DBusMessage    *message,
                                                   DBusConnection *connection) {
    DBusMessage *reply;
    DBusMessageIter iter;
    gchar *bda;
    ConnStatus status;


    dbus_message_iter_init(message, &iter);
    if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
        bda = dbus_message_iter_get_string(&iter);
        if (bda == NULL)
            die("OOM during dbus_message_iter_string");
        if (!bda_ok(bda)) {
            reply = new_dbus_error(message, BTCOND_ERROR_INVALID_DEV, NULL);
            if (!send_and_unref(connection, reply))
                error("sending D-BUS message failed!");
            dbus_free(bda);
            return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
    else
        bda = NULL;
   
    status = connection_status(bda);
    if (bda != NULL)
        dbus_free(bda);

    reply = new_dbus_method_return(message);

    switch (status) {
        case  CONN_STATUS_CONNECTED:
            append_dbus_args(reply,
                    DBUS_TYPE_STRING, "connected",
                    DBUS_TYPE_INVALID);
            break;
            
        case  CONN_STATUS_DISCONNECTED:
        case  CONN_STATUS_DISABLED:
            append_dbus_args(reply,
                    DBUS_TYPE_STRING, "disconnected",
                    DBUS_TYPE_INVALID);
            break;
        default:
            /* Never reached */
            g_assert_not_reached();
    }

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult disconnect_request(DBusMessage    *message,
                                            DBusConnection *connection) {
    DBusMessage *reply;
    gchar *bda;

    if (dbus_message_has_signature(message, disconnect_sgn))
        get_dbus_args(message,
                      DBUS_TYPE_STRING, &bda,
                      DBUS_TYPE_INVALID);
    else
        bda = NULL;
   
    if (connection_status(bda) == CONN_STATUS_CONNECTED) {
        bt_disconnect(bda, FALSE);
        reply = new_dbus_method_return(message);
    }
    else
        reply = new_dbus_error(message, BTCOND_ERROR_NOT_CONNECTED, NULL);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    if (bda)
        dbus_free(bda);

    return DBUS_HANDLER_RESULT_HANDLED;
}

/* D-Bus: No parameters, returs list of capabilities */
static DBusHandlerResult dev_capability_request(DBusMessage    *message,
                                                DBusConnection *connection) {
    gchar *bda;
    GError *err = NULL;
    DBusMessageIter iter;
    DBusMessage *reply;

    dbus_message_iter_init(message, &iter);
    if (dbus_message_iter_get_arg_type(&iter) == DBUS_TYPE_STRING) {
        bda = dbus_message_iter_get_string(&iter);
        if (bda == NULL)
            die("OOM during dbus_message_iter_string");
    }
    else
        bda = NULL;

    dbus_message_ref(message);
    if (!schedule_capability_update(bda, (btsdp_cb_t)dev_cap_reply_cb,
                message, (GFreeFunc)dbus_message_unref, &err)) {
        dbus_message_unref(message);
        reply = new_dbus_error_gerr(message, err);
        g_clear_error(&err);
    }
    else {
        if (bda != NULL)
            dbus_free(bda);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (bda != NULL)
        dbus_free(bda);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult rfcomm_connect_request(DBusMessage    *message,
                                                DBusConnection *connection) {
    DBusMessage *reply;
    GError *err = NULL;
    char *bda, *svc;

    reply = NULL;

    if (dbus_message_has_signature(message, rfcomm_connect_old_sgn)) {
        bda = NULL;
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else if (dbus_message_has_signature(message, rfcomm_connect_sgn) ||
             dbus_message_has_signature(message, rfcomm_connect_ext_sgn)) {
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (need_capability_update(bda, svc)) {
        dbus_message_ref(message);
        if (!schedule_capability_update(bda, (btsdp_cb_t)rfcomm_connect_sdp_cb,
                message, (GFreeFunc)dbus_message_unref, &err)) {
            dbus_message_unref(message);
            reply = new_dbus_error_gerr(message, err);
            g_clear_error(&err);
        }
    }
    else
        reply = rfcomm_connect_reply(message);

    if (reply != NULL) {
        if (!send_and_unref(connection, reply))
            error("sending D-BUS message failed!");
    }

    dbus_free(svc);
    if (bda != NULL)
        dbus_free(bda);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult update_dev_info_request(DBusMessage    *message,
                                                 DBusConnection *connection) {
    DBusMessage *reply;
    GError *err = NULL;
    char *bda;

    reply = NULL;

    if (dbus_message_has_signature(message, update_dev_info_sgn)) {
        get_dbus_args(message,
                      DBUS_TYPE_STRING, &bda,
                      DBUS_TYPE_INVALID);
    }
    else if (dbus_message_has_signature(message, update_dev_info_old_sgn))
        bda = NULL;
    else {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed!");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    dbus_message_ref(message);
    if (!schedule_capability_update(bda, (btsdp_cb_t)update_dev_info_cb, message,
                                    (GFreeFunc)dbus_message_unref, &err)) {
        dbus_message_unref(message);
        reply = new_dbus_error_gerr(message, err);
        g_clear_error(&err);
    }

    if (reply) {
        if (!send_and_unref(connection, reply))
            error("sending D-BUS message failed!");
    }

    if (bda != NULL)
        dbus_free(bda);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult rfcomm_cancel_connect_request(DBusMessage    *message,
                                                       DBusConnection *connection) {
    GError *err = NULL;
    DBusMessage *reply;
    BtService *sv;
    char *bda, *svc;

    if (!dbus_message_has_signature(message, rfcomm_cancel_connect_sgn)) {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    get_dbus_args(message,
                  DBUS_TYPE_STRING, &bda,
                  DBUS_TYPE_STRING, &svc,
                  DBUS_TYPE_INVALID);

    sv = get_service(bda, svc, &err);

    if (sv == NULL || !rfcomm_cancel_connect(bda, sv->channel, &err)) {
        reply = new_dbus_error_gerr(message, err);
        g_clear_error(&err);
    }
    else
        reply = new_dbus_method_return(message);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    dbus_free(bda);
    dbus_free(svc);

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult rfcomm_release_request(DBusMessage    *message,
                                                DBusConnection *connection) {
    DBusMessage *reply;
    GError *err = NULL;
    char *bda, *svc;

    if (dbus_message_has_signature(message, rfcomm_bind_old_sgn)) {
        bda = NULL;
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
        if (g_str_has_prefix(svc, "/dev/rfcomm")) {
            int dev_id;
            BtService *sv;

            sv = NULL;

            if (sscanf(svc, "/dev/rfcomm%d", &dev_id) == 1)
                sv = get_bound_service((uint8_t)dev_id);

            if (sv != NULL) {
                dbus_free(svc);
                svc = dbus_strdup(sv->name);
                bda = dbus_strdup(sv->bda);
            }
            else {
                gchar *msg = g_strdup_printf("Device %s is not bound", svc);
                reply = new_dbus_error(message, BTCOND_ERROR_NOT_BOUND, msg);
                if (!send_and_unref(connection, reply))
                    error("sending D-BUS message failed!");
                g_free(msg);
                dbus_free(svc);
                return DBUS_HANDLER_RESULT_HANDLED;
            }
        }
    }
    else if (dbus_message_has_signature(message, rfcomm_bind_sgn)) {
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (!release_dev(bda, svc, &err)) {
        reply = new_dbus_error_gerr(message, err);
        g_clear_error(&err);
    }
    else
        reply = new_dbus_method_return(message);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    dbus_free(svc);
    if (bda != NULL)
        dbus_free(bda);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


static DBusHandlerResult rfcomm_disconnect_request(DBusMessage    *message,
                                                   DBusConnection *connection) {
    DBusMessage *reply;
    GError *err = NULL;
    char *bda, *svc;

    if (dbus_message_has_signature(message, rfcomm_disconnect_old_sgn)) {
        bda = NULL;
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
        if (g_str_has_prefix(svc, "/dev/rfcomm")) {
            int dev_id;
            BtService *sv;

            sv = NULL;

            if (sscanf(svc, "/dev/rfcomm%d", &dev_id) == 1)
                sv = get_connected_service((uint8_t)dev_id);

            dbus_free(svc);

            if (sv != NULL) {
                svc = dbus_strdup(sv->name);
                bda = dbus_strdup(sv->bda);
            }
            else {
                reply = new_dbus_error(message, BTCOND_ERROR_NOT_CONNECTED, NULL);
                if (!send_and_unref(connection, reply))
                    error("sending D-BUS message failed!");
                return DBUS_HANDLER_RESULT_HANDLED;
            }
        }
    }
    else if (dbus_message_has_signature(message, rfcomm_disconnect_sgn)) {
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (!disconnect_service(bda, svc, &err)) {
        reply = new_dbus_error_gerr(message, err);
        g_clear_error(&err);
    }
    else
        reply = new_dbus_method_return(message);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    dbus_free(svc);
    if (bda != NULL)
        dbus_free(bda);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusHandlerResult rfcomm_bind_request(DBusMessage    *message,
                                             DBusConnection *connection) {
    gchar *bda, *svc;
    GError *err = NULL;
    DBusMessage *reply;

    if (dbus_message_has_signature(message, rfcomm_bind_old_sgn)) {
        bda = NULL;
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else if (dbus_message_has_signature(message, rfcomm_bind_sgn)) {
        (void) get_dbus_args(message,
                             DBUS_TYPE_STRING, &bda,
                             DBUS_TYPE_STRING, &svc,
                             DBUS_TYPE_INVALID);
    }
    else {
        if (!send_invalid_args(connection, message))
            error("sending D-BUS message failed");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    if (need_capability_update(bda, svc)) {
        dbus_message_ref(message);
        schedule_capability_update(bda, (btsdp_cb_t)rfcomm_bind_reply_cb,
                                   message, (GFreeFunc)dbus_message_unref,
                                   &err);
        if (err != NULL) {
            dbus_message_unref(message);
            reply = new_dbus_error_gerr(message, err);
            g_clear_error(&err);
        }
        else {
            if (bda != NULL)
                dbus_free(bda);
            dbus_free(svc);
            return DBUS_HANDLER_RESULT_HANDLED;
        }
    }
    else {
        reply = rfcomm_bind_reply(message);
        if (should_wait_for_udev(reply, message)) {
            if (bda != NULL)
                dbus_free(bda);
            dbus_free(svc);
            return DBUS_HANDLER_RESULT_HANDLED;
        }
    }

    if (bda != NULL)
        dbus_free(bda);
    dbus_free(svc);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult get_bda_list_request(DBusMessage    *message,
                                              DBusConnection *connection) {
    DBusMessage *reply;
    GSList *list;
    gchar **array;
    gint length;

    reply = new_dbus_method_return(message);

    list = get_bda_list();
    string_list_to_array(list, &array, &length);

    append_dbus_args(reply,
            DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, array, length,
            DBUS_TYPE_INVALID);

    g_free(array);
    g_slist_free(list);

    if (!send_and_unref(connection, reply))
        error("sending D-BUS message failed!");

    return DBUS_HANDLER_RESULT_HANDLED;
}


typedef DBusHandlerResult (*handler_func)(DBusMessage *message,
                                          DBusConnection *connection);

typedef struct {
    const char *interface;
    const char *name;
    handler_func func;
} method_handler_t;

static method_handler_t handlers[] = {
    { BTCOND_REQ_INTERFACE, BTCOND_CONNECTION_STATUS_REQ,     connection_status_request     },
    { BTCOND_REQ_INTERFACE, BTCOND_RFCOMM_STATUS_REQ,         rfcomm_status_request         },
    { BTCOND_REQ_INTERFACE, BTCOND_DISCONNECT_REQ,            disconnect_request            },
    { BTCOND_REQ_INTERFACE, BTCOND_DEV_CAPABILITY_REQ,        dev_capability_request        },
    { BTCOND_REQ_INTERFACE, BTCOND_RFCOMM_CONNECT_REQ,        rfcomm_connect_request        },
    { BTCOND_REQ_INTERFACE, BTCOND_RFCOMM_DISCONNECT_REQ,     rfcomm_disconnect_request     },
    { BTCOND_REQ_INTERFACE, BTCOND_RFCOMM_BIND_REQ,           rfcomm_bind_request           },
    { BTCOND_REQ_INTERFACE, BTCOND_RFCOMM_RELEASE_REQ,        rfcomm_release_request        },
    { BTCOND_REQ_INTERFACE, BTCOND_RFCOMM_CANCEL_CONNECT_REQ, rfcomm_cancel_connect_request },
    { BTCOND_REQ_INTERFACE, BTCOND_GET_BDA_LIST_REQ,          get_bda_list_request          },
    { BTCOND_REQ_INTERFACE, BTCOND_UPDATE_DEV_INFO_REQ,       update_dev_info_request       },
    { NULL }
};

/* Handler for D-Bus requests */
static DBusHandlerResult btcond_req_handler(DBusConnection     *connection,
                                            DBusMessage        *message,
                                            void               *user_data) {
    method_handler_t *handler;
    const char *dest;

    if (dbus_message_get_type(message) != DBUS_MESSAGE_TYPE_METHOD_CALL)
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    dest = dbus_message_get_destination(message);
    if (!g_str_equal(dest, BTCOND_SERVICE)) {
        debug("Received D-Bus message not addressed to me.");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    for (handler = handlers; handler->interface != NULL; handler++) {
        if (dbus_message_is_method_call(message,
                                        handler->interface,
                                        handler->name)) {
            debug("Received %s", handler->name);
            return handler->func(message, connection);
        }
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

#ifdef USE_PINAGENT
static DBusHandlerResult service_wait_filter(DBusConnection *connection,
                                             DBusMessage    *message,
                                             struct service_filter_data *data) {
    char *service;

    if (dbus_message_is_signal(message,
                               DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                               "ServiceCreated")) {
        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceAquired signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
    }
    else if (dbus_message_is_signal(message,
                                    DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS,
                                    "ServiceOwnerChanged")) {
        char *old, *new;
        gboolean deleted;

        if (!get_dbus_args(message,
                    DBUS_TYPE_STRING, &service,
                    DBUS_TYPE_STRING, &old,
                    DBUS_TYPE_STRING, &new,
                    DBUS_TYPE_INVALID)) {
            error("Invalid arguments for ServiceOwnerChanged signal");
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }

        if (new[0] == '\0')
            deleted = TRUE;
        else
            deleted = FALSE;

        dbus_free(new);
        dbus_free(old);

        /* We are not interested in service deletions */
        if (deleted) {
            dbus_free(service);
            return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        }
    }
    else
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    if (!g_str_equal(data->service, service)) {
        dbus_free(service);
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    dbus_free(service);

    debug("Target service \"%s\" was aquired.", data->service);

    if (!data->func())
        die("Calling service aquired callback function failed!");

    dbus_bus_remove_match(connection, data->filter_string, NULL);
    dbus_connection_remove_filter(connection,
                                  (DBusHandleMessageFunction)service_wait_filter,
                                  data);

    g_free(data->filter_string);
    g_free(data->service);
    g_free(data);

    return DBUS_HANDLER_RESULT_HANDLED;
}
#endif

static DBusObjectPathVTable btcond_req_vtable = {
    .message_function    = btcond_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection) {
    dbus_bool_t ret;
    ret = dbus_connection_register_object_path(connection,
                                               BTCOND_REQ_PATH,
                                               &btcond_req_vtable,
                                               NULL);
    if (ret == FALSE)
        error("dbus_connection_register_object_path failed");

#ifdef USE_MCE
    if (!add_mode_listener(connection, mode_change))
        error("Adding mode listener failed");
    dbus_bus_add_match(connection,
                       "interface=" KEVENT_DBUS_IF
                       ",member=change"
                       ",path=" COVER_SWITCH_PATH, NULL);
    if (!dbus_connection_add_filter(connection,
                                    (DBusHandleMessageFunction)cover_filter,
                                    update_cover_state, NULL)) {
        error("Adding cover listener failed");
    }
#endif

    if (!dbus_connection_add_filter(connection,
                                    service_exit_filter, NULL, NULL))
        error("dbus_connection_add_filter() failed");

    dbus_bus_add_match(connection,
            "interface=" DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS
            ",member=ServiceDeleted", NULL);
    dbus_bus_add_match(connection,
            "interface=" DBUS_INTERFACE_ORG_FREEDESKTOP_DBUS
            ",member=ServiceOwnerChanged", NULL);

}

void destroy_dbus_handlers(DBusConnection *connection) {
    dbus_connection_unregister_object_path(connection, BTCOND_REQ_PATH);
}

